import traceback
import numpy as np
from scipy.optimize import linear_sum_assignment
import json
import click
from datasets import load_metric
import Levenshtein
from sklearn.metrics import precision_recall_fscore_support as prfs
import warnings
import logging
import os
import itertools

warnings.filterwarnings("ignore")

logger = logging.getLogger(__name__)
logger.setLevel(logging.INFO)
logger.handlers = []

def sim(name_1, name_2):
    if name_1 == "<PAD>" or name_2 == "<PAD>":
        return 0
    return 1 if name_1 == name_2 else 0

def create_cost_matrix(graph_1, graph_2):
    nodes_1 = graph_1["nodes"]
    nodes_2 = graph_2["nodes"]

    num_nodes_1 = len(nodes_1)
    num_nodes_2 = len(nodes_2)

    nodes_similarity_matrix = np.zeros((num_nodes_1, num_nodes_2))
    
    for i, node_1 in enumerate(graph_1["nodes"]):
        for j, node_2 in enumerate(graph_2["nodes"]):
            nodes_similarity_matrix[i, j] = sim(node_1, node_2)  

    links_similarity_matrix = np.zeros((num_nodes_1, num_nodes_2))
    for link_1 in graph_1["links"]:
        for link_2 in graph_2["links"]:
            if link_1["source"] == link_2["source"] and link_1["target"] == link_2["target"]:
                try:
                    i_index_1 = nodes_1.index(link_1["source"])
                    i_index_2 = nodes_2.index(link_2["source"])
                    j_index_1 = nodes_1.index(link_1["target"])
                    j_index_2 = nodes_2.index(link_2["target"])
                except ValueError:
                    continue
                links_similarity_matrix[i_index_1, i_index_2] += 1
                links_similarity_matrix[j_index_1, j_index_2] += 1
    
    cost_matrix = 2 - nodes_similarity_matrix - 0.5 * links_similarity_matrix
    return cost_matrix

def compute_assignment_matrix(graph_1, graph_2):
    cost_matrix = create_cost_matrix(graph_1, graph_2)
    row_ind, col_ind = linear_sum_assignment(cost_matrix) 
    return row_ind, col_ind, cost_matrix[row_ind, col_ind].sum()

def matching(graph_1, graph_2):
    indices_1, indices_2, total_cost = compute_assignment_matrix(graph_1, graph_2)
    return indices_1, indices_2

def ratio_levenshtein(x, y):
    assert len(x) == len(y)
    n = len(x)
    total = 0
    for i in range(n):
        total += Levenshtein.ratio(x[i], y[i])
    return total / n


def flatten(gt, pred, types = None):
    assert len(gt) == len(pred)

    gt_flat = []
    pred_flat = []

    for (sample_gt, sample_pred) in zip(gt, pred):
        union = set()

        union.update(sample_gt)
        union.update(sample_pred)

        for s in union:
            if types: 
                if s in types:
                    if s in sample_gt:
                        gt_flat.append(types.index(s)+1)
                    else:
                        gt_flat.append(0)

                    if s in sample_pred:
                        pred_flat.append(types.index(s)+1)
                    else:
                        pred_flat.append(0)
                else:
                    gt_flat.append(0)
                    pred_flat.append(0)
            else:
                if s in sample_gt:
                    gt_flat.append(1)
                else:
                    gt_flat.append(0)

                if s in sample_pred:
                    pred_flat.append(1)
                else:
                    pred_flat.append(0)
    return gt_flat, pred_flat

def print_results(per_type, micro, macro, types, result_dict = None):
    columns = ('type', 'precision', 'recall', 'f1-score', 'support')

    row_fmt = "%30s" + (" %12s" * (len(columns) - 1))
    logger.info(row_fmt % columns)

    metrics_per_type = []
    for i, t in enumerate(types):
        metrics = []
        for j in range(len(per_type)):
            metrics.append(per_type[j][i])
        metrics_per_type.append(metrics)

    for m, t in zip(metrics_per_type, types):
        logger.info(row_fmt % get_row(m, t))
        if result_dict is not None:
            result_dict[t] = {}
            result_dict[t]["precision"] = m[0]
            result_dict[t]["recall"] = m[1]
            result_dict[t]["f1-score"] = m[2]
            result_dict[t]["support"] = int(m[3])

    logger.info('')

    # micro
    logger.info(row_fmt % get_row(micro, 'micro'))

    # macro
    logger.info(row_fmt % get_row(macro, 'macro'))

def get_row(data, label):
    row = [label]
    for i in range(len(data) - 1):
        row.append("%.2f" % (data[i] * 100))
    row.append(data[3])
    return tuple(row)

def get_content_type(content):
    content = content.strip('\'')
    assert isinstance(content, str), content
    for ext in ["jpg", "png", "jpeg", "gif", "bmp", "tiff", "svg", "ico"]:
        if "."+ext in content:
            return "image"
    for ext in ["mp3", "wav", "wma", "ogg", "aac", "flac", "aiff", "au"]:
        if "."+ext in content:
            return "audio"
    for ext in ["mp4", "avi", "mov", "flv", "wmv", "mkv", "webm", "m4v", "mpg", "mpeg"]:
        if "."+ext in content:
            return "video"
    return "text"

@click.command()
@click.option("--data_dir", default="data_huggingface", help="The directory of the data.")
@click.option("--prediction_dir", default="predictions", help="The directory of the data.")
@click.option("--save_dir", default=None, help="The directory to save the evaluation results")
@click.option("--alignment", default=None)
@click.option("--splits", "-s", multiple=True, default=["overall"])
@click.option("--n_tools", "-n", multiple=True, default=["overall"])
@click.option("--mode", default="add")
@click.option("--metric", "-m", multiple=True, default=["all"])
@click.option("--llm", default="gpt-3.5-turbo")
@click.option("--ignore_tool_type", default=False)
def main(data_dir, prediction_dir, save_dir, splits, n_tools, mode, metric, llm, ignore_tool_type, alignment):
    args = locals()
    
    if save_dir is None:
        save_dir = prediction_dir.replace("predictions", "metrics") 
        save_dir = save_dir + f"_alignment_{alignment}" if alignment is not None else save_dir

    formatter = logging.Formatter(f'%(asctime)s - [ {llm} ] - %(levelname)s - %(message)s')
    if not os.path.exists(f'{data_dir}/{save_dir}'):
        os.makedirs(f'{data_dir}/{save_dir}')
    
    metric_file = f'{data_dir}/{save_dir}/{llm}.json'
    if os.path.exists(metric_file):
        all_metric_dict = json.load(open(metric_file, "r"))
    else:
        all_metric_dict = {}
    
    file_handler = logging.FileHandler(f'{data_dir}/{save_dir}/{llm}.log')
    stream_handler = logging.StreamHandler()
    
    file_handler.setFormatter(formatter)
    stream_handler.setFormatter(formatter)

    logger.addHandler(file_handler)
    logger.addHandler(stream_handler)

    if "all" in metric:
        metric = ["f1", "ed", "link", "argument", "rouge", "bertscore"]
        # metric = ["f1", "ed", "link", "argument", "rouge"]

    logger.info(f"Starts with: {args}")

    tool_desc = json.load(open(f"{data_dir}/tool_desc.json", "r"))
    tool_map = {tool["id"]: i+1 for i, tool in enumerate(tool_desc["nodes"])}
    tool_map_reverse = {i+1: tool["id"] for i, tool in enumerate(tool_desc["nodes"])}
    tool_map_reverse[0] = "NEGATIVE"
    tool_map["<PAD>"] = -1

    tool_output_type_map = None
    if not ignore_tool_type:
        tool_output_type_map = {tool["id"]: tool["output-type"][0] if len(tool["output-type"]) else "none" for tool in tool_desc["nodes"]}

    splits = list(splits)
    n_tools = list(n_tools)

    if "all" in splits:
        splits = ["overall", "single", "chain", "dag", ]
    if "all" in n_tools:
        n_tools = ["overall"] + [str(i) for i in range(1, 11)]
    
    group = []

    if mode == "mul":
        for s in splits:
            for n in n_tools:
                if (s, n) not in group:
                    group.append((s, n))
    elif mode == "add":
        for s in splits:
            if (s, "overall") not in group:
                group.append((s, "overall"))
        for n in n_tools:
            if ("overall", n) not in group:
                group.append(("overall", n))
    else:
        assert False, "mode should be mul or add"

    for s, n in group:
        logger.info("-"*15)
        logger.info(f"Tools Number: {n}, Task Split: {s}")
        evaluate(data_dir, prediction_dir, llm, s, n, metric, tool_desc, tool_map, tool_output_type_map, tool_map_reverse, all_metric_dict, ignore_tool_type=ignore_tool_type, alignment=alignment)

    metric_json = open(metric_file, "w")
    metric_json.write(json.dumps(all_metric_dict, indent=2))

def evaluate(data_dir, prediction_dir, llm, split, n_tool, metric, tool_desc, tool_map, tool_output_type_map, tool_map_reverse, all_metric_dict, ignore_tool_type=False, alignment = None):
    if f"{split}_{n_tool}" in all_metric_dict:
        metric_dict = all_metric_dict[f"{split}_{n_tool}"]
    else:
        metric_dict = {}
        all_metric_dict[f"{split}_{n_tool}"] = metric_dict

    label_rf = open(f"{data_dir}/data_formulated.json", "r")
    
    alignment_ids = None
    if alignment is not None:
        
        alignment_file = open(f"{data_dir}/alignment_ids.json", "r")
        alignment_ids = json.load(alignment_file)
        alignment_ids = list(itertools.chain(*alignment_ids[f"{alignment}_alignment_id"].values()))
        logger.info(f"Alignment Mode: {alignment} ({len(alignment_ids)})")
    predcition_rf = open(f"{data_dir}/{prediction_dir}/{llm}.json", "r")

    predcitions = {}
    labels = {}
    for line in label_rf:
        data = json.loads(line)
        real_tool_num = len(data["task_nodes"])
        sampled_tool_num = data["n_tools"]
        if data["id"] in alignment_ids:
            if split == "overall" or data["type"] == split:
                if n_tool == "overall" or str(real_tool_num) == n_tool:
                    id = data["id"]
                    labels[id] = data

    for line in predcition_rf:
        try:
            data = json.loads(line)
        except Exception as e:
            print(e)
            print(line)
            exit()
            # continue
        id = data["id"]
        predcitions[id] = data

    ids = set(labels.keys()).intersection(set(predcitions.keys()))
    labels = {id: labels[id] for id in ids}
    predcitions = {id: predcitions[id] for id in ids}

    predcition_task_steps = []
    label_task_steps = []
    predcition_names = []
    label_names = []
    label_graphs = []
    predcition_graphs = []
    label_links = []
    predcition_links = []
    label_task_arg_names = []
    predcition_task_arg_names = []
    label_task_arg_name_values = []
    predcition_task_arg_name_values = []

    for id in ids:
        try:
            label = labels[id]
            predcition = predcitions[id]

            predcition_task_step = predcition["result"]["task_steps"]
            label_task_step = label["task_steps"]
            
            try:
                if isinstance(predcition_task_step[0], str):
                    predcition_task_steps.append("\n".join(predcition_task_step))
                else:
                    if "task" in predcition_task_step[0]:
                        predcition_task_steps.append("\n".join([step["task"] for step in predcition_task_step]))
                    elif "step" in predcition_task_step[0]:
                        predcition_task_steps.append("\n".join([step["step"] for step in predcition_task_step]))
                    elif "id" in predcition_task_step[0]:
                        predcition_task_steps.append("\n".join([step["id"] for step in predcition_task_step]))
                    elif "step_name" in predcition_task_step[0]:
                        predcition_task_steps.append("\n".join([step["step_name"] for step in predcition_task_step]))
                    else:
                        predcition_task_steps.append("\n".join([step["description"] for step in predcition_task_step]))
            except Exception as e:
                predcition_task_steps.append(str(predcition_task_step))

            label_task_steps.append("\n".join(label_task_step))

            label_nodes = label["task_nodes"]
            predcition_nodes = predcition["result"]["task_nodes"] 

            label_node_name = [node["task"] for node in label_nodes]
            predcition_node_name = [node["task"] for node in predcition_nodes]

            label_task_arg_name = []
            predcition_task_arg_name = []

            label_task_arg_name_value = []
            predcition_task_arg_name_value = []
                
            if not ignore_tool_type:
                predcition_node_name = [name.replace("_", " ") for name in predcition_node_name]
                label_node_name = [name.replace("_", " ") for name in label_node_name]
                label_link = []
                predcition_link = []
                for inx, node in enumerate(label_nodes):
                    new_arguments = []
                    for i, argument in enumerate(node["arguments"]):
                        try:
                            if isinstance(argument, dict):
                                argument = list(argument.values())[0]
                            if isinstance(argument, list):
                                argument = " ".join(argument)
                            if "<node-" in argument:
                                index_start = argument.index("<node-") + 6
                                index_end = argument.index(">")
                                if int(argument[index_start: index_end]) == inx:
                                    continue
                                argument_tool_name = label_node_name[int(argument[index_start: index_end])]
                                label_link.append({"source": argument_tool_name, "target": node["task"]})
                                new_argument = {"name": tool_output_type_map.get(argument_tool_name, "other"), "value": argument_tool_name}
                            else:
                                new_argument = {"name": get_content_type(argument), "value": argument}
                            
                        except Exception as e:
                            pass
                        new_arguments.append(new_argument)
                    node["arguments"] = new_arguments
                    
                for inx, node in enumerate(predcition_nodes):
                    new_arguments = []
                    for i, argument in enumerate(node.get("arguments", [])):
                        try:
                            if isinstance(argument, dict):
                                argument = list(argument.values())[0]
                            if isinstance(argument, list):
                                argument = " ".join(argument)
                            if isinstance(argument, str) and "<node-" in argument:
                                index_start = argument.index("<node-") + 6
                                index_end = argument.index(">")
                            
                                if int(argument[index_start: index_end]) == inx:
                                    continue
                                prediction_tool_name = predcition_node_name[int(argument[index_start: index_end])]
                                predcition_link.append({"source": prediction_tool_name, "target": node["task"]})
                                new_argument = {"name": tool_output_type_map.get(prediction_tool_name, "other"), "value": prediction_tool_name}
                            else:
                                new_argument = {"name": get_content_type(argument), "value": argument}
                        except Exception as e:
                            pass
                        new_arguments.append(new_argument)
                    node["arguments"] = new_arguments
            else:
                predcition_link = predcition["result"]["task_links"]
                label_link = label["task_links"]
                for link in predcition_link:
                    assert link["source"] in predcition_node_name, link
                    assert link["target"] in predcition_node_name, link
                for link in label_link:
                    assert link["source"] in label_node_name, link
                    assert link["target"] in label_node_name, link

            predcition_node_argument = [node.get("arguments", []) for node in predcition_nodes]
            label_node_argument = [node["arguments"] for node in label_nodes]

            for task, arguments in zip (predcition_node_name, predcition_node_argument):
                for argument in arguments:
                    label_task_arg_name.append(f"{task}-{argument['name']}")
                    label_task_arg_name_value.append(f"{task}-{argument['name']}-{argument['value']}")
            
            for task, arguments in zip (label_node_name, label_node_argument):
                for argument in arguments:
                    predcition_task_arg_name.append(f"{task}-{argument['name']}")
                    predcition_task_arg_name_value.append(f"{task}-{argument['name']}-{argument['value']}")

            label_graph = {
                "nodes": label_node_name,
                "links": label_link,
                "arguments": label_node_argument
            }
            predcition_graph = {
                "nodes": predcition_node_name,
                "links": predcition_link,
                "arguments": predcition_node_argument
            }

            label_graphs.append(label_graph)
            predcition_graphs.append(predcition_graph)

            for node_name in predcition_node_name:
                assert isinstance(node_name, str), node_name

            predcition_names.append(predcition_node_name)
            label_names.append(label_node_name)

            predcition_task_arg_names.append(predcition_task_arg_name)
            label_task_arg_names.append(label_task_arg_name)
        
            predcition_task_arg_name_values.append(predcition_task_arg_name_value)
            label_task_arg_name_values.append(label_task_arg_name_value)

            label_links.append(label_link)
            predcition_links.append(predcition_link)

        except Exception as e:
            logger.info(f"Parsing Error: {e}, Ignore #id {id}")
            logger.info(traceback.format_exc())
            # exit()
            
    logger.info(f"Step Supports: {len(label_task_steps)} / {len(ids)}")
    logger.info(f"Node Support: {len(label_names)} / {len(ids)}")
    logger.info(f"Link Support: {len(label_links)} / {len(ids)}")
    logger.info(f"Argument Support: {len(label_graphs)} / {len(ids)}")

    metric_dict["all_samples"] = len(ids)
    metric_dict["step_supports"] = len(label_task_steps)
    metric_dict["node_supports"] = len(label_names)
    metric_dict["link_supports"] = len(label_links)
    metric_dict["argument_supports"] = len(label_graphs)

    if len(label_graphs) == 0 or len(label_names) == 0 or len(label_links) == 0:
        logger.info("No supports, skip")
        return

    if "rouge" in metric:
        # rouge = evaluate.load("rouge")
        rouge = load_metric("rouge")
        rouge_scores = rouge.compute(predictions=predcition_task_steps, references=label_task_steps, use_aggregator=True)
        for key in rouge_scores:
            logger.info(f"Step {key}: {rouge_scores[key].mid.fmeasure}")
            metric_dict[f"step_{key}"] = rouge_scores[key].mid.fmeasure

    if "bertscore" in metric:
        bertscore = load_metric("bertscore")
        bertscore_scores = bertscore.compute(predictions=predcition_task_steps, references=label_task_steps, model_type="roberta-large")
        for key in bertscore_scores:
            if key in ["precision", "recall", "f1"]:
                bertscore_scores[key] = np.mean(bertscore_scores[key])
                logger.info(f"Step BERTScore {key}: {bertscore_scores[key]}")
                metric_dict[f"step_bertscore_{key}"] = bertscore_scores[key]
    
    if "f1" in metric or "argument" in metric:
        types = list(range(1, len(tool_desc["nodes"])+1))
        types_name = [tool_map_reverse[i] for i in types]
        gt_flat, pred_flat = flatten(label_names, predcition_names, types = types_name)

        per_type = prfs(gt_flat, pred_flat, labels=types, average=None)
        micro = prfs(gt_flat, pred_flat, labels=types, average='micro')[:-1]
        macro = prfs(gt_flat, pred_flat, labels=types, average='macro')[:-1]
        total_support = sum(per_type[-1])

        logger.info(f"Node Micro Precision [ No Matching ]: {micro[0]}")
        logger.info(f"Node Micro Recall [ No Matching ]: {micro[1]}")
        logger.info(f"Node Micro F1 [ No Matching ]: {micro[2]}")
        logger.info(f"Node Macro Precision [ No Matching ]: {macro[0]}")
        logger.info(f"Node Macro Recall [ No Matching ]: {macro[1]}")
        logger.info(f"Node Macro F1 [ No Matching ]: {macro[2]}")
        logger.info("Node Detailed Report [ No Matching ]: ")
        metric_dict["node_micro_precision_no_matching"] = micro[0]
        metric_dict["node_micro_recall_no_matching"] = micro[1]
        metric_dict["node_micro_f1_no_matching"] = micro[2]
        metric_dict["node_macro_precision_no_matching"] = macro[0]
        metric_dict["node_macro_recall_no_matching"] = macro[1]
        metric_dict["node_macro_f1_no_matching"] = macro[2]

        per_type_metric = {}
        metric_dict["node_per_type_no_matchcing"] = per_type_metric
        print_results(per_type, list(micro) + [total_support], list(macro) + [total_support], types_name, result_dict = per_type_metric)


        gt_flat, pred_flat = flatten(label_task_arg_names, predcition_task_arg_names)
        micro = prfs(gt_flat, pred_flat, average="binary")[:-1]
        logger.info(f"Argument Task-ArgName Binary F1: [ No Matching ]: {micro[-1]}")
        metric_dict["argument_task_argname_binary_f1_no_matching"] = micro[-1]

        gt_flat, pred_flat = flatten(label_task_arg_name_values, predcition_task_arg_name_values)
        micro = prfs(gt_flat, pred_flat, average="binary")[:-1]
        logger.info(f"Argument Task-ArgName-Value Binary F1 [ No Matching ]: {micro[-1]}")
        metric_dict["argument_task_argname_value_binary_f1_no_matching"] = micro[-1]

        accuracy = load_metric("accuracy")
        labels = []
        predcitions = []
        predcition_arguments = []
        label_arguments = []
        label_task_arg_names = []
        predcition_task_arg_names = []
        label_task_arg_names_values = []
        predcition_task_arg_names_values = []
        for label_graph, predcition_graph in zip(label_graphs, predcition_graphs):
            num_nodes_1 = len(label_graph["nodes"])
            num_nodes_2 = len(predcition_graph["nodes"])
            if num_nodes_1 > num_nodes_2:
                predcition_graph["nodes"] += ["<PAD>"] * (num_nodes_1 - num_nodes_2)
            elif num_nodes_1 < num_nodes_2:
                label_graph["nodes"] += ["<PAD>"] * (num_nodes_2 - num_nodes_1)

            index_1, index_2 = matching(label_graph, predcition_graph)
            label_task_names = []
            predcition_task_names = []
            

            for i, j in zip(index_1, index_2):
                task_name = label_graph["nodes"][i]
                label_task_names.append(task_name) 
                predcition_task_names.append(predcition_graph["nodes"][j])
                if "argument" in metric and task_name != "<PAD>" and predcition_graph["nodes"][j] != "<PAD>":
                    predcition_args_list = predcition_graph["arguments"][j]
                    for argument_j_ele in predcition_args_list:
                        if not isinstance(argument_j_ele, dict):
                            predcition_arguments.append([])
                            predcition_task_arg_names.append([])
                            predcition_task_arg_names_values.append([])
                            break
                    else:
                        predcition_arguments.append(predcition_args_list)
                        predcition_task_arg_names.append([f"{task_name}:{argument.get('name', '')}" for argument in predcition_args_list])
                        predcition_task_arg_names_values.append([f"{task_name}:{argument.get('name', '')}:{argument.get('value', '')}" for argument in predcition_args_list])
                    
                    label_args_list = label_graph["arguments"][i]
                    for argument_i_ele in label_args_list:
                        if not isinstance(argument_i_ele, dict):
                            label_arguments.append([])
                            label_task_arg_names.append([])
                            label_task_arg_names_values.append([])
                            break
                    else:
                        label_arguments.append(label_args_list)
                        label_task_arg_names.append([f"{task_name}:{argument.get('name', '')}" for argument in label_args_list])
                        label_task_arg_names_values.append([f"{task_name}:{argument.get('name', '')}:{argument.get('value', '')}" for argument in label_args_list])
  
            labels.extend([tool_map.get(name, 0) for name in label_task_names])
            predcitions.extend([tool_map.get(name, 0) for name in predcition_task_names])

        types = list(range(1, len(tool_desc["nodes"])+1))
        accuracy_scores = accuracy.compute(predictions=predcitions, references=labels)
        logger.info(f"Node Accuracy: {accuracy_scores['accuracy']}")
        metric_dict["node_accuracy"] = accuracy_scores["accuracy"]

        per_type = prfs(labels, predcitions, labels=types, average=None)
        micro = prfs(labels, predcitions, labels=types, average='micro')[:-1]
        macro = prfs(labels, predcitions, labels=types, average='macro')[:-1]
        total_support = sum(per_type[-1])

        logger.info(f"Node Micro Precision: {micro[0]}")
        logger.info(f"Node Micro Recall: {micro[1]}")
        logger.info(f"Node Micro F1: {micro[2]}")
        logger.info(f"Node Macro Precision: {macro[0]}")
        logger.info(f"Node Macro Recall: {macro[1]}")
        logger.info(f"Node Macro F1: {macro[2]}")
        logger.info("Node Detailed Report:")

        metric_dict["node_micro_precision"] = micro[0]
        metric_dict["node_micro_recall"] = micro[1]
        metric_dict["node_micro_f1"] = micro[2]
        metric_dict["node_macro_precision"] = macro[0]
        metric_dict["node_macro_recall"] = macro[1]
        metric_dict["node_macro_f1"] = macro[2]

        types_name = [tool_map_reverse[i] for i in types]
        per_type_metric = {}
        metric_dict["node_per_type"] = per_type_metric
        print_results(per_type, list(micro) + [total_support], list(macro) + [total_support], types_name, result_dict = per_type_metric)

        if "argument" in metric:
            label_node_arguments_name = []
            predcition_node_arguments_name = []
            label_node_arguments_value = []
            predcition_node_arguments_value = []
            label_node_arguments_name_value = []
            predcition_node_arguments_name_value = []
            for label_sample_node_arguments, prediction_sample_node_arguments in zip(label_arguments, predcition_arguments):
                try:
                    label_argument_name = []
                    label_argument_value = []
                    for item in label_sample_node_arguments:
                        if isinstance(item, dict):
                            name = item.get("name", "")
                            value = item.get("value", "")
                            if not isinstance(name, str) or not isinstance(value, str):
                                continue
                            label_argument_name.append(name)
                            label_argument_value.append(value)
                        else:
                            label_argument_name.append("")
                            label_argument_value.append("")
                    label_node_arguments_name.append(label_argument_name)
                    label_node_arguments_value.append(label_argument_value)
                    label_node_arguments_name_value.append([f"{name} {value}" for name, value in zip(label_argument_name, label_argument_value)])

                    predcition_argument_name = []
                    predcition_argument_value = []
                    for item in prediction_sample_node_arguments:
                        if isinstance(item, dict):
                            name = item.get("name", "")
                            value = item.get("value", "")
                            if not isinstance(name, str) or not isinstance(value, str):
                                continue
                            predcition_argument_name.append(name)
                            predcition_argument_value.append(value)
                        else:
                            predcition_argument_name.append("")
                            predcition_argument_value.append("")
                        
                    predcition_node_arguments_name.append(predcition_argument_name)
                    predcition_node_arguments_value.append(predcition_argument_value)
                    predcition_node_arguments_name_value.append([f"{name} {value}" for name, value in zip(predcition_argument_name, predcition_argument_value)])

                except Exception as e:
                    print(label_node_argument)
                    print(predcition_node_argument)
                    print(e)
                    raise e
            gt_flat, pred_flat = flatten(label_node_arguments_name, predcition_node_arguments_name)
            micro = prfs(gt_flat, pred_flat, average="binary")[:-1]
            logger.info(f"Argument Name Binary F1: {micro[-1]}")
            metric_dict["argument_name_binary_f1"] = micro[-1]
            gt_flat, pred_flat = flatten(label_node_arguments_value, predcition_node_arguments_value)
            micro = prfs(gt_flat, pred_flat, average="binary")[:-1]
            logger.info(f"Argument Value Binary F1: {micro[-1]}")
            metric_dict["argument_value_binary_f1"] = micro[-1]
            gt_flat, pred_flat = flatten(label_node_arguments_name_value, predcition_node_arguments_name_value)
            micro = prfs(gt_flat, pred_flat, average="binary")[:-1]
            logger.info(f"Argument Name-Value Binary F1: {micro[-1]}")
            metric_dict["argument_name_value_binary_f1"] = micro[-1]

            gt_flat, pred_flat = flatten(label_task_arg_names, predcition_task_arg_names)
            micro = prfs(gt_flat, pred_flat, average="binary")[:-1]
            logger.info(f"Argument Task-ArgName Binary F1: {micro[-1]}")
            metric_dict["argument_task_argname_binary_f1"] = micro[-1]

            gt_flat, pred_flat = flatten(label_task_arg_names_values, predcition_task_arg_names_values)
            micro = prfs(gt_flat, pred_flat, average="binary")[:-1]
            logger.info(f"Argument Task-ArgName-Value Binary F1: {micro[-1]}")
            metric_dict["argument_task_argname_value_binary_f1"] = micro[-1]


    
    if "ed" in metric:
        labels = []
        predcitions = []
        for label_name, predcition_name in zip(label_names, predcition_names):
            labels.append([tool_map.get(name, 0) for name in label_name])
            predcitions.append([tool_map.get(name, 0) for name in predcition_name])
        ed = ratio_levenshtein(predcitions, labels)
        logger.info(f"Edit Distance: {1-ed}")
        metric_dict["edit_distance"] = 1-ed
    
    if "link" in metric:
        tuple_label_links = []
        tuple_predcition_links = []
        for label_link, predcition_link in zip(label_links, predcition_links):
            tuple_label_links.append([(link["source"], link["target"]) for link in label_link])
            tuple_predcition_links.append([(link["source"], link["target"]) for link in predcition_link])
        
        gt_flat, pred_flat = flatten(tuple_label_links, tuple_predcition_links)


        micro = prfs(gt_flat, pred_flat, average="binary")[:-1]
        logger.info(f"Link Binary F1: {micro[-1]}")
        metric_dict["link_binary_f1"] = micro[-1]

if __name__ == "__main__":
    main()