from __future__ import annotations

import numpy as np
import torch


def adjust_learning_rate(optimizer, epoch, args):
    if getattr(args, 'lradj', 'type1') == 'type1':
        lr = args.learning_rate * (0.5 ** max(0, epoch - 1))
    else:
        lr = args.learning_rate
    for param_group in optimizer.param_groups:
        param_group['lr'] = lr
    print(f'Adjust learning rate to {lr:.6f}')


class EarlyStopping:
    def __init__(self, patience=7, verbose=False, delta=0.0):
        self.patience = patience
        self.verbose = verbose
        self.counter = 0
        self.best_score = None
        self.early_stop = False
        self.val_loss_min = np.inf
        self.delta = delta

    def __call__(self, val_loss, model, path):
        score = -val_loss
        if self.best_score is None:
            self.best_score = score
            self._save_checkpoint(val_loss, model, path)
            return

        if score < self.best_score + self.delta:
            self.counter += 1
            print(
                f'EarlyStopping counter: {self.counter} out of {self.patience}')
            if self.counter >= self.patience:
                self.early_stop = True
        else:
            self.best_score = score
            self._save_checkpoint(val_loss, model, path)
            self.counter = 0

    def _save_checkpoint(self, val_loss, model, path):
        if self.verbose:
            print(
                f'Validation loss decreased ({self.val_loss_min:.6f} -> {val_loss:.6f}). Saving model...')
        torch.save(model.state_dict(), f'{path}/checkpoint.pth')
        self.val_loss_min = val_loss
