# -*- coding: utf-8 -*-
# *-----------------------------------------------------------------------*
# |                                                                       |
# |  Copyright (c) 2016 by Paul Scherrer Institute (http://www.psi.ch)    |
# |                                                                       |
# |              Author: alain.studer@psi.ch                              |
# *-----------------------------------------------------------------------*
'''
This code can calculates the Fourier Transform of the Radon transform.
The implementation bases upon the Fourier slice theorem.
(Radially sampled data points are interpolated on the cartesian grid)
'''


import numpy
import math
import ctypes



def legendre2(coefs,x):
    return numpy.polynomial.legendre.legval(x, coefs)
        
def setupInterpolationLookupTable(numberOfSamplingPoints):
    coefficients = numpy.array([0.5767616E+02, 0.0, -0.8931343E+02, 0.0, 0.4167596E+02, 0.0, -0.1053599E+02, 0.0, 0.1662374E+01, 0.0, -0.1780527E-00, 0.0, 0.1372983E-01, 0.0, -0.7963169E-03, 0.0, 0.3593372E-04, 0.0, -0.1295941E-05, 0.0, 0.3817796E-07])
    lookupTableOfConvolventInFourierSpace = legendre2(coefficients, numpy.arange(numberOfSamplingPoints)/float(numberOfSamplingPoints) )/legendre2(coefficients, 0.)
    #due to rounding effects we may be beond the size of the convolvent, so we zero pad to avoid seg-fault
    lookupTableOfConvolventInFourierSpace = numpy.lib.pad(lookupTableOfConvolventInFourierSpace, (0, numberOfSamplingPoints), 'constant', constant_values=(0.0, 0.0)).astype(numpy.float32)
    return lookupTableOfConvolventInFourierSpace

def interpolateUsingPSWF_C(doBackward, doInverse, cartesianGridFFT, fft1dOfSinogram, lookupTableOfConvolventInFourierSpace, numberOfAngularPositions, angularArray, numberOfSubsampledAngles, subsampledAngles, numberOfFrequencies, C):
    try:
        _libInterpolate = numpy.ctypeslib.load_library('../chip/gridrec/libinterpolate.so', '.')
    except:
        _libInterpolate = numpy.ctypeslib.load_library('chip/gridrec/libinterpolate.so', '.')
    #_libInterpolate = numpy.ctypeslib.load_library('libinterpolate', './gridRec_TRP/')
    _libInterpolate.interpolateUsingPSWF.argtypes = [ctypes.c_int, ctypes.c_int, numpy.ctypeslib.ndpointer(dtype = numpy.complex64), numpy.ctypeslib.ndpointer(dtype = numpy.complex64), numpy.ctypeslib.ndpointer(dtype = numpy.float32), ctypes.c_int, numpy.ctypeslib.ndpointer(dtype = numpy.float32), ctypes.c_int, numpy.ctypeslib.ndpointer(dtype = numpy.int32), ctypes.c_int, ctypes.c_float]
    _libInterpolate.interpolateUsingPSWF.restype  = ctypes.c_int
    exit_status = _libInterpolate.interpolateUsingPSWF(doBackward, doInverse, cartesianGridFFT, fft1dOfSinogram, lookupTableOfConvolventInFourierSpace, numberOfAngularPositions, angularArray, numberOfSubsampledAngles, subsampledAngles, numberOfFrequencies, C)
    return exit_status
     
     
def gridRecFFT(doInverse, sinogramRef, angularArrayRef, subsampledAngles, centerOfRotation):
    #First, we make a deep copy of the handed over sinogram, since we dont work on it
    #(Without deep copy, it will be modified within this method, this is an undesired side effect)
    sinogram = numpy.copy(sinogramRef)
    angularArray = numpy.copy(angularArrayRef).astype(numpy.float32)
    numberOfXPixels = sinogram.shape[1]
    numberOfAngularPositions  = sinogram.shape[0]
    numberOfFrequencies = numberOfXPixels
    lookupTableOfConvolventInFourierSpace = setupInterpolationLookupTable(numberOfFrequencies//2)
    #fft along first axis
    detectorSpaceFFT = numpy.fft.fft(sinogram)
    # centerOfRotation can be float
    detectorSpaceFFT *= numpy.exp(1.j*2.0*numpy.pi*centerOfRotation*numpy.fft.fftfreq(numberOfXPixels, d=1.0))
    detectorSpaceFFT = numpy.roll(detectorSpaceFFT, numberOfFrequencies//2, axis=1)
    detectorSpaceFFT = detectorSpaceFFT.astype(numpy.complex64)
    #cartesianGridInterpolatedFFT = numpy.ascontiguousarray((numberOfFrequencies, numberOfFrequencies), dtype = 'complex64')
    cartesianGridInterpolatedFFT = numpy.zeros((numberOfFrequencies, numberOfFrequencies)).astype(numpy.complex64)
    C = 7.0 #PSWF related constant (must equal value in postCorrectRec.py) 
    interpolateUsingPSWF_C(1, doInverse, cartesianGridInterpolatedFFT, detectorSpaceFFT, lookupTableOfConvolventInFourierSpace, detectorSpaceFFT.shape[0], angularArray, subsampledAngles.shape[0], subsampledAngles, detectorSpaceFFT.shape[1], C)
    return cartesianGridInterpolatedFFT
        

