import os,sys
import logging
from distutils.util import strtobool
from fairseq import utils
from transformers.activations import ACT2FN
from fairseq.dataclass import ChoiceEnum, FairseqDataclass
from dataclasses import dataclass, field
from omegaconf import II
from typing import Dict, List, Optional, Tuple
from torch import nn
import torch.nn.functional as F
import torch
import math
import numpy as np
import pdb
from fairseq.data.dictionary import Dictionary
import torch.nn.functional as F
from fairseq.modules import (
    Fp32GroupNorm,
    Fp32LayerNorm,
    GradMultiply,
    GumbelVectorQuantizer,
    LayerNorm,
    SamePad,
    TransposeLast,
)
from fairseq.modules.transformer_sentence_encoder import init_bert_params
from fairseq.utils import buffered_arange, index_put, is_xla_tensor
from fairseq.models import BaseFairseqModel, FairseqEncoder, FairseqEncoderDecoderModel, register_model
DBG=True if len(sys.argv) == 1 else False
if DBG:
    from hubert_pretraining import (
        AVHubertPretrainingConfig,
        AVHubertPretrainingTask,
    )
    from resnet import ResEncoder
    logging.basicConfig(
        format="%(asctime)s | %(levelname)s | %(name)s | %(message)s",
        datefmt="%Y-%m-%d %H:%M:%S",
        level=os.environ.get("LOGLEVEL", "INFO").upper(),
        stream=sys.stdout,
    )
    from utils import compute_mask_indices
    from decoder import TransformerDecoder

else:
    from .hubert_pretraining import (
        AVHubertPretrainingConfig,
        AVHubertPretrainingTask,
    )
    from .resnet import ResEncoder
    from .utils import compute_mask_indices
    from .decoder import TransformerDecoder
    from .multihead_attention import MultiheadAttention
from omegaconf import II    
from copy import deepcopy
logger = logging.getLogger(__name__)
EXTRACTOR_MODE_CHOICES = ChoiceEnum(["default", "layer_norm"])
MASKING_DISTRIBUTION_CHOICES = ChoiceEnum(
    ["static", "uniform", "normal", "poisson"]
)
@dataclass
class AVHubertConfig(FairseqDataclass):
    label_rate: int = II("task.label_rate")
    input_modality: str = II("task.input_modality")
    extractor_mode: EXTRACTOR_MODE_CHOICES = field(
        default="default",
        metadata={
            "help": "mode for feature extractor. default has a single group "
            "norm with d groups in the first conv block, whereas layer_norm "
            "has layer norms in every block (meant to use with normalize=True)"
        },
    )
    encoder_layers: int = field(
        default=12, metadata={"help": "num encoder layers in the transformer"}
    )
    encoder_embed_dim: int = field(
        default=768, metadata={"help": "encoder embedding dimension"}
    )
    encoder_ffn_embed_dim: int = field(
        default=3072, metadata={"help": "encoder embedding dimension for FFN"}
    )
    encoder_attention_heads: int = field(
        default=12, metadata={"help": "num encoder attention heads"}
    )
    activation_fn: ChoiceEnum(utils.get_available_activation_fns()) = field(
        default="gelu", metadata={"help": "activation function to use"}
    )

    # dropouts
    dropout: float = field(
        default=0.1,
        metadata={"help": "dropout probability for the transformer"},
    )
    attention_dropout: float = field(
        default=0.1,
        metadata={"help": "dropout probability for attention weights"},
    )
    activation_dropout: float = field(
        default=0.0,
        metadata={"help": "dropout probability after activation in FFN"},
    )
    encoder_layerdrop: float = field(
        default=0.0,
        metadata={"help": "probability of dropping a tarnsformer layer"},
    )
    dropout_input: float = field(
        default=0.0,
        metadata={"help": "dropout to apply to the input (after feat extr)"},
    )
    dropout_features: float = field(
        default=0.0,
        metadata={
            "help": "dropout to apply to the features (after feat extr)"
        },
    )

    final_dim: int = field(
        default=0,
        metadata={
            "help": "project final representations and targets to this many "
            "dimensions. set to encoder_embed_dim is <= 0"
        },
    )
    untie_final_proj: bool = field(
        default=False,
        metadata={"help": "use separate projection for each target"},
    )
    layer_norm_first: bool = field(
        default=False,
        metadata={"help": "apply layernorm first in the transformer"},
    )
    conv_feature_layers: str = field(
        default="[(512,10,5)] + [(512,3,2)] * 4 + [(512,2,2)] * 2",
        metadata={
            "help": "string describing convolutional feature extraction "
            "layers in form of a python list that contains "
            "[(dim, kernel_size, stride), ...]"
        },
    )
    conv_bias: bool = field(
        default=False, metadata={"help": "include bias in conv encoder"}
    )
    logit_temp: float = field(
        default=0.1, metadata={"help": "temperature to divide logits by"}
    )
    target_glu: bool = field(
        default=False, metadata={"help": "adds projection + glu to targets"}
    )
    feature_grad_mult: float = field(
        default=1.0,
        metadata={"help": "multiply feature extractor var grads by this"},
    )

    # masking
    mask_length_audio: int = field(default=10, metadata={"help": "mask length"})
    mask_prob_audio: float = field(
        default=0.65,
        metadata={"help": "probability of replacing a token with mask"},
    )
    mask_length_image: int = field(default=10, metadata={"help": "mask length"})
    mask_prob_image: float = field(
        default=0.65,
        metadata={"help": "probability of replacing a token with mask"},
    )
    mask_selection: MASKING_DISTRIBUTION_CHOICES = field(
        default="static", metadata={"help": "how to choose mask length"}
    )
    mask_other: float = field(
        default=0,
        metadata={
            "help": "secondary mask argument "
            "(used for more complex distributions), "
            "see help in compute_mask_indicesh"
        },
    )
    no_mask_overlap: bool = field(
        default=False, metadata={"help": "whether to allow masks to overlap"}
    )
    mask_min_space: int = field(
        default=1,
        metadata={
            "help": "min space between spans (if no overlap is enabled)"
        },
    )

    # channel masking
    mask_channel_length: int = field(
        default=10,
        metadata={"help": "length of the mask for features (channels)"},
    )
    mask_channel_prob: float = field(
        default=0.0,
        metadata={"help": "probability of replacing a feature with 0"},
    )
    mask_channel_selection: MASKING_DISTRIBUTION_CHOICES = field(
        default="static",
        metadata={"help": "how to choose mask length for channel masking"},
    )
    mask_channel_other: float = field(
        default=0,
        metadata={
            "help": "secondary mask argument "
            "(used for more complex distributions), "
            "see help in compute_mask_indicesh"
        },
    )
    no_mask_channel_overlap: bool = field(
        default=False,
        metadata={"help": "whether to allow channel masks to overlap"},
    )
    mask_channel_min_space: int = field(
        default=1,
        metadata={
            "help": "min space between spans (if no overlap is enabled)"
        },
    )

    # positional embeddings
    conv_pos: int = field(
        default=128,
        metadata={
            "help": "number of filters for convolutional positional embeddings"
        },
    )
    conv_pos_groups: int = field(
        default=16,
        metadata={
            "help": "number of groups for convolutional positional embedding"
        },
    )

    latent_temp: Tuple[float, float, float] = field(
        default=(2, 0.5, 0.999995),
        metadata={"help": "legacy (to be removed)"},
    )

    # loss computation
    skip_masked: bool = field(
        default=False,
        metadata={"help": "skip computing losses over masked frames"},
    )
    skip_nomask: bool = field(
        default=False,
        metadata={"help": "skip computing losses over unmasked frames"},
    )
    resnet_relu_type: str = field(default='prelu', metadata={"help": 'relu type for resnet'})
    resnet_weights: Optional[str] = field(default=None, metadata={"help": 'resnet weights'})
    sim_type: str = field(default='cosine', metadata={"help": 'similarity type'})

    sub_encoder_layers: int = field(default=0, metadata={'help': 'number of transformer layers for single modality'})
    audio_feat_dim: int = field(default=-1, metadata={'help': 'audio feature dimension'})
    modality_dropout: float = field(default=0, metadata={'help': 'drop one modality'})
    audio_dropout: float = field(default=0, metadata={'help': 'drop audio feature'})
    modality_fuse: str = field(default='concat', metadata={'help': 'fusing two modalities: add,concat,fuse'})
    selection_type : str = field(default='same_other_seq', metadata={'help': 'type of selectig images, same_other_seq: replace masked span with span from another sequence, same_seq: repace masked span with span of the same sequence'})
    masking_type : str = field(default='input', metadata={'help': 'input or feature masking'})

    decoder_embed_dim: int = field(
        default=768, metadata={"help": "decoder embedding dimension"}
    )
    decoder_ffn_embed_dim: int = field(
        default=3072, metadata={"help": "decoder embedding dimension for FFN"}
    )
    decoder_layers: int = field(
        default=6, metadata={"help": "num of decoder layers"}
    )
    decoder_layerdrop: float = field(
        default=0.0, metadata={"help": "decoder layerdrop chance"}
    )
    decoder_attention_heads: int = field(
        default=4, metadata={"help": "num decoder attention heads"}
    )
    decoder_learned_pos: bool = field(
        default=False,
        metadata={"help": "use learned positional embeddings in the decoder"},
    )
    decoder_normalize_before: bool = field(
        default=False,
        metadata={"help": "apply layernorm before each decoder block"},
    )
    no_token_positional_embeddings: bool = field(
        default=False,
        metadata={
            "help": "if set, disables positional embeddings "
            "(outside self attention)"
        },
    )
    decoder_dropout: float = field(
        default=0.1, metadata={"help": "dropout probability in the decoder"}
    )
    decoder_attention_dropout: float = field(
        default=0.1,
        metadata={
            "help": "dropout probability for attention weights "
            "inside the decoder"
        },
    )
    decoder_activation_dropout: float = field(
        default=0.0,
        metadata={
            "help": "dropout probability after activation in FFN "
            "inside the decoder"
        },
    )
    max_target_positions: int = field(
        default=2048, metadata={"help": "max target positions"}
    )
    share_decoder_input_output_embed: bool = field(
        default=False,
        metadata={"help": "share decoder input and output embeddings"},
    )
    no_scale_embedding: bool = field(default=True, metadata={'help': 'scale embedding'})

@dataclass
class AdaAVHubertConfig(AVHubertConfig):
    eada_emb_size: int = field(
        default = 256,
    )
    #adapter_embedding_size: Dict[str,int] = field(
    #    default_factory = {str(i):256 for i in range(0,12)}
    #)
    adapter_embedding_size: Dict[str,int] = field(
        default_factory = dict,
    )
    use_adapter_attn: bool = field(
        default = True,
    )
    use_adapter_ff: bool = field(
        default = True,
    )
    #use_adapter_to_output: bool = field(
    #    default = False,
    #)
    lada_emb_size: int = field(
        default = 256,
    )
    adapter_to_output_layer_size: Dict[str,int] = field(
        default_factory = dict,
    )
    use_adapter_to_output_weighted_sum: bool = field(
        default = False,
    )
    use_skip: bool = field(
        default = False,
    )
    use_upsampling: bool = field(
        default = False,
    )
    use_residual: bool = field(
        default = False,
    )
    eadapter_act: ChoiceEnum(utils.get_available_activation_fns()) = field(
        default="gelu", metadata={"help": "activation function to use"}
    )
    ladapter_act: ChoiceEnum(utils.get_available_activation_fns()) = field(
        default="gelu", metadata={"help": "activation function to use"}
    )
    adapter_init_std: float = field(
        default = 1e-3,
    )
    ladapter_init_std: float = field(
        default = 1e-3,
    )
    adapter_init_mean: int = field(
        default = 0, 
    )
    adapter_init_value: float = field(
        default = 1e-5,
    )
    use_statistics_pooling: bool = field(
        default = False,
    )
    use_amsoftmax: bool = field(
        default = False,
    )
    adapter_dropproba: int = field(
        default = 0,
    )
    use_adapter_norm: bool = field(
        default = True,
    )
    use_adapter_fc: bool = field(
        default = True,
    )
    layer_norm_eps: float = field(
        default = 1e-5,
    )
    classifier_lr: float = field(
        default = 1e-3,
    )
    encoder_lr: float = field(
        default = 1e-4,
    )
    ladapter_lr: float = field(
        default = 1e-3,
    )
    eadapter_lr: float = field(
        default = 1e-3,
    )
    share_alpha: int = field(
        default=1,
    )
    use_adapterbias: bool = field(
        default = True,
    )
    prefix_length: int = field(
        default = 4,
    )
    prefix_act: ChoiceEnum(utils.get_available_activation_fns()) = field(
        default="gelu", metadata={"help": "activation function to use"}
    )
    bottleneck_size: int = field(
        default = 512,
    )
    scaling: int = field(
        default = 10,
    )

class AdapterLayer(nn.Module):
    def __init__(self, config:AdaAVHubertConfig, layer):
        super().__init__()
        self.config = config
        self.linear_down = nn.Linear(config.encoder_embed_dim, config.bottleneck_size)
        self.act = ACT2FN[config.eadapter_act] if config.eadapter_act else None
        self.linear_up = nn.Linear(config.bottleneck_size, config.encoder_embed_dim)
        self.layernorm = nn.LayerNorm(config.encoder_embed_dim, eps=config.layer_norm_eps)
                
    def forward(self, hidden_states):
        res = hidden_states
        hidden_states = self.act(self.linear_down(hidden_states)) \
                            if self.act else self.linear_down(hidden_states)
        hidden_states = self.linear_up(hidden_states)
        hidden_states = self.layernorm(hidden_states)
        hidden_states = hidden_states + res
        return hidden_states

class AdapterLayerFuse(nn.Module):
    def __init__(self, config:AdaAVHubertConfig,layer,t=10):
        super().__init__()
        self.config = config
        self.linear_down_audio = nn.Linear(config.encoder_embed_dim, config.bottleneck_size)
        self.linear_down_av = nn.Linear(config.encoder_embed_dim, config.bottleneck_size)
        self.linear_down_video = nn.Linear(config.encoder_embed_dim, config.bottleneck_size)
        self.act = ACT2FN[config.eadapter_act] if config.eadapter_act else None
        self.linear_up_audio = nn.Linear(config.bottleneck_size, config.encoder_embed_dim)
        self.linear_up_av = nn.Linear(config.bottleneck_size, config.encoder_embed_dim)
        self.linear_up_video = nn.Linear(config.bottleneck_size, config.encoder_embed_dim)
        self.layernorm = nn.LayerNorm(config.encoder_embed_dim, eps=config.layer_norm_eps)
        self.expert_weights=nn.Linear(config.encoder_embed_dim,3)
        self.t = t

    def forward(self, hidden_states,modality_embedding):
        res = hidden_states #modality_embedding [768]
        hidden_states_audio = self.act(self.linear_down_audio(hidden_states)) \
                            if self.act else self.linear_down_audio(hidden_states)
        hidden_states_audio = self.linear_up_audio(hidden_states_audio)
        hidden_states_av = self.act(self.linear_down_av(hidden_states)) \
                            if self.act else self.linear_down_av(hidden_states)
        hidden_states_av = self.linear_up_audio(hidden_states_av)
        hidden_states_video = self.act(self.linear_down_video(hidden_states)) \
                            if self.act else self.linear_down_video(hidden_states)
        hidden_states_video = self.linear_up_audio(hidden_states_video)
        weights = torch.softmax(self.expert_weights(modality_embedding)/self.t,-1)
        hidden_states = weights[0]*hidden_states_audio+weights[1]*hidden_states_av+weights[2]*hidden_states_video
        hidden_states = self.layernorm(hidden_states)
        hidden_states = hidden_states + res
        return hidden_states

class PrefixTuning(nn.Module):
    def __init__(self,config:AdaAVHubertConfig,layer):
        super().__init__()
        self.config = config
        self.emb_dim = config.encoder_embed_dim
        self.wte = nn.Embedding(config.prefix_length,self.emb_dim)
        self.control_trans = nn.Sequential(
            nn.Linear(self.emb_dim,config.bottleneck_size),
            ACT2FN[config.prefix_act],
            nn.Linear(config.bottleneck_size,  2 * self.emb_dim),
        )
        #self.dropout = nn.Dropout()
    def forward(self,x,is_only=False):
        #x (T,B,C)
        T,B,C = x.shape
        if is_only:
            """
            for param in self.control_trans:
                param.requires_grad = False
            """
            input_tokens = torch.arange(self.config.prefix_length).long()
            input_tokens = input_tokens.unsqueeze(0).expand(B, -1).to(x.device)        
            embs = self.wte(input_tokens) #(B,T,C)
            embs = embs.transpose(0,1) #(T,B,C)
            key_values = self.control_trans(embs)  # (T,B,2C)
            key_values = key_values.view(
                -1, B, 2, self.emb_dim
            )  # *2 for key and value
            key_values = key_values.permute(2, 0, 1, 3)
        else:
            embs = x.detach()
            key_values = self.control_trans(embs)  # (T,B,2C)
            key_values = key_values.view(
                T, B, 2, self.emb_dim
            )  # *2 for key and value
            key_values = key_values.permute(2, 0, 1, 3)

        return key_values


class Prompt_Base(nn.Module):
    def __init__(self,config:AdaAVHubertConfig, center_K=100, dropout=0.1):
        super().__init__()
        self.encoder_embed_dim = config.encoder_embed_dim
        self.dropout = dropout
        self.center_K = center_K

        self.meta = nn.Linear(self.encoder_embed_dim, self.center_K)

        self.emb_center = nn.Parameter(torch.empty(self.center_K, self.encoder_embed_dim))
        nn.init.normal_(self.emb_center, std=64 ** -0.5)

    def forward(self, x):
        soft_weight = self.meta(x)  # T,B,C
        soft_weight = torch.softmax(soft_weight, dim=-1) #(T,B,K) @ (K,C)
        return soft_weight @ self.emb_center

class AdapterToOutputLayer(nn.Module):
    def __init__(self, config:AdaAVHubertConfig, layer):
        super().__init__()
        self.config = config
  
        if config.use_adapter_fc:
            self.linear_down = nn.Linear(config.encoder_embed_dim, config.adapter_to_output_layer_size[layer])
        self.act = ACT2FN[config.ladapter_act] if config.ladapter_act else None 
    
        if config.use_upsampling:
            self.linear_up = nn.Linear(config.adapter_to_output_layer_size[layer], config.encoder_embed_dim)
        
        if config.adapter_dropproba:
            self.dropout = nn.Dropout(config.adapter_dropproba)
        
        if config.use_adapter_norm:
            self.layernorm = nn.LayerNorm(config.lada_emb_size, eps=config.layer_norm_eps)

    def forward(self, hidden_states):
        res = hidden_states #(T,B,C)
        if  self.config.use_adapter_fc:
            hidden_states = self.linear_down(hidden_states)
            if self.act:
                hidden_states = self.act(hidden_states) 
        else:
            if self.act:
                hidden_states = self.act(hidden_states)
        
        if self.config.use_upsampling:
            hidden_states = self.linear_up(hidden_states)
            if self.config.use_adapter_postact and self.config.adapter_act:
                hidden_states = self.act(hidden_states)
        
        if self.config.adapter_dropproba:
            hidden_states = self.dropout(hidden_states)
        
        if self.config.use_adapter_norm:
            hidden_states = self.layernorm(hidden_states)
            
        if self.config.use_residual and self.config.use_upsampling:
            hidden_states = hidden_states + res

        return hidden_states

class LayerScale(nn.Module):
    def __init__(self,dim):
        super().__init__()
        self.gamma = nn.Parameter(torch.ones(dim))
    def forward(self,x):
        return x * self.gamma

class TransformerSentenceEncoderLayer(nn.Module):
    """
    Implements a Transformer Encoder Layer used in BERT/XLM style pre-trained
    models.
    """

    def __init__(
        self,
        layer,
        config: AdaAVHubertConfig,
        embedding_dim: float = 768,
        ffn_embedding_dim: float = 3072,
        num_attention_heads: float = 8,
        dropout: float = 0.1,
        attention_dropout: float = 0.1,
        activation_dropout: float = 0.1,
        activation_fn: str = "relu",
        layer_norm_first: bool = False,
    ) -> None:

        super().__init__()
        # Initialize parameters
        self.embedding_dim = embedding_dim
        self.dropout = dropout
        self.activation_dropout = activation_dropout
        self.config = config

        # Initialize blocks
        self.activation_fn = utils.get_activation_fn(activation_fn)
        self.self_attn = MultiheadAttention(
            self.embedding_dim,
            num_attention_heads,
            dropout=attention_dropout,
            self_attention=True,
        )

        self.dropout1 = nn.Dropout(dropout)
        self.dropout2 = nn.Dropout(self.activation_dropout)
        self.dropout3 = nn.Dropout(dropout)

        self.layer_norm_first = layer_norm_first
        # layer norm associated with the self attention layer
        self.self_attn_layer_norm = LayerNorm(self.embedding_dim)
        self.fc1 = nn.Linear(self.embedding_dim, ffn_embedding_dim)
        self.fc2 = nn.Linear(ffn_embedding_dim, self.embedding_dim)
        # layer norm associated with the position wise feed-forward NN
        self.final_layer_norm = LayerNorm(self.embedding_dim) 
        self.av_adapter = AdapterLayer(config,layer)

        # self.adapter_ls1 = LayerScale(self.embedding_dim)
        # self.adapter_ls2 = LayerScale(self.embedding_dim)
        # self.adapter_ls3 = LayerScale(self.embedding_dim)

    def forward(
        self,
        x: torch.Tensor,    
        prefix_modality: torch.Tensor,
        self_attn_mask: torch.Tensor = None,
        self_attn_padding_mask: torch.Tensor = None,
        prefix_mask: torch.Tensor = None,
        need_weights: bool = False,
        att_args=None,
        is_audio: bool = False,
        only_audio: bool = False,
        only_video: bool = False,
        mask_indices_audio=None,
        mask_indices_video=None,
    ):
        """
        LayerNorm is applied either before or after the self-attention/ffn
        modules similar to the original Transformer imlementation.
        """
        #token = torch.arange(3).to(x.device).long()
        #modality_embedding = self.adapter_modality_embedding(token)

        if self.layer_norm_first:

            if only_audio:
                #x = self.adapterfuse(x,modality_embedding[0])
                residual = x #(T,B,C)
                x = self.self_attn_layer_norm(x)
                x, attn = self.self_attn(
                    query=x,
                    key=x,
                    value=x,
                    key_padding_mask=self_attn_padding_mask,
                    attn_mask=self_attn_mask,
                    is_only=True,
                    gate = None,
                    prefix_length = self.config.prefix_length,
                )
                x = self.dropout1(x)
                x = residual + x

                residual = x
                x = self.final_layer_norm(x)
                x = self.activation_fn(self.fc1(x))
                x = self.dropout2(x)
                x = self.fc2(x)
                x = self.dropout3(x)
                x = residual + x
                return x, attn

            elif only_video:

                #x = self.adapterfuse(x,modality_embedding[1])
                residual = x #(T,B,C)
                x = self.self_attn_layer_norm(x)
                #prefix_kv_audio = self.prefix_layer_audio(x,is_only=True)#(2,T,B,C)
                #prefix_padding = torch.zeros((self_attn_padding_mask.shape[0],self.config.prefix_length),dtype=torch.bool).to(x.device)
                x, attn = self.self_attn(
                    query=x,
                    key=x,
                    value=x,
                    key_padding_mask=self_attn_padding_mask,
                    attn_mask=self_attn_mask,
                    is_only=True,
                    gate = None,
                    prefix_length = self.config.prefix_length,
                )  
                x = self.dropout1(x)
                x = residual + x

                residual = x
                x = self.final_layer_norm(x)
                x = self.activation_fn(self.fc1(x))
                x = self.dropout2(x)
                x = self.fc2(x)
                x = self.dropout3(x)
                x = residual + x
                return x, attn

            else:
                if is_audio:
                    #x = self.adapterfuse(x,modality_embedding[2])
                    #layer_iter = 1
                    #prefix_padding = torch.zeros((self_attn_padding_mask.shape[0],self.config.prefix_length*layer_iter),dtype=torch.bool).to(x.device)
                    
                    x = self.av_adapter(x)
                    residual = x #(2T,B,C)
                    x = self.self_attn_layer_norm(x)
                    x_audio, x_video = torch.split(x,x.shape[0]//2,dim=0)     
                    x_video_s, attn = self.self_attn(
                        query=x_video,
                        key=torch.cat([x_audio,x_video],dim=0),
                        value=torch.cat([x_audio,x_video],dim=0),
                        key_padding_mask=torch.cat([self_attn_padding_mask,prefix_mask],dim=1),
                        attn_mask=self_attn_mask,
                        gate = None,
                        prefix_length = 0,
                    )              
                    x_audio_s, attn = self.self_attn(
                        query=x_audio,
                        key=torch.cat([x_audio,x_video],dim=0),
                        value=torch.cat([x_audio,x_video],dim=0),
                        key_padding_mask=torch.cat([self_attn_padding_mask,prefix_mask],dim=1),
                        attn_mask=self_attn_mask,
                        gate = None,
                        prefix_length = 0,
                    )
                                                 
                    x_cat = torch.cat([x_audio_s,x_video_s],dim=0) #(2T,B,C)              
                    x_cat = self.dropout1(x_cat)
                    x_cat = residual + x_cat

                    residual = x_cat
                    x_cat = self.final_layer_norm(x_cat)
                    x_cat = self.activation_fn(self.fc1(x_cat))
                    x_cat = self.dropout2(x_cat)
                    x_cat = self.fc2(x_cat)
                    x_cat = self.dropout3(x_cat)
                    x_cat = residual + x_cat

                    return x_cat, attn

                else:
                    pass

        else:
            pass
        

class AdaLayerToOutTransformerEncoder(nn.Module):
    def __init__(self, args):
        super().__init__()

        self.args = args
        self.dropout = args.dropout
        self.embedding_dim = args.encoder_embed_dim

        self.pos_conv = nn.Conv1d(
            self.embedding_dim,
            self.embedding_dim,
            kernel_size=args.conv_pos,
            padding=args.conv_pos // 2,
            groups=args.conv_pos_groups,
        )
        dropout = 0
        std = math.sqrt((4 * (1.0 - dropout)) / (args.conv_pos * self.embedding_dim))
        nn.init.normal_(self.pos_conv.weight, mean=0, std=std)
        nn.init.constant_(self.pos_conv.bias, 0)

        self.pos_conv = nn.utils.weight_norm(self.pos_conv, name="weight", dim=2)
        self.pos_conv = nn.Sequential(self.pos_conv, SamePad(args.conv_pos), nn.GELU())
        if self.args.use_adapterbias:
            self.adapter_alpha = nn.Linear(args.encoder_ffn_embed_dim,1)
        else:
            self.adapter_alpha = None
        self.layers = nn.ModuleList(
            [
                TransformerSentenceEncoderLayer(
                    layer=str(i),
                    config=args,
                    embedding_dim=self.embedding_dim,
                    ffn_embedding_dim=args.encoder_ffn_embed_dim,
                    num_attention_heads=args.encoder_attention_heads,
                    dropout=self.dropout,
                    attention_dropout=args.attention_dropout,
                    activation_dropout=args.activation_dropout,
                    activation_fn=args.activation_fn,
                    layer_norm_first=args.layer_norm_first,
                ) for i in range(args.encoder_layers) 
            ]
        )

        self.adapter_to_output = nn.ModuleDict(
            {layer:AdapterToOutputLayer(args, layer) for layer in list(args.adapter_to_output_layer_size.keys())}
        )

        self.num_adapter_to_output= len(args.adapter_to_output_layer_size.keys())
        self.num_adapter_layer = len(args.adapter_embedding_size.keys())

        if args.use_adapter_to_output_weighted_sum:
            if self.num_adapter_to_output:
                num_adapter_to_output_layers = self.num_adapter_to_output
            else:
                num_adapter_to_output_layers = args.encoder_layers
            self.adapter_to_output_layer_weights = nn.Parameter(torch.ones(num_adapter_to_output_layers) / num_adapter_to_output_layers)

        self.layer_norm_first = args.layer_norm_first
        self.layer_norm = LayerNorm(self.embedding_dim)
        self.layer_norm_fuse = LayerNorm(self.embedding_dim)
        self.layerdrop = args.encoder_layerdrop
        
        #self.adapter_ls = LayerScale(self.embedding_dim)
        self.register_parameter('adapter_gate', nn.Parameter(torch.tensor(1.)))

        self.apply(init_bert_params)

    def forward(self, x_audio,x_video,padding_mask_audio=None,padding_mask_video=None, only_audio=False,only_video=False,layer=None,mask_indices_audio=None,mask_indices_video=None,current_num_updates=None):
        
        x,layer_results = self.extract_features(x_audio,x_video,padding_mask_audio,padding_mask_video,only_audio,only_video,layer,mask_indices_audio,mask_indices_video,current_num_updates)

        layer_results = torch.stack(layer_results)

        if self.layer_norm_first and layer is None: #layer determine which layer output as output layer
            layer_results = self.layer_norm(layer_results)
            x = self.layer_norm(x) #(B,2T,C)
            
        L,B,T,C = layer_results.shape #(12,6,145,768)
        if self.args.use_adapter_to_output_weighted_sum:
            norm_weights = F.softmax(self.adapter_to_output_layer_weights, dim=-1) #(6,12)
            layer_results = layer_results.reshape(L,-1)
            layer_results = torch.matmul(norm_weights,layer_results).reshape(B,T,C) #(6,145,768)

        else:
            layer_results = layer_results[-1]
            
        
        return x , layer_results #(6,145,768)

    def extract_features(self, x_audio,x_video,padding_mask_audio=None,padding_mask_video=None,only_audio=False,only_video=False,tgt_layer=None,mask_indices_audio=None,mask_indices_video=None,current_num_updates=None):
        if padding_mask_audio is not None:
            x_audio = index_put(x_audio, padding_mask_audio, 0)
            x_video = index_put(x_video, padding_mask_video, 0)
        x_conv_audio = self.pos_conv(x_audio.transpose(1, 2))
        x_conv_video = self.pos_conv(x_video.transpose(1, 2))
        x_conv_audio = x_conv_audio.transpose(1, 2)
        x_conv_video = x_conv_video.transpose(1, 2)
        x_audio = x_audio + x_conv_audio
        x_video = x_video + x_conv_video       
        if not self.layer_norm_first:
            x_audio = self.layer_norm(x_audio) #encoder.layer_norm
            x_video = self.layer_norm(x_video)
        x_audio = F.dropout(x_audio, p=self.dropout, training=self.training)
        x_video = F.dropout(x_video, p=self.dropout, training=self.training)
        # B x T x C -> T x B x C
        x_audio = x_audio.transpose(0, 1)
        x_video = x_video.transpose(0, 1)

        ################################################################################################################
        T,B,C = x_audio.shape
        mask_indices_fuse = None
        # if self.training:
        #     max_update=30000 #18000/2
        #     #print("current_num_updates",current_num_updates)
        #     if current_num_updates < 30000: #the first 9000 has mask
        #         mask_rate = current_num_updates/max_update
        #         mask_indices_fuse, starts, ends, batch_indexes = compute_mask_indices(
        #             (B, T),
        #             padding_mask_video,
        #             mask_rate,
        #             6,
        #             self.args.mask_selection,
        #             self.args.mask_other,
        #             min_masks=2,
        #             no_overlap=self.args.no_mask_overlap,
        #             min_space=1,
        #         ) 
        #         mask_indices_fuse = mask_indices_fuse.astype(int)
        #         mask_indices_fuse = torch.from_numpy(mask_indices_fuse).unsqueeze(2).expand(B,T,C).transpose(0,1) #(T,B,C)
        ##################################################################################################################

        layer_results = []
        r = None
        layer_ada_keys = list(self.args.adapter_to_output_layer_size.keys())
        for i, layer in enumerate(self.layers):
            dropout_probability = np.random.random()
            if not self.training or (dropout_probability > self.layerdrop):
                if only_audio:
                    x_audio, z_audio = layer(x_audio, x_video,self_attn_padding_mask=padding_mask_audio,prefix_mask=padding_mask_video,need_weights=False,is_audio=True,only_audio=only_audio,only_video=only_video) 
                    av_fuse = x_audio #(T,B,2C)
                    if str(i) in layer_ada_keys:
                        layer_out = self.adapter_to_output[str(i)](av_fuse).transpose(0,1)
                    else:
                        layer_out = av_fuse.transpose(0,1)
                elif only_video:
                    x_video, z_video = layer(x_video, x_audio,self_attn_padding_mask=padding_mask_video, prefix_mask=padding_mask_audio,need_weights=False,is_audio=False,only_audio=only_audio,only_video=only_video) 
                    av_fuse = x_video
                    if str(i) in layer_ada_keys:
                        layer_out = self.adapter_to_output[str(i)](av_fuse).transpose(0,1)
                    else:
                        layer_out = av_fuse.transpose(0,1)
                else:
                    if i==0:
                        x_fuse = torch.cat([x_audio,x_video],dim=0)
                    x_fuse, z_audio = layer(x_fuse,x_video,self_attn_padding_mask=padding_mask_audio, prefix_mask=padding_mask_video,need_weights=False,is_audio=True,only_audio=only_audio,only_video=only_video,mask_indices_audio=mask_indices_audio,mask_indices_video=mask_indices_video)
                    L,_,_ = x_fuse.shape
                    a_fuse,v_fuse = torch.split(x_fuse,L//2,dim=0) #(T,B,C)
                    #pdb.set_trace()
                    if mask_indices_fuse != None:
                        mask_indices_fuse = mask_indices_fuse.to(v_fuse.device)
                        av_fuse = a_fuse + mask_indices_fuse * v_fuse #(2T,B,C)
                    else:
                        av_fuse = a_fuse + self.adapter_gate * v_fuse

                    if str(i) in layer_ada_keys:
                        layer_out = self.adapter_to_output[str(i)](av_fuse).transpose(0,1)
                    else:
                        layer_out = av_fuse.transpose(0,1)

                layer_results.append(layer_out) #(n_layer,B,T,C)
        x = x_fuse.transpose(0, 1) #(B,2T,C)
        #############################################################################
        return x, layer_results

    def max_positions(self):
        """Maximum output length supported by the encoder."""
        return self.args.max_positions

    def upgrade_state_dict_named(self, state_dict, name):
        """Upgrade a (possibly old) state dict for new versions of fairseq."""
        return state_dict

class SubModel(nn.Module):
    def __init__(self, resnet=None, input_dim=None, cfg=None):
        super().__init__()
        self.resnet = resnet
        self.proj = nn.Linear(input_dim, cfg.encoder_embed_dim)
        self.encoder = TransformerEncoder(cfg) if cfg.encoder_layers > 0 else None

    def forward(self, x):
        if self.resnet is not None:
            x = self.resnet(x)
        x = self.proj(x.transpose(1, 2))
        if self.encoder is not None:
            x = self.encoder(x)[0].transpose(1, 2)
        else:
            x = x.transpose(1, 2)
        return x

@register_model("ada_av_hubert", dataclass=AdaAVHubertConfig)
class AdaAVHubertModel(BaseFairseqModel):
    def __init__(
        self,
        cfg: AdaAVHubertConfig,
        task_cfg: AVHubertPretrainingConfig,
        dictionaries: List[Dictionary],
        **kwargs
    ) -> None:
        super().__init__()
        #logger.info(f"AdaAVHubertModel Config: {cfg}")

        feature_ds_rate = 1
        self.feat2tar_ratio = cfg.label_rate * feature_ds_rate / task_cfg.sample_rate
        sub_cfg = deepcopy(cfg)
        sub_cfg.encoder_layers = sub_cfg.sub_encoder_layers
        resnet = ResEncoder(relu_type=cfg.resnet_relu_type, weights=cfg.resnet_weights)
        self.feature_extractor_audio = SubModel(resnet=None, input_dim=cfg.audio_feat_dim, cfg=sub_cfg)
        self.feature_extractor_video = SubModel(resnet=resnet, input_dim=resnet.backend_out, cfg=sub_cfg)
        self.modality_dropout, self.audio_dropout = cfg.modality_dropout, cfg.audio_dropout
        self.modality_fuse = cfg.modality_fuse
        self.encoder_embed_dim = cfg.encoder_embed_dim

        if self.modality_fuse == 'concat':
            self.embed = cfg.encoder_embed_dim * 2
        elif self.modality_fuse == 'add':
            self.embed = cfg.encoder_embed_dim
        elif self.modality_fuse == 'fuse':
            self.embed = cfg.encoder_embed_dim * 2
        self.post_extract_proj = (
            nn.Linear(self.embed, cfg.encoder_embed_dim)
            if self.embed != cfg.encoder_embed_dim
            else None
        )

        self.mask_prob_image, self.mask_prob_audio = cfg.mask_prob_image, cfg.mask_prob_audio
        self.mask_selection = cfg.mask_selection
        self.mask_other = cfg.mask_other
        self.mask_length_image, self.mask_length_audio = cfg.mask_length_image, cfg.mask_length_audio
        self.no_mask_overlap = cfg.no_mask_overlap
        self.mask_min_space = cfg.mask_min_space

        self.mask_channel_prob = cfg.mask_channel_prob
        self.mask_channel_selection = cfg.mask_channel_selection
        self.mask_channel_other = cfg.mask_channel_other
        self.mask_channel_length = cfg.mask_channel_length
        self.no_mask_channel_overlap = cfg.no_mask_channel_overlap
        self.mask_channel_min_space = cfg.mask_channel_min_space

        self.dropout_input = nn.Dropout(cfg.dropout_input)
        self.dropout_features = nn.Dropout(cfg.dropout_features)

        self.feature_grad_mult = cfg.feature_grad_mult
        self.logit_temp = cfg.logit_temp
        self.skip_masked = cfg.skip_masked
        self.skip_nomask = cfg.skip_nomask
        self.sim_type = cfg.sim_type
        self.selection_type = cfg.selection_type
        self.masking_type = cfg.masking_type

        final_dim = (
            cfg.final_dim if cfg.final_dim > 0 else cfg.encoder_embed_dim
        )

        self.mask_emb = nn.Parameter(
            torch.FloatTensor(cfg.audio_feat_dim).uniform_() if self.masking_type == 'input' else torch.FloatTensor(cfg.encoder_embed_dim).uniform_()
        )

        self.encoder = AdaLayerToOutTransformerEncoder(cfg)
        ####################### adapter_mask to place audio zero and video zero #####################################
        # self.adapter_mask_audio = nn.Parameter(torch.zeros(1,1,cfg.encoder_embed_dim))
        # self.adapter_mask_video = nn.Parameter(torch.zeros(1,1,cfg.encoder_embed_dim))
        # nn.init.normal_(self.adapter_mask_audio,std=.02)
        # nn.init.normal_(self.adapter_mask_video,std=.02)
        ###############################################################################################################
        self.layer_norm = LayerNorm(self.embed) #feature.layer_norm

        self.target_glu = None
        if cfg.target_glu:
            self.target_glu = nn.Sequential(
                nn.Linear(final_dim, final_dim * 2), nn.GLU()
            )

        self.untie_final_proj = cfg.untie_final_proj
        if self.untie_final_proj:
            self.final_proj = nn.Linear(
                cfg.encoder_embed_dim, final_dim * len(dictionaries)
            )
        else:
            self.final_proj = nn.Linear(cfg.encoder_embed_dim, final_dim)

        # modules below are not needed during fine-tuning
        if any([d is None for d in dictionaries]):
            logger.info(
                "cannot find dictionary. assume will be used for fine-tuning"
            )
        else:
            self.num_classes = [len(d) for d in dictionaries]
            self.label_embs_concat = nn.Parameter(
                torch.FloatTensor(sum(self.num_classes), final_dim)
            )
            nn.init.uniform_(self.label_embs_concat)

    def upgrade_state_dict_named(self, state_dict, name):
        """Upgrade a (possibly old) state dict for new versions of fairseq."""

        super().upgrade_state_dict_named(state_dict, name)
        return state_dict

    @classmethod
    def build_model(cls, cfg: AdaAVHubertConfig, task: AVHubertPretrainingTask):
        """Build a new model instance."""

        kwargs = {}
        model = AdaAVHubertModel(cfg, task.cfg, task.dictionaries, **kwargs)
        return model

    def apply_input_mask(self, x, padding_mask, target_list):
        B, C, T = x.shape[:3]
        is_audio = True if len(x.shape) == 3 else False
        if is_audio:
            mask_prob, mask_length = self.mask_prob_audio, self.mask_length_audio
        else:
            mask_prob, mask_length = self.mask_prob_image, self.mask_length_image
        if mask_prob > 0:

            mask_indices, starts, ends, batch_indexes = compute_mask_indices(
                (B, T),
                padding_mask,
                mask_prob,
                mask_length,
                self.mask_selection,
                self.mask_other,
                min_masks=2,
                no_overlap=self.no_mask_overlap,
                min_space=self.mask_min_space,
            )
            mask_indices_np = mask_indices
            mask_indices = torch.from_numpy(mask_indices).to(x.device)
            x = x.transpose(1, 2).contiguous() # [B, T, C, H, W]
            if B == 1:
                x[mask_indices] = 0
            elif is_audio:
                x[mask_indices] = self.mask_emb
            elif self.selection_type == 'same_other_seq':
                perm = (torch.arange(B) + torch.randint(low=1, high=B, size=(1,))) % B
                x_perm = x[perm]
                x[mask_indices] = x_perm[mask_indices]
            elif self.selection_type == 'same_seq':
                batch_indexes_, other_indexes = [], []
                for batch_index, start, end in zip(batch_indexes, starts, ends):
                    length = end-start
                    other_start = np.setdiff1d(np.arange(T), np.arange(max(0, start-length), end))
                    if len(other_start) > 0:
                        other_start = np.random.choice(other_start, size=1)
                    else:
                        other_start = 0
                    other_end = other_start + length
                    other_indexes.append(np.arange(other_start, other_end).clip(max=T-1))
                    batch_indexes_.append(np.zeros([length], dtype=np.int64)+batch_index)
                batch_indexes, other_indexes = np.concatenate(batch_indexes_), np.concatenate(other_indexes)
                x[mask_indices] = x[batch_indexes, other_indexes]

            x = x.transpose(1, 2).contiguous()
        else:
            mask_indices = None

        if self.mask_channel_prob > 0:
            logger.info(f"No mask channel prob for input masking")
        return x, mask_indices

    def apply_feature_mask(self, x, padding_mask, target_list):
        B, T, C = x.shape
        assert self.mask_prob_audio == self.mask_prob_image and self.mask_length_audio == self.mask_length_image, f"masking prob/length for image/audio be same for feature masking"
        mask_prob, mask_length = self.mask_prob_audio, self.mask_length_image
        if mask_prob > 0:
            mask_indices, _, _, _ = compute_mask_indices(
                (B, T),
                padding_mask,
                mask_prob,
                mask_length,
                self.mask_selection,
                self.mask_other,
                min_masks=2,
                no_overlap=self.no_mask_overlap,
                min_space=self.mask_min_space,
            )
            mask_indices = torch.from_numpy(mask_indices).to(x.device)
            x[mask_indices] = self.mask_emb
        else:
            mask_indices = None

        if self.mask_channel_prob > 0:
            mask_channel_indices, _, _, _ = compute_mask_indices(
                (B, C),
                None,
                self.mask_channel_prob,
                self.mask_channel_length,
                self.mask_channel_selection,
                self.mask_channel_other,
                no_overlap=self.no_mask_channel_overlap,
                min_space=self.mask_channel_min_space,
            )
            mask_channel_indices = (
                torch.from_numpy(mask_channel_indices)
                .to(x.device)
                .unsqueeze(1)
                .expand(-1, T, -1)
            )
            x[mask_channel_indices] = 0

        return x, mask_indices

    def forward_features(self, source: torch.Tensor, modality: str) -> torch.Tensor:
        extractor = eval(f"self.feature_extractor_{modality}")
        if self.feature_grad_mult > 0:
            features = extractor(source)
            if self.feature_grad_mult != 1.0:
                features = GradMultiply.apply(features, self.feature_grad_mult)
        else:
            with torch.no_grad():
                features = extractor(source)
        return features

    def forward_targets(
            self, features: torch.Tensor, mask_indices: torch.Tensor, target_list: List[torch.Tensor],
    ) -> Tuple[torch.Tensor, torch.Tensor]:
        # Trim features to ensure labels exist and then get aligned labels
        feat_tsz = features.size(2)
        targ_tsz = min([t.size(1) for t in target_list])
        if self.feat2tar_ratio * feat_tsz > targ_tsz:
            feat_tsz = int(targ_tsz / self.feat2tar_ratio)
            features = features[..., :feat_tsz]
            if mask_indices is not None:
                mask_indices = mask_indices[..., :feat_tsz]
        target_inds = torch.arange(feat_tsz).float() * self.feat2tar_ratio
        target_list = [t[:, target_inds.long()] for t in target_list]
        return features, mask_indices, target_list

    def forward_padding_mask(
        self, features: torch.Tensor, padding_mask: torch.Tensor,
    ) -> torch.Tensor:
        extra = padding_mask.size(1) % features.size(1)
        if extra > 0:
            padding_mask = padding_mask[:, :-extra]
        padding_mask = padding_mask.view(
            padding_mask.size(0), features.size(1), -1
        )
        padding_mask = padding_mask.all(-1)
        return padding_mask

    def compute_logits(self, feats, emb_mat):
        # feats: [B, T, F], emb_mat: [V, F]
        if self.sim_type == 'dot':
            logits = torch.matmul(feats, emb_mat.transpose(0, 1))
        elif self.sim_type == 'cosine':
            batch_size, timesteps, emb_dim = feats.size()
            feats_ = feats.view(-1, emb_dim)
            nom = (feats_.unsqueeze(dim=1) * emb_mat.unsqueeze(dim=0)).sum(dim=-1) # [B*T, V]
            denom = (feats_**2).sum(dim=-1).sqrt().unsqueeze(dim=1) * (emb_mat**2).sum(dim=-1).sqrt().unsqueeze(dim=0) # [B*T, V]
            logits = (nom/denom.clamp(min=1e-6)).view(batch_size, timesteps, -1)
        else:
            raise NotImplementedError
        logits = logits / self.logit_temp
        return logits

    def forward(
        self,
        source: torch.Tensor,
        target_list: Optional[List[torch.Tensor]] = None,
        padding_mask: Optional[torch.Tensor] = None,
        mask: bool = True,
        features_only: bool = False,
        output_layer: Optional[int] = None
    ) -> Dict[str, torch.Tensor]:
        """output layer is 1-based"""
        src_audio, src_video = source['audio'], source['video']
        if mask and self.masking_type == 'input':
            src_video, mask_indices_video = self.apply_input_mask(src_video, padding_mask, target_list)
            src_audio, mask_indices_audio = self.apply_input_mask(src_audio, padding_mask, target_list)
            mask_indices = torch.logical_or(mask_indices_audio, mask_indices_video)
        else:
            src_audio, src_video, mask_indices = src_audio, src_video, None

        features_audio = self.forward_features(src_audio, modality='audio') # features: [B, F, T]
        features_video = self.forward_features(src_video, modality='video')
        modality_drop_prob, audio_drop_prob = np.random.random(), np.random.random()
        if self.training:
            if modality_drop_prob < self.modality_dropout:
                if audio_drop_prob < self.audio_dropout:
                    features_audio = 0 * features_audio
                else:
                    features_video = 0 * features_video

        if self.modality_fuse == 'concat' or self.modality_fuse == 'add':
            if self.modality_fuse == 'concat':
                features = torch.cat([features_audio, features_video], dim=1)
            elif self.modality_fuse == 'add':
                features = features_audio + features_video

            if target_list is not None:
                features, mask_indices, target_list = self.forward_targets(features, mask_indices, target_list)
            features_pen = features.float().pow(2).mean()
            features = features.transpose(1, 2)
            features = self.layer_norm(features)
            if padding_mask is not None:
                padding_mask = self.forward_padding_mask(features, padding_mask)
            if self.post_extract_proj is not None:
                features = self.post_extract_proj(features)
            features = self.dropout_input(features)
            if self.masking_type == 'feature' and mask:
                x, mask_indices = self.apply_feature_mask(features, padding_mask, target_list)
            else:
                x = features

        elif self.modality_fuse == 'fuse':
            ########################################rewrite modality_fuse####################################################
            #features_audio = torch.cat([features_audio, 0*features_video], dim=1) ???
            #features_video = torch.cat([0*features_audio, features_video], dim=1) ???
            if target_list is not None:
                features_audio, mask_indices, target_list_audio = self.forward_targets(features_audio, mask_indices, target_list)
                features_video, mask_indices, target_list_video = self.forward_targets(features_video, mask_indices, target_list)
            features_pen_audio = features_audio.float().pow(2).mean()
            features_pen_video = features_video.float().pow(2).mean()            
            features_audio = features_audio.transpose(1, 2)
            features_video = features_video.transpose(1, 2)

            features_audio = self.layer_norm(features_audio)
            features_video = self.layer_norm(features_video)            

            if padding_mask is not None:
                padding_mask_audio = self.forward_padding_mask(features_audio, padding_mask)
                padding_mask_video = self.forward_padding_mask(features_video, padding_mask)                
            if self.post_extract_proj is not None:
                features_audio = self.post_extract_proj(features_audio)
                features_video = self.post_extract_proj(features_video)
            features_audio = self.dropout_input(features_audio)
            features_video = self.dropout_input(features_video)            
            if self.masking_type == 'feature' and mask:
                x_audio, mask_indices_audio = self.apply_feature_mask(features_audio, padding_mask_audio, target_list_audio)
                x_video, mask_indices_video = self.apply_feature_mask(features_video, padding_mask_video, target_list_video)
            else:
                x = features
        #####################################################################################################################
        # feature: (B, T, D), float
        # target: (B, T), long
        # x: (B, T, D), float
        # padding_mask: (B, T), bool
        # mask_indices: (B, T), bool
        x, _ = self.encoder(
            x,
            padding_mask=padding_mask,
            layer=None if output_layer is None else output_layer - 1
        )

        if features_only:
            return {"x": x, "padding_mask": padding_mask, "features": features}

        label_embs_list = self.label_embs_concat.split(self.num_classes, 0)
        proj_x = self.final_proj(x)
        if self.untie_final_proj:
            proj_x_list = proj_x.chunk(len(self.num_classes), dim=-1)
        else:
            proj_x_list = [proj_x for _ in self.num_classes]
        logit_list = [self.compute_logits(proj, emb).view(-1, num_class) for proj, emb, num_class in zip(proj_x_list, label_embs_list, self.num_classes)] # [[B*T, V]]
        mask, unmask = torch.logical_and(mask_indices, ~padding_mask).view(-1), torch.logical_and(~mask_indices, ~padding_mask).view(-1) # [B*T]
        logit_m_list, logit_u_list = [logit[mask] for logit in logit_list], [logit[unmask] for logit in logit_list]
        target_m_list, target_u_list = [target.view(-1)[mask].long() for target in target_list], [target.view(-1)[unmask].long() for target in target_list]
        result = {
            "logit_m_list": logit_m_list,
            "logit_u_list": logit_u_list,
            "target_m_list": target_m_list,
            "target_u_list": target_u_list,
            "padding_mask": padding_mask,
            "features_pen": features_pen,
        }
        return result

    def extract_features(
        self,
        source: torch.Tensor,
        padding_mask: Optional[torch.Tensor] = None,
        mask: bool = False,
        ret_conv: bool = False,
        output_layer: Optional[int] = None,
    ) -> Tuple[torch.Tensor, torch.Tensor]:
        res = self.forward(
            source,
            padding_mask=padding_mask,
            mask=mask,
            features_only=True,
            output_layer=output_layer,
        )
        feature = res["features"] if ret_conv else res["x"]
        return feature, res["padding_mask"]

    def extract_finetune(self, source, padding_mask=None, current_num_updates=None, mask=False, ret_conv=False, output_layer=None):
        src_audio, src_video = source['audio'], source['video']
        if mask and self.masking_type == 'input':
            src_video, mask_indices_video = self.apply_input_mask(src_video, padding_mask, target_list=None)
            src_audio, mask_indices_audio = self.apply_input_mask(src_audio, padding_mask, target_list=None)
            mask_indices = torch.logical_or(mask_indices_audio, mask_indices_video) # mask_indices not used in fine-tuning
        else:
            src_audio, src_video, mask_indices = src_audio, src_video, None

        if src_audio is not None and src_video is None:
            features_audio = self.forward_features(src_audio, modality='audio') # features: [B, F, T]
            features_video = features_audio.new_zeros(features_audio.size(0), self.encoder_embed_dim, features_audio.size(-1))
        elif src_audio is None and src_video is not None:
            features_video = self.forward_features(src_video, modality='video')
            features_audio = features_video.new_zeros(features_video.size(0), self.encoder_embed_dim, features_video.size(-1))
        elif src_audio is not None and src_video is not None:
            features_video = self.forward_features(src_video, modality='video')
            features_audio = self.forward_features(src_audio, modality='audio') # features: [B, F, T]
            
        ######################################################################################################################
        only_audio,only_video = False,False
        modality_drop_prob, audio_drop_prob = np.random.random(), np.random.random()
        if self.training: #it is used to find it the process in under train state or eval state
            if modality_drop_prob < self.modality_dropout:
                if audio_drop_prob > self.audio_dropout:
                    #features_audio = 0 * features_audio
                    only_video = True
                else:
                    #features_video = 0 * features_video
                    only_audio = True
        else:
            #When infer self.training=False
            if src_audio is not None and src_video is None:
                only_audio = True
            if src_audio is None and src_video is not None:
                only_video = True   

        if self.modality_fuse == 'concat' or self.modality_fuse == 'add':
            if self.modality_fuse == 'concat':
                features = torch.cat([features_audio, features_video], dim=1)
            else:
                features = features_audio + features_video
            features_pen = features.float().pow(2).mean()
            features = features.transpose(1, 2)
            features = self.layer_norm(features)
            unmasked_features = features.clone()
            if padding_mask is not None:
                padding_mask = self.forward_padding_mask(features, padding_mask)

            if self.post_extract_proj is not None:
                features = self.post_extract_proj(features)

            features = self.dropout_input(features)
            unmasked_features = self.dropout_features(unmasked_features) #no need when finetune，no need to set dropout_feature
            x = features 
            mask_indices = None
            # feature: (B, T, D), float
            # target: (B, T), long
            # x: (B, T, D), float
            # padding_mask: (B, T), bool
            # mask_indices: (B, T), bool
            x, layer_results = self.encoder(
                x,
                padding_mask=padding_mask,
                layer=None if output_layer is None else output_layer - 1
            )

        elif self.modality_fuse == 'fuse':
            tmp=features_audio #(B,C,T)
            features_audio = torch.cat([features_audio, 0*features_video], dim=1)
            features_video = torch.cat([0*tmp, features_video], dim=1)           
            features_audio = features_audio.transpose(1, 2) #(B,T,C)
            features_video = features_video.transpose(1, 2)            
            features_audio = self.layer_norm(features_audio)
            features_video = self.layer_norm(features_video)

            if padding_mask is not None:
                padding_mask_audio = self.forward_padding_mask(features_audio, padding_mask)
                padding_mask_video = self.forward_padding_mask(features_video, padding_mask)                
            if self.post_extract_proj is not None:
                features_audio = self.post_extract_proj(features_audio)
                features_video = self.post_extract_proj(features_video)

            features_audio = self.dropout_input(features_audio)
            features_video = self.dropout_input(features_video)            

            mask_indices_audio,mask_indices_video = None,None
            #########################Add sparse drop for both audio and video################################
            #src_audio, mask_indices_audio = self.apply_input_mask(src_audio, padding_mask, target_list=None)           
            if self.training:
                B,T,C = features_audio.shape
                mask_indices_audio, starts, ends, batch_indexes = compute_mask_indices(
                    (B, T),
                    padding_mask,
                    self.mask_prob_audio,
                    self.mask_length_audio,
                    self.mask_selection,
                    self.mask_other,
                    min_masks=2,
                    no_overlap=self.no_mask_overlap,
                    min_space=self.mask_min_space,
                )
                mask_indices_video, starts, ends, batch_indexes = compute_mask_indices(
                    (B, T),
                    padding_mask,
                    self.mask_prob_image,
                    self.mask_length_image,
                    self.mask_selection,
                    self.mask_other,
                    min_masks=2,
                    no_overlap=self.no_mask_overlap,
                    min_space=self.mask_min_space,
                )
                mask_indices_audio = torch.from_numpy(mask_indices_audio).unsqueeze(2).expand(B,T,C)
                mask_indices_video = torch.from_numpy(mask_indices_video).unsqueeze(2).expand(B,T,C)
                
                mask_features_audio = features_audio.detach().clone()
                mask_features_video = features_video.detach().clone()    

                mask_features_audio[mask_indices_audio] = 0
                mask_features_video[mask_indices_video] = 0 
            #################################################################################################
            with torch.no_grad():
                x, layer_results= self.encoder(
                    features_audio,
                    features_video,
                    padding_mask_audio,
                    padding_mask_video,
                    only_audio,
                    only_video,
                    layer=None if output_layer is None else output_layer - 1,
                    mask_indices_audio=None,
                    mask_indices_video=None,
                    current_num_updates=current_num_updates,
                )
            x_mask=None
            if self.training:
                x_mask, layer_results= self.encoder(
                    mask_features_audio,
                    mask_features_video,
                    padding_mask_audio,
                    padding_mask_video,
                    only_audio,
                    only_video,
                    layer=None if output_layer is None else output_layer - 1,
                    mask_indices_audio=mask_indices_audio,
                    mask_indices_video=mask_indices_video,
                    current_num_updates=current_num_updates,
                )
            #if only_audio or only_video:
            return x_mask,x,layer_results, padding_mask_audio,mask_indices_audio,mask_indices_video
            #else:
            #return x, layer_results, torch.cat([padding_mask_audio,padding_mask_video],dim=1),pred_audio_list,pred_video_list,original_audio_list,original_video_list
        


    def get_extra_losses(self, net_output):
        extra_losses = []
        names = []
        if "features_pen" in net_output:
            extra_losses.append(net_output["features_pen"])
            names.append("features_pen")

        return extra_losses, names

    def remove_pretraining_modules(self):
        self.target_glu = None
        self.final_proj = None

    def get_logits(self, net_output, is_masked=True):
        raise NotImplementedError

    def get_targets(self, net_output, is_masked=True):
        raise NotImplementedError

    def compute_nce(self, x, pos, negs):
        neg_is_pos = (pos == negs).all(-1)
        pos = pos.unsqueeze(0)
        targets = torch.cat([pos, negs], dim=0)

        logits = torch.cosine_similarity(
            x.float(), targets.float(), dim=-1
        ).type_as(x)
        logits /= self.logit_temp
        if neg_is_pos.any():
            logits[1:][neg_is_pos] = float("-inf")
        logits = logits.transpose(0, 1)  # (num_x, num_cls+1)
        return logits

if __name__ == "__main__":
    conf = AdaAVHubertConfig
    conf.adapter_embedding_size = {str(i):conf.eada_emb_size for i in range(0,12)}
    conf.adapter_to_output_layer_size = {str(i):256 for i in range(0,12)}
    #ggg = AdapterLayer(conf,'1')
    yyy = AdaLayerToOutTransformerEncoder(conf)
    pdb.set_trace()
    layer_names = ['layers.'+k for k in conf.adapter_to_output_layer_size.keys()]