import jax
import jax.numpy as jnp


# Mixer matrix for quadrotor in X configuration in NED body frame
quad_A_Xconf = jnp.array(
    [
        [-0.707, 0.707, 0.707, -0.707],
        [-0.707, 0.707, -0.707, 0.707],
        [-1.0, -1.0, 1.0, 1.0],
        [1.0, 1.0, 1.0, 1.0]
    ]
)

# Mixer matrix for hexacopter in X configuration in NED body frame
hexa_A_Xconf = jnp.array(
    [
        [-1.0,  1.0,    0.5,    -0.5,   -0.5,   0.5],
        [   0,  0,      -0.866, 0.866,  -0.866, 0.866],
        [1.0,  -1.0,     1.0,    -1.0,    -1.0,   1.0],
        [1.0,  1.0,     1.0,    1.0,     1.0,    1.0]
    ]
)


def motor_model(get_params_fn, u, order=1, use_constant_term=False, assume_sym=True, learn_mixer=False):
    """
    Compute the thrust and moment generated by the motors.
    order = 1: linear model
    order = 2: quadratic model
    order = 3: cubic model
    same_model: if True, the same model is used for all the motors
    """
    # Get the u vector depending on the order of the model
    if order == 1:
        uvect = jnp.stack((u, jnp.ones_like(u)), axis=1) if use_constant_term else u.reshape(-1, 1)
    elif order == 2:
        uvect = jnp.stack((u**2, u, jnp.ones_like(u)), axis=1) if use_constant_term else jnp.stack((u**2, u), axis=1)
    elif order == 3:
        uvect = jnp.stack((u**3, u**2, u, jnp.ones_like(u)), axis=1) if use_constant_term else jnp.stack((u**3, u**2, u), axis=1)
    else:
        raise ValueError('Invalid order')
    # The parameters 
    motParams = jnp.array([get_params_fn('coeffMot{}'.format(i)) for i in range(uvect.shape[-1])])

    # Compute the trust for each motor
    # print('uvect: {}'.format(uvect.shape))
    # print('motParams: {}'.format(motParams.shape))
    thurstPerMotor = jnp.dot(uvect, motParams)
    # print('Thrust per motor: {}'.format(thurstPerMotor.shape))

    # If the mixer is learned, use the learned values
    if learn_mixer:
        mixer = jnp.array([get_params_fn('mix{}'.format(i)) for i in range( 2 * u.shape[0])])
        mixer = mixer.reshape((2, u.shape[0]))
        mixer = jnp.vstack([mixer, jnp.array([[1.0,-1.0,1.0,-1.0,-1.0,1.0],[1.0, 1.0, 1.0, 1.0, 1.0, 1.0]])])
        mixerMz = get_params_fn('CoeffDrag')
        mixer = mixer.at[2, :].multiply(mixerMz)
        thurstAndMoment = mixer @ thurstPerMotor
        return thurstAndMoment[-1], thurstAndMoment[:-1]

    mixerMx = get_params_fn('Lmx')
    mixerMy = mixerMx if assume_sym else get_params_fn('Lmy')
    mixerMz = get_params_fn('CoeffDrag')

    # Pick the mixer matrix depending on the number of motors
    if thurstPerMotor.shape[0] == 4:
        mixer = quad_A_Xconf
    elif thurstPerMotor.shape[0] == 6:
        mixer = hexa_A_Xconf
    else:
        raise ValueError('Invalid number of motors')
    
    # Modify the mixer
    mixer = mixer.at[0, :].multiply(mixerMx)
    mixer = mixer.at[1, :].multiply(mixerMy)
    mixer = mixer.at[2, :].multiply(mixerMz)

    # Compute the thrust and moment
    thurstAndMoment = mixer @ thurstPerMotor

    return thurstAndMoment[-1], thurstAndMoment[:-1]