import numpy as np, pandas as pd
import copy, os, sys
from scipy.stats import spearmanr
import matplotlib.pyplot as plt
from sklearn.linear_model import LogisticRegression
from pyhessian import hessian # Hessian computation

def sigmoid(x):
  return 1/(1+np.exp(-x))

def compute_hessian_lr(f, data):
    """
    Compute the Hessian matrix for logistic loss w.r.t model parameter f
    inputs: f is logistic regression parameters, (d+1)*1 size where d is the number of input features
            data is input data of size n*(d+1), additional column is for intercept/bias term
  
    return the Hessian matrix of size (d+1)*(d+1)
    Formula of Hessian matrix is from   https://stats.stackexchange.com/questions/68391/hessian-of-logistic-function
    """
    y_pred = data @ f
    z_pred = sigmoid(y_pred)
    l = len(f)
    H = np.zeros((l, l))
    L = len(y_pred)
    for i in range(L):
        temp_x = data[i, :].reshape(-1, 1)
        H += np.dot(temp_x, temp_x.T) * z_pred[i] * (1 - z_pred[i]) / L

    return H


def logistic_loss(x, data, labels):
    """
    Compute the Logistic loss over pair of data and labels
    data size nxd
    labels size nx1
    """
    L = len(labels)
    y_pred = data @ x
    z_pred = sigmoid(y_pred)
    z_pred[z_pred >= 1 - 1e-4] = 1 - 1e-4
    z_pred[z_pred <= 1e-4] = 1e-4
    f = -labels * np.log(z_pred) + (1 - labels) * (1 - np.log(z_pred))

    return f