"""
Duplicate all clients's local training data
"""

from datasets import concatenate_datasets, load_dataset, DatasetDict, Dataset
from accelerate import Accelerator
from rank_bm25 import BM25Okapi
import numpy as np
from tqdm import trange
from nltk.tokenize import word_tokenize

from openicl.icl_retriever import BaseRetriever, BM25Retriever
from openicl.utils.check_type import _check_str
from openicl import DatasetReader, PromptTemplate, PPLInferencer, AccEvaluator

import argparse
import json
import os
import wandb
import socket
from copy import deepcopy

from util.template import gen_template_classification
from util.retriever import get_retriever, get_fedretriever
from util.partition import (
    cls_iid_partition,
    cls_noniid_partition,
    data2usename,
    data2numcls,
)
from util.icl_bm25_fedretriever import BM25FedRetriever
from util.icl_topk_fedretriever import TopkFedRetriever
from util.icl_ppl_fed_inferencer import PPLFedInferencer
from util.misc import setup_seed


def print_score(args):
    # load
    prediction_file = os.path.join(
        args.log_dir, f"prediction_{'debug' if args.debug else 'run'}.json"
    )
    label_file = os.path.join(
        args.log_dir, f"label_{'debug' if args.debug else 'run'}.json"
    )
    with open(prediction_file, "r") as f:
        prediction = data_loaded = json.load(f)
    with open(label_file, "r") as f:
        reference = data_loaded = json.load(f)

    evaluator = AccEvaluator()
    score = evaluator.score(predictions=prediction, references=reference)
    print(score)
    return score


def run(args):
    # ===== 1. Data Setting Preparation =====
    # load dataset
    dataset = load_dataset(data2usename[args.dataset])

    if "test" not in dataset:
        test_split = "validation"
    else:
        test_split = "test"

    # partition data into clients
    if args.partition == "iid":
        fed_dataset = cls_iid_partition(
            dataset=dataset,
            split="train",
            data_name=args.dataset,
            num_clients=args.num_clients,
            test_split=test_split,
            subset_num=args.subset_num,
        )
    elif args.partition == "noniid":
        fed_dataset = cls_noniid_partition(
            dataset=dataset,
            split="train",
            data_name=args.dataset,
            major_classes_num=args.major_classes_num,
            num_clients=args.num_clients,
            test_split=test_split,
            subset_num=args.subset_num,
        )
    else:
        raise ValueError(
            f"args.partition can only be 'iid' or 'noniid', rather than '{args.partition}'."
        )

    # try to duplicate clients
    for dup in range(args.duplicate - 1):
        for cid in range(args.num_clients):
            new_cid = (dup + 1) * args.num_clients + cid
            data_cp = deepcopy(fed_dataset[f"train-client{cid}"])
            fed_dataset[f"train-client{new_cid}"] = data_cp
    print(f"Clients duplication {args.duplicate} times done.")

    # save selected test query subset
    subset_orig_idxs = fed_dataset[test_split]["idx"]
    subset_orig_idxs_file = os.path.join(args.log_dir, "query_subset_orig_idxs.json")
    with open(subset_orig_idxs_file, "w") as f:
        json.dump(subset_orig_idxs, f)

    # make datareader for all clients' local training data
    fed_reader = []
    for cid in range(args.num_clients * args.duplicate):
        fed_reader.append(
            DatasetReader(
                fed_dataset[f"train-client{cid}"],
                input_columns=["sentence"],
                output_column="label",
            )
        )

    # get template for classification
    tp_dict = gen_template_classification(args)
    template = PromptTemplate(tp_dict, {"sentence": "</text>"}, ice_token="</E>")

    # generate retriever for each client: each local train data will be used to build local train corpus
    fed_retrievers = []
    RETRIEVER = get_fedretriever(args)
    for cid in range(args.num_clients * args.duplicate):
        retriever = RETRIEVER(fed_reader[cid], ice_num=args.local_ice_num)
        fed_retrievers.append(retriever)

    # ===== 2. Execute Federated Inference Pipeline =====
    setup_seed(args.seed)
    inferencer = PPLFedInferencer(
        model_name=args.model, output_json_filepath=args.log_dir, args=args
    )
    # perform local retriever and server inference
    predictions = inferencer.inference(
        fed_retrievers,
        query_dataset=fed_dataset["test"],
        ice_template=template,
        concat=args.concat,
        output_json_filepath=args.log_dir,
        args=args,
    )

    # ===== 3. Save Prediction =====
    prediction_file = os.path.join(
        args.log_dir, f"prediction_{'debug' if args.debug else 'run'}.json"
    )
    with open(prediction_file, "w") as f:
        json.dump(predictions, f)

    label_file = os.path.join(
        args.log_dir, f"label_{'debug' if args.debug else 'run'}.json"
    )
    with open(label_file, "w") as f:
        # json.dump(data.references, f)
        json.dump(fed_dataset["test"]["label"], f)


if __name__ == "__main__":
    args = argparse.ArgumentParser()
    args.add_argument("--dataset", type=str, default="sst2", help="Dataset name")
    args.add_argument(
        "--subset_num",
        type=int,
        default=None,
        help="Number of subset test set for query",
    )
    args.add_argument("--partition", default="iid", choices=["iid", "noniid"], type=str)
    args.add_argument("--num_clients", type=int, default=3)
    args.add_argument(
        "--duplicate",
        default=1,
        type=int,
        help="Duplicate time for all clients. If num_clients=4, duplicate=2, then the final overall clients is num_clients*duplicate=8.",
    )
    args.add_argument("--major_classes_num", default=-1, type=int)
    args.add_argument(
        "--model",
        type=str,
        default="EleutherAI/gpt-neo-2.7B",
        help="Pretrained LLM model name",
    )
    args.add_argument("--local_ice_num", default=None, type=int)
    args.add_argument(
        "--overall_local_ice_num",
        default=None,
        type=int,
        help="Optional choice, automatically assign local_ice_num based on overall_local_ice_num and num_clients",
    )
    args.add_argument(
        "--server_ice_num",
        default=-1,
        type=int,
        help="Server side ICE Number, server_ice_num <= num_clients * local_ice_num",
    )
    args.add_argument(
        "--concat", default="simple", type=str, choices=["simple", "merge", "reorder"]
    )
    args.add_argument(
        "--retriever", type=str, default="bm25", help="Server side Retriever Type"
    )  # use 'bm25'
    args.add_argument(
        "--log_dir",
        type=str,
        default=f"fed_icl_log",
        help="Logging directory",
    )
    args.add_argument("--debug", action="store_true")
    args.add_argument("--run", action="store_true")
    args.add_argument("--seed", default=0, type=int)
    args.add_argument("--proj_name", default="FL-ICL-debug")
    args.add_argument("--group_name", default=None, type=str)

    args = args.parse_args()

    args.num_classes = data2numcls[args.dataset]

    if args.overall_local_ice_num is not None and args.local_ice_num is None:
        args.local_ice_num = int(args.overall_local_ice_num / args.num_clients)

    if args.server_ice_num == -1:
        args.server_ice_num = int(args.local_ice_num * args.num_clients)

    if args.local_ice_num * args.num_clients > args.server_ice_num:
        args.concat = "reorder"

    # prepare output files
    model = args.model.replace("/", "_")
    part_name = f"{args.partition}_clients={args.num_clients}"
    if args.partition == "noniid":
        part_name += f"_majorclass={args.major_classes_num}"

    if args.duplicate > 1:
        part_name += f"_duplicate={args.duplicate}"

    data_folder = f"{args.dataset}"
    if args.subset_num is not None:
        data_folder += f"_query-num={args.subset_num}"

    host_name = socket.gethostname()
    if "server_name" in host_name.lower():
        cache_root = "cache/root/directory"
    else:
        raise ValueError(f"Check the server hostname for log_dir initialization.")

    args.log_dir = os.path.join(
        cache_root,
        args.log_dir,
        f"{data_folder}/{part_name}/model={model}_retriever=fed{args.retriever}_local-ice={args.local_ice_num}_server-ice={args.server_ice_num}_concat={args.concat}/seed={args.seed}",
    )
    os.makedirs(args.log_dir, exist_ok=True)

    # perform FL-ICL pipeline
    if args.run:
        run_name = f"fed_{data_folder}_{part_name}_model={model}_retriever=fed{args.retriever}_local-ice={args.local_ice_num}_server-ice={args.server_ice_num}_concat={args.concat}_seed={args.seed}"
        wb_run = wandb.init(
            config=args, project=args.proj_name, name=run_name, group=args.group_name
        )

        if (
            args.partition == "noniid"
            and args.major_classes_num * args.num_clients < args.num_classes
        ):
            wb_run.finish()
            exit()

    if args.run:
        run(args)

    # read result from saved files and calculate performance score
    scores = print_score(args)
    if args.run:
        wb_run.log(scores)
        wb_run.finish()
