"""
This script is for, centralized ICL only using proxy dataset (from test set) as training set for ICE retrieve.
"""

from datasets import load_dataset
from openicl import DatasetReader, PromptTemplate, PPLInferencer, AccEvaluator
from util.noise import gen_noise_classification
from util.template import gen_template_classification
from util.retriever import get_retriever, get_fedretriever
from util.verification import data_verification
from util.icl_ppl_fed_inferencer import PPLFedInferencer
from util.misc import setup_seed
from util.partition import (
    data2usename,
    data2numcls,
    datasplit_subset,
    cls_iid_partition,
)

import argparse
import os
import json
import wandb
import socket

from huggingface_hub import login, HfApi, HfFolder

token = "HFTOKEN"
login(token=token)


def run(args):
    # load the data
    dataset = load_dataset(data2usename[args.dataset])

    if args.dataset.startswith("gen-"):
        input_col = "paraphrase"
    else:
        input_col = "sentence"

    if "test" not in dataset:
        test_split = "validation"
    else:
        test_split = "test"

    proxy_data, remain_data = datasplit_subset(
        dataset[args.proxy_split],
        subset_num=args.proxy_size,
        split=test_split,
        verbose=True,
        return_remain=True,
        seed=args.seed,
    )
    if args.proxy_split == test_split:
        dataset[test_split] = remain_data

    dataset["train"] = proxy_data  # replace training data with proxy dataset

    # evaluate on subset of test
    fed_dataset = cls_iid_partition(
        dataset=dataset,
        split="train",
        data_name=args.dataset,
        num_clients=1,
        test_split=test_split,
        subset_num=args.subset_num,
    )

    # save selected test query subset
    subset_orig_idxs = fed_dataset[test_split]["idx"]
    subset_orig_idxs_file = os.path.join(args.log_dir, "query_subset_orig_idxs.json")
    with open(subset_orig_idxs_file, "w") as f:
        json.dump(subset_orig_idxs, f)

    # make datareader for all clients' local training data
    fed_reader = [
        DatasetReader(
            fed_dataset[f"train-client0"],
            input_columns=[input_col],
            output_column="label",
        )
    ]

    # get template for classification

    tp_dict = gen_template_classification(args)
    template = PromptTemplate(tp_dict, {input_col: "</text>"}, ice_token="</E>")

    # generate retriever for each client: each local train data will be used to build local train corpus
    RETRIEVER = get_fedretriever(args)
    fed_retrievers = [RETRIEVER(fed_reader[0], ice_num=args.local_ice_num)]

    #  ===== 2. Obtain the optimal budget allocation for proxy dataset =====
    setup_seed(args.seed)
    inferencer = PPLFedInferencer(
        model_name=args.model, output_json_filepath=args.log_dir, args=args
    )
    # perform local retriever and server inference
    predictions = inferencer.inference(
        fed_retrievers,
        query_dataset=fed_dataset["test"],
        ice_template=template,
        concat=args.concat,
        output_json_filepath=args.log_dir,
        args=args,
    )

    # ===== 3. Save Prediction =====
    prediction_file = os.path.join(
        args.log_dir, f"prediction_{'debug' if args.debug else 'run'}.json"
    )
    with open(prediction_file, "w") as f:
        json.dump(predictions, f)

    label_file = os.path.join(
        args.log_dir, f"label_{'debug' if args.debug else 'run'}.json"
    )
    with open(label_file, "w") as f:
        # json.dump(data.references, f)
        json.dump(fed_dataset["test"]["label"], f)


def print_score(args):
    # load
    prediction_file = os.path.join(
        args.log_dir, f"prediction_{'debug' if args.debug else 'run'}.json"
    )
    label_file = os.path.join(
        args.log_dir, f"label_{'debug' if args.debug else 'run'}.json"
    )
    with open(prediction_file, "r") as f:
        prediction = data_loaded = json.load(f)
    with open(label_file, "r") as f:
        reference = data_loaded = json.load(f)

    evaluator = AccEvaluator()
    score = evaluator.score(predictions=prediction, references=reference)
    print(score)
    return score


if __name__ == "__main__":
    args = argparse.ArgumentParser()
    args.add_argument("--dataset", type=str, default="sst2", help="Dataset name")
    args.add_argument("--proxy_split", default="test", type=str)
    args.add_argument("--proxy_size", default=500, type=int)
    args.add_argument(
        "--subset_num",
        type=int,
        default=None,
        help="Number of subset test set for query",
    )
    args.add_argument(
        "--model",
        type=str,
        default="EleutherAI/gpt-neo-2.7B",
        help="Pretrained LLM model name",
    )
    args.add_argument(
        "--ice_num",
        default=-1,
        type=int,
        help="ICE Number, same as server_ice_num and local_ice_num in this experiment",
    )
    args.add_argument(
        "--retriever", type=str, default="topk", help="Server side Retriever Type"
    )  # use 'bm25'
    args.add_argument(
        "--log_dir",
        type=str,
        default=f"fed_icl_log",
        help="Logging directory",
    )
    args.add_argument("--debug", action="store_true")
    args.add_argument("--run", action="store_true")
    args.add_argument("--seed", default=0, type=int)
    args.add_argument("--proj_name", default="FL-ICL-debug")
    args.add_argument("--group_name", default=None, type=str)

    args = args.parse_args()

    args.num_classes = data2numcls[args.dataset]

    args.local_ice_num = args.ice_num
    args.server_ice_num = args.ice_num
    args.concat = "simple"

    # prepare output files
    model = args.model.replace("/", "_")

    data_folder = f"{args.dataset}"
    data_folder += f"_proxy_split={args.proxy_split}-proxy_size={args.proxy_size}"
    if args.subset_num is not None:
        data_folder += f"_query-num={args.subset_num}"

    host_name = socket.gethostname()
    if "server_name" in host_name.lower():
        cache_root = "cache/root/directory"
    else:
        raise ValueError(f"Check the server hostname for log_dir initialization.")

    args.log_dir = os.path.join(
        cache_root,
        args.log_dir,
        f"{data_folder}/model={model}_retriever=fed{args.retriever}_ice-num={args.ice_num}/seed={args.seed}",
    )
    os.makedirs(args.log_dir, exist_ok=True)

    # perform FL-ICL pipeline
    if args.run:
        run_name = f"fed_{data_folder}_model={model}_retriever=fed{args.retriever}_ice-num={args.ice_num}_seed={args.seed}"
        wb_run = wandb.init(
            config=args, project=args.proj_name, name=run_name, group=args.group_name
        )

    if args.run:
        run(args)

    # read result from saved files and calculate performance score
    scores = print_score(args)
    if args.run:
        wb_run.log(scores)
        wb_run.finish()
