from modules.agents import REGISTRY as agent_REGISTRY
from components.action_selectors import REGISTRY as action_REGISTRY
import torch as th


# This multi-agent controller shares parameters between agents
class DAVEMAC:
    def __init__(self, scheme, groups, args):
        self.n_agents = args.n_agents
        self.args = args
        input_shape = self._get_input_shape(scheme)
        self._build_agents(input_shape)
        self.agent_output_type = args.agent_output_type

        self.action_selector = action_REGISTRY[args.action_selector](args)


    def select_actions(self, ep_batch, t_ep, t_env, bs=slice(None), test_mode=False):
        # Only select actions for the selected batch elements in bs
        avail_actions = ep_batch["avail_actions"][:, t_ep]
        agent_alterego_outputs, agent_ego_outputs = self.forward(ep_batch, t_ep, test_mode=test_mode)
        chosen_actions = self.action_selector.select_action(agent_ego_outputs[bs], avail_actions[bs], t_env, test_mode=test_mode)
        return chosen_actions

    def forward(self, ep_batch, t, test_mode=False):
        agent_inputs = self._build_inputs(ep_batch, t)
        avail_actions = ep_batch["avail_actions"][:, t]
        agent_alterego_outs, self.alterego_hidden_states = self.alter_ego_agent(agent_inputs, self.alterego_hidden_states)
        agent_ego_outs, self.ego_hidden_states = self.ego_agent(agent_inputs, self.ego_hidden_states)
        return agent_alterego_outs.view(ep_batch.batch_size, self.n_agents, -1), agent_ego_outs.view(ep_batch.batch_size, self.n_agents, -1)

    def init_hidden(self, batch_size):
        self.alterego_hidden_states = self.alter_ego_agent.init_hidden().unsqueeze(0).expand(batch_size, self.n_agents, -1)  # bav
        self.ego_hidden_states = self.ego_agent.init_hidden().unsqueeze(0).expand(batch_size, self.n_agents, -1)  # bav

    def parameters(self):
        return self.alter_ego_agent.parameters()
    
    def ego_parameters(self):
        return self.ego_agent.parameters()

    def load_state(self, other_mac):
        self.alter_ego_agent.load_state_dict(other_mac.alter_ego_agent.state_dict())
        self.ego_agent.load_state_dict(other_mac.ego_agent.state_dict())

    def cuda(self):
        self.alter_ego_agent.cuda()
        self.ego_agent.cuda()

    def save_models(self, path):
        th.save(self.alter_ego_agent.state_dict(), "{}/alter_ego_agent.th".format(path))
        th.save(self.ego_agent.state_dict(), "{}/ego_agent.th".format(path))

    def load_models(self, path):
        self.alter_ego_agent.load_state_dict(th.load("{}/alter_ego_agent.th".format(path), map_location=lambda storage, loc: storage))
        self.ego_agent.load_state_dict(th.load("{}/ego_agent.th".format(path), map_location=lambda storage, loc: storage))

    def _build_agents(self, input_shape):
        self.ego_agent = agent_REGISTRY['ego'](input_shape, self.args)
        self.alter_ego_agent = agent_REGISTRY['alterego'](input_shape, self.args)

    def _build_inputs(self, batch, t):
        # Assumes homogenous agents with flat observations.
        # Other MACs might want to e.g. delegate building inputs to each agent
        bs = batch.batch_size
        inputs = []
        inputs.append(batch["obs"][:, t])  # b1av
        if self.args.obs_last_action:
            if t == 0:
                inputs.append(th.zeros_like(batch["actions_onehot"][:, t]))
            else:
                inputs.append(batch["actions_onehot"][:, t-1])
        if self.args.obs_agent_id:
            inputs.append(th.eye(self.n_agents, device=batch.device).unsqueeze(0).expand(bs, -1, -1))

        inputs = th.cat([x.reshape(bs*self.n_agents, -1) for x in inputs], dim=1)
        return inputs

    def _get_input_shape(self, scheme):
        input_shape = scheme["obs"]["vshape"]
        if self.args.obs_last_action:
            input_shape += scheme["actions_onehot"]["vshape"][0]
        if self.args.obs_agent_id:
            input_shape += self.n_agents

        return input_shape
