import copy
from components.episode_buffer import EpisodeBatch
from modules.mixers.vdn import VDNMixer
from modules.mixers.qmix import QMixer
from modules.mixers.igm_free_qmix import IGMFREEQMixer
import torch as th
from torch.optim import RMSprop
import torch.nn as nn
import torch.nn.functional as F
from torch.distributions import Categorical, Distribution
from typing import List
from modules.exploration.encode import Encoder
import numpy as np


class DAVELearner:
    def __init__(self, mac, scheme, logger, args):
        self.args = args
        self.mac = mac
        self.logger = logger

        self.alterego_params = list(mac.parameters())

        self.last_target_update_episode = 0
        
        self.encoder = Encoder(int(np.prod(args.state_shape)), args)

        self.mixer = None
        if args.mixer is not None:
            if args.mixer == "vdn":
                self.mixer = VDNMixer()
            elif args.mixer == "qmix":
                self.mixer = QMixer(args)
            elif args.mixer == "dave":
                self.mixer = IGMFREEQMixer(args)
            else:
                raise ValueError("Mixer {} not recognised.".format(args.mixer))
            self.alterego_params += list(self.mixer.parameters())
            self.target_mixer = copy.deepcopy(self.mixer)

        self.ego_params = list(mac.ego_parameters())
        self.alterego_optimiser = RMSprop(params=self.alterego_params, lr=args.lr, alpha=args.optim_alpha, eps=args.optim_eps)
        self.ego_optimiser = RMSprop(params=self.ego_params, lr=args.lr, alpha=args.optim_alpha, eps=args.optim_eps)
        self.encoder_optimiser = RMSprop(params=self.encoder.parameters(), lr=args.lr, alpha=args.optim_alpha, eps=args.optim_eps)

        # a little wasteful to deepcopy (e.g. duplicates action selector), but should work for any MAC
        self.target_mac = copy.deepcopy(mac)

        self.log_stats_t = -self.args.learner_log_interval - 1

    def train(self, batch: EpisodeBatch, t_env: int, episode_num: int):
        if t_env < self.args.anti_ego_decay:
            anti_ego_weight = self.args.anti_ego_start - (self.args.anti_ego_start - self.args.anti_ego_end) * (t_env / self.args.anti_ego_decay)
        else:
            anti_ego_weight = self.args.anti_ego_end
        # Get the relevant quantities
        rewards = batch["reward"][:, :-1]
        actions = batch["actions"][:, :-1]
        terminated = batch["terminated"][:, :-1].float()
        mask = batch["filled"][:, :-1].float()
        mask[:, 1:] = mask[:, 1:] * (1 - terminated[:, :-1])
        avail_actions = batch["avail_actions"]
        
        if anti_ego_weight > 0.0:
            hidden_states = self.encoder.encode(batch['state'], batch['actions_onehot'])
            recon_states, recon_actions = self.encoder.decode(hidden_states)
            
            recon_state_loss = (batch['state'] - recon_states).pow(2)
            recon_action_loss = - batch['actions_onehot'] * recon_actions
            
            recon_state_mask = batch["filled"].repeat([1, 1, batch['state'].size(-1)])
            recon_action_mask = batch["filled"].unsqueeze(-1).repeat([1, 1, *batch['actions_onehot'].size()[-2:]])
            recon_state_loss = (recon_state_loss * recon_state_mask).sum() / recon_state_mask.sum()
            recon_action_loss = (recon_action_loss * recon_action_mask).sum() / recon_action_mask.sum()
            recon_loss = recon_state_loss + recon_action_loss
            log_recon_loss = recon_loss.detach().item()
            
            # Optimise
            self.encoder_optimiser.zero_grad()
            recon_loss.backward()
            grad_norm = th.nn.utils.clip_grad_norm_(self.encoder.parameters(), self.args.grad_norm_clip)
            self.encoder_optimiser.step()
        else:
            log_recon_loss = 0.0


        # Calculate estimated Q-Values
        mac_out = []
        mac_policy = []
        self.mac.init_hidden(batch.batch_size)
        for t in range(batch.max_seq_length):
            agent_outs, proposal = self.mac.forward(batch, t=t)
            mac_out.append(agent_outs)
            mac_policy.append(proposal)
        mac_out = th.stack(mac_out, dim=1)  # Concat over time
        mac_policy = th.stack(mac_policy, dim=1)  # Concat over time

        # Pick the Q-Values for the actions taken by each agent
        chosen_action_qvals = th.gather(mac_out[:, :-1], dim=3, index=actions).squeeze(3)  # Remove the last dim

        # Calculate the Q-Values necessary for the target
        with th.no_grad():
            target_mac_out = []
            self.target_mac.init_hidden(batch.batch_size)
            for t in range(batch.max_seq_length):
                target_agent_outs, target_proposal = self.target_mac.forward(batch, t=t)
                target_mac_out.append(target_agent_outs)

        # We don't need the first timesteps Q-Value estimate for calculating targets
        target_mac_out = th.stack(target_mac_out[1:], dim=1)  # Concat across time

        # Mask out unavailable actions
        target_mac_out[avail_actions[:, 1:] == 0] = -9999999
            
        
        mac_anti_ego = mac_policy.clone()
        mac_policy[avail_actions == 0] = -9999999
        mac_anti_ego[avail_actions == 0] = 9999999
        mac_policy = F.softmax(mac_policy, dim=-1)
        mac_anti_ego = F.softmin(mac_anti_ego, dim=-1)
        dist = multi_categorical_maker([self.args.n_actions] * self.args.n_agents)
        ego_d = dist(probs=mac_policy.flatten(0, 1).flatten(-2))
        anti_ego_d = dist(probs=mac_anti_ego.flatten(0, 1).flatten(-2))
        ego_policy_loss = 0.
        
        sample_actions = []
        sample_log_probs = []
        for _ in range(self.args.sample_num):
            s_acts = ego_d.sample()
            s_logps = ego_d.log_prob(s_acts)
            sample_actions.append(s_acts)
            sample_log_probs.append(s_logps)
        if anti_ego_weight > 0.0:
            for _ in range(self.args.sample_num):
                s_acts = anti_ego_d.sample()
                s_logps = ego_d.log_prob(s_acts)
                sample_actions.append(s_acts)
                sample_log_probs.append(s_logps)
        sample_actions = th.stack(sample_actions, dim=0)
        sample_log_probs = th.stack(sample_log_probs, dim=0)
        
        total_sample_num = sample_actions.size(0)
        
        sample_actions = sample_actions.view(total_sample_num, self.args.batch_size, -1, self.args.n_agents, 1)
        sample_log_probs = sample_log_probs.view(total_sample_num, self.args.batch_size, -1, 1)
        
        with th.no_grad():
            selected_mac_out = th.gather(mac_out.detach().unsqueeze(0).repeat([self.args.sample_num, 1, 1, 1, 1]), dim=4, index=sample_actions[:self.args.sample_num]).squeeze(4).flatten(0, 1)
            selected_q_tot = self.mixer(selected_mac_out, batch["state"].unsqueeze(0).repeat([self.args.sample_num, 1, 1, 1]).flatten(0, 1), sample_actions[:self.args.sample_num].flatten(0, 1).squeeze(-1)).view(self.args.sample_num, self.args.batch_size, -1, 1)
            _, ego_selected_q_tot_id = selected_q_tot.max(0)
            ego_selected_q_tot_id = ego_selected_q_tot_id.unsqueeze(0)
        ego_policy_loss_1 = th.gather(sample_log_probs, dim=0, index=ego_selected_q_tot_id).squeeze(0)
        ego_policy_loss_1 = (-ego_policy_loss_1[:, :-1] * mask).sum() / mask.sum()
        if anti_ego_weight > 0.0:
            with th.no_grad():
                hidden_states = self.encoder.encode(batch['state'].unsqueeze(0).repeat([self.args.sample_num, 1, 1, 1]), F.one_hot(sample_actions[self. args.sample_num:].squeeze(-1), num_classes=self.args.n_actions).float())
                recon_states, recon_actions = self.encoder.decode(hidden_states)
                recon_state_loss = (batch['state'].unsqueeze(0) - recon_states).pow(2)
                recon_action_loss = - F.one_hot(sample_actions[self.args.sample_num:].squeeze(-1), num_classes=self.args.n_actions).float() *   recon_actions
                recon_state_loss = recon_state_loss.sum(-1, keepdim=True)
                recon_action_loss = recon_action_loss.sum(-1).sum(-1, keepdim=True)
                recon_loss = recon_state_loss + recon_action_loss
                _, anti_ego_selected_q_tot_id = recon_loss.max(0)
                anti_ego_selected_q_tot_id = anti_ego_selected_q_tot_id.unsqueeze(0) + self.args.sample_num
            ego_policy_loss_2 = th.gather(sample_log_probs, dim=0, index=anti_ego_selected_q_tot_id).squeeze(0)
            ego_policy_loss_2 = (-ego_policy_loss_2[:, :-1] * mask).sum() / mask.sum()
            ego_policy_loss = ego_policy_loss + ego_policy_loss_1 + anti_ego_weight * ego_policy_loss_2
        else:
            ego_policy_loss = ego_policy_loss + ego_policy_loss_1

        with th.no_grad():
            selected_target_mac_out = th.gather(target_mac_out.unsqueeze(0).repeat([self.args.sample_num, 1, 1, 1, 1]), dim=-1, index=sample_actions[:self.args.sample_num, :, 1:]).squeeze(-1).flatten(0, 1)
        
         # Mix
        if self.mixer is not None:
            chosen_action_qvals = self.mixer(chosen_action_qvals, batch["state"][:, :-1], batch['actions'][:, :-1].squeeze(-1))
            with th.no_grad():
                target_max_qvals = self.target_mixer(selected_target_mac_out, batch["state"][:, 1:].unsqueeze(0).repeat([self.args.sample_num, 1, 1, 1]).flatten(0, 1), sample_actions[:self.args.sample_num, :, 1:].flatten(0, 1).squeeze(-1))
                target_max_qvals = target_max_qvals.view(self.args.sample_num, self.args.batch_size, -1, 1).mean(0)

        # Calculate 1-step Q-Learning targets
        targets = rewards + self.args.gamma * (1 - terminated) * target_max_qvals

        # Td-error
        td_error = (chosen_action_qvals - targets.detach())

        mask = mask.expand_as(td_error)

        # 0-out the targets that came from padded data
        masked_td_error = td_error * mask

        # Normal L2 loss, take mean over actual data
        rl_loss = (masked_td_error ** 2).sum() / mask.sum()
        
        # Optimise
        self.alterego_optimiser.zero_grad()
        rl_loss.backward()
        grad_norm = th.nn.utils.clip_grad_norm_(self.alterego_params, self.args.grad_norm_clip)
        self.alterego_optimiser.step()
        
        # Optimise
        self.ego_optimiser.zero_grad()
        ego_policy_loss.backward()
        proposal_grad_norm = th.nn.utils.clip_grad_norm_(self.ego_params, self.args.grad_norm_clip)
        self.ego_optimiser.step()


        if (episode_num - self.last_target_update_episode) / self.args.target_update_interval >= 1.0:
            self._update_targets()
            self.last_target_update_episode = episode_num

        if t_env - self.log_stats_t >= self.args.learner_log_interval:
            self.logger.log_stat("rl_loss", rl_loss.item(), t_env)
            self.logger.log_stat("anti_ego_weight", anti_ego_weight, t_env)
            self.logger.log_stat("recon_loss", log_recon_loss, t_env)
            self.logger.log_stat("ego_policy_loss", ego_policy_loss.item(), t_env)
            self.logger.log_stat("grad_norm", grad_norm, t_env)
            self.logger.log_stat("policy_grad_norm", proposal_grad_norm, t_env)
            mask_elems = mask.sum().item()
            self.logger.log_stat("td_error_abs", (masked_td_error.abs().sum().item()/mask_elems), t_env)
            self.logger.log_stat("q_taken_mean", (chosen_action_qvals * mask).sum().item()/(mask_elems * self.args.n_agents), t_env)
            self.logger.log_stat("target_mean", (targets * mask).sum().item()/(mask_elems * self.args.n_agents), t_env)
            self.log_stats_t = t_env

    def _update_targets(self):
        self.target_mac.load_state(self.mac)
        if self.mixer is not None:
            self.target_mixer.load_state_dict(self.mixer.state_dict())
        self.logger.console_logger.info("Updated target network")

    def cuda(self):
        self.mac.cuda()
        self.target_mac.cuda()
        if self.mixer is not None:
            self.mixer.cuda()
            self.target_mixer.cuda()
        self.encoder.cuda()

    def save_models(self, path):
        self.mac.save_models(path)
        if self.mixer is not None:
            th.save(self.mixer.state_dict(), "{}/mixer.th".format(path))
        th.save(self.alterego_optimiser.state_dict(), "{}/opt.th".format(path))

    def load_models(self, path):
        self.mac.load_models(path)
        # Not quite right but I don't want to save target networks
        self.target_mac.load_models(path)
        if self.mixer is not None:
            self.mixer.load_state_dict(th.load("{}/mixer.th".format(path), map_location=lambda storage, loc: storage))
        self.alterego_optimiser.load_state_dict(th.load("{}/opt.th".format(path), map_location=lambda storage, loc: storage))



class MultiCategorical(Distribution):
    def __init__(self, dists: List[Categorical]):
        super().__init__(validate_args=False)
        self.dists = dists

    def log_prob(self, value):
        ans = []
        for d, v in zip(self.dists, th.split(value, 1, dim=-1)):
            ans.append(d.log_prob(v.squeeze(-1)))
        return th.stack(ans, dim=-1).sum(dim=-1)

    def entropy(self):
        return th.stack([d.entropy() for d in self.dists], dim=-1).sum(dim=-1)

    def sample(self, sample_shape=th.Size()):
        return th.stack([d.sample(sample_shape) for d in self.dists], dim=-1)


def multi_categorical_maker(nvec):
    def get_multi_categorical(probs):
        start = 0
        ans = []
        for n in nvec:
            ans.append(Categorical(probs=probs[:, start: start + n]))
            start += n
        return MultiCategorical(ans)
    return get_multi_categorical