"""Tests for stochastic_occupancy."""

from absl.testing import absltest
from absl.testing import parameterized
import numpy as np
import pandas as pd

from smart_control.simulator.stochastic_occupancy import LIGHTSWITCHOccupancy
from smart_control.simulator.stochastic_occupancy import OccupancyStateEnum
from smart_control.simulator.stochastic_occupancy import ZoneOccupant

# fmt: off
# pylint: disable=line-too-long
_EXPECTED_ZONE_OCCUPANCIES_UTC = [0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 2.0, 2.0, 2.0, 2.0, 2.0, 2.0, 2.0, 2.0, 2.0, 2.0, 2.0, 2.0, 4.0, 4.0, 4.0, 4.0, 4.0, 4.0, 4.0, 4.0, 4.0, 4.0, 4.0, 4.0, 10.0, 10.0, 10.0, 10.0, 10.0, 10.0, 10.0, 10.0, 10.0, 10.0, 10.0, 10.0, 10.0, 10.0, 10.0, 10.0, 10.0, 10.0, 10.0, 10.0, 10.0, 10.0, 10.0, 10.0, 7.0, 7.0, 7.0, 7.0, 7.0, 7.0, 7.0, 7.0, 7.0, 7.0, 7.0, 7.0, 3.0, 4.0, 5.0, 5.0, 6.0, 6.0, 6.0, 6.0, 6.0, 6.0, 7.0, 8.0, 5.0, 5.0, 6.0, 7.0, 7.0, 7.0, 7.0, 7.0, 7.0, 8.0, 8.0, 9.0, 9.0, 10.0, 10.0, 10.0, 10.0, 10.0, 10.0, 10.0, 10.0, 10.0, 10.0, 10.0, 6.0, 6.0, 6.0, 6.0, 6.0, 6.0, 6.0, 6.0, 6.0, 6.0, 6.0, 6.0, 2.0, 2.0, 2.0, 2.0, 2.0, 2.0, 2.0, 2.0, 2.0, 2.0, 2.0, 2.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0]
_EXPECTED_ZONE_OCCUPANCIES_PACIFIC = [2.0, 2.0, 2.0, 2.0, 2.0, 2.0, 2.0, 2.0, 2.0, 2.0, 2.0, 2.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 2.0, 2.0, 2.0, 2.0, 2.0, 2.0, 2.0, 2.0, 2.0, 2.0, 2.0, 2.0, 4.0, 4.0, 4.0, 4.0, 4.0, 4.0, 4.0, 4.0, 4.0, 4.0, 4.0, 4.0, 10.0, 10.0, 10.0, 10.0, 10.0, 10.0, 10.0, 10.0, 10.0, 10.0, 10.0, 10.0, 10.0, 10.0, 10.0, 10.0, 10.0, 10.0, 10.0, 10.0, 10.0, 10.0, 10.0, 10.0, 7.0, 7.0, 7.0, 7.0, 7.0, 7.0, 7.0, 7.0, 7.0, 7.0, 7.0, 7.0, 3.0, 4.0, 5.0, 5.0, 6.0, 6.0, 6.0, 6.0, 6.0, 6.0, 7.0, 8.0, 5.0, 5.0, 6.0, 7.0, 7.0, 7.0, 7.0, 7.0, 7.0, 8.0, 8.0, 9.0, 9.0, 10.0, 10.0, 10.0, 10.0, 10.0, 10.0, 10.0, 10.0, 10.0, 10.0, 10.0]
_EXPECTED_ZONE_OCCUPANCIES_EASTERN = [0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 2.0, 2.0, 2.0, 2.0, 2.0, 2.0, 2.0, 2.0, 2.0, 2.0, 2.0, 2.0, 4.0, 4.0, 4.0, 4.0, 4.0, 4.0, 4.0, 4.0, 4.0, 4.0, 4.0, 4.0, 10.0, 10.0, 10.0, 10.0, 10.0, 10.0, 10.0, 10.0, 10.0, 10.0, 10.0, 10.0, 10.0, 10.0, 10.0, 10.0, 10.0, 10.0, 10.0, 10.0, 10.0, 10.0, 10.0, 10.0, 7.0, 7.0, 7.0, 7.0, 7.0, 7.0, 7.0, 7.0, 7.0, 7.0, 7.0, 7.0, 3.0, 4.0, 5.0, 5.0, 6.0, 6.0, 6.0, 6.0, 6.0, 6.0, 7.0, 8.0, 5.0, 5.0, 6.0, 7.0, 7.0, 7.0, 7.0, 7.0, 7.0, 8.0, 8.0, 9.0, 9.0, 10.0, 10.0, 10.0, 10.0, 10.0, 10.0, 10.0, 10.0, 10.0, 10.0, 10.0, 6.0, 6.0, 6.0, 6.0, 6.0, 6.0, 6.0, 6.0, 6.0, 6.0, 6.0, 6.0, 2.0, 2.0, 2.0, 2.0, 2.0, 2.0, 2.0, 2.0, 2.0, 2.0, 2.0, 2.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0]
# pylint: enable=line-too-long
# fmt: on

# Constants for test cases
debug_print = False
SEED = 511211
STEP_SIZE = pd.Timedelta(minutes=5)
EARLIEST_EXPECTED_ARRIVAL_HOUR = 8
LATEST_EXPECTED_ARRIVAL_HOUR = 10
EARLIEST_EXPECTED_DEPARTURE_HOUR = 16
LATEST_EXPECTED_DEPARTURE_HOUR = 18
NUM_OCCUPANTS = 10


class LIGHTSWITCHOccupancyTest(parameterized.TestCase):

  @parameterized.parameters(
      (None, _EXPECTED_ZONE_OCCUPANCIES_UTC),
      ('UTC', _EXPECTED_ZONE_OCCUPANCIES_UTC),
      ('US/Pacific', _EXPECTED_ZONE_OCCUPANCIES_PACIFIC),
      ('US/Eastern', _EXPECTED_ZONE_OCCUPANCIES_EASTERN),
  )
  def test_average_zone_occupancy_weekday(self, tz, expected_zone_occupancies):

    occupancy = LIGHTSWITCHOccupancy(
        zone_assignment=NUM_OCCUPANTS,
        earliest_expected_arrival_hour=EARLIEST_EXPECTED_ARRIVAL_HOUR,
        latest_expected_arrival_hour=LATEST_EXPECTED_ARRIVAL_HOUR,
        earliest_expected_departure_hour=EARLIEST_EXPECTED_DEPARTURE_HOUR,
        latest_expected_departure_hour=LATEST_EXPECTED_DEPARTURE_HOUR,
        lunch_start_hour=12,
        lunch_end_hour=14,
        time_step_sec=STEP_SIZE.total_seconds(),
        seed=SEED,
        time_zone=tz,
    )
    current_time = pd.Timestamp('2021-09-01 00:00', tz='UTC')
    actual_occupancies = []
    while current_time < pd.Timestamp('2021-09-01 23:00', tz='UTC'):
      n = occupancy.average_zone_occupancy(
          'zone_0', current_time, current_time + STEP_SIZE
      )
      actual_occupancies.append(n)

      current_time += STEP_SIZE
    self.assertSequenceEqual(expected_zone_occupancies, actual_occupancies)

  @parameterized.parameters(None, 'UTC', 'US/Eastern', 'US/Pacific')
  def test_peek(self, tz):
    random_state = np.random.RandomState(seed=SEED)
    occupant = ZoneOccupant(
        earliest_expected_arrival_hour=EARLIEST_EXPECTED_ARRIVAL_HOUR,
        latest_expected_arrival_hour=LATEST_EXPECTED_ARRIVAL_HOUR,
        earliest_expected_departure_hour=EARLIEST_EXPECTED_DEPARTURE_HOUR,
        latest_expected_departure_hour=LATEST_EXPECTED_DEPARTURE_HOUR,
        lunch_start_hour=12,
        lunch_end_hour=14,
        step_size=STEP_SIZE,
        random_state=random_state,
        time_zone=tz,
    )
    current_time = pd.Timestamp('2021-09-01 00:00', tz=tz)

    while current_time < pd.Timestamp('2021-09-01 23:00', tz=tz):
      state = occupant.peek(current_time=current_time)

      # Debugging information
      # print(f"Current time: {current_time}, Local time: {local_time}")
      # print(f"Current state: {state}")

      if current_time < pd.Timestamp(
          '2021-09-01 10:00', tz=tz
      ) or current_time >= pd.Timestamp('2021-09-01 17:00', tz=tz):
        self.assertEqual(OccupancyStateEnum.AWAY, state)
      elif (
          pd.Timestamp('2021-09-01 13:00', tz=tz)
          <= current_time
          < pd.Timestamp('2021-09-01 13:54', tz=tz)
      ):
        self.assertEqual(OccupancyStateEnum.AWAY, state)  # Lunch break
      else:
        self.assertEqual(OccupancyStateEnum.WORK, state)
      current_time += STEP_SIZE


if __name__ == '__main__':
  absltest.main()
