import os, sys
import fitz
import re
import json
from datetime import datetime
from typing import Optional, List, Callable, Any, Tuple, Dict
from abc import abstractmethod, ABC
import random
import numpy as np
import pandas as pd
import copy
import nltk
from nltk.corpus import stopwords
import pickle
from tqdm.autonotebook import tqdm
import itertools
from dotenv import load_dotenv

sys.path.insert(0,"../")

load_dotenv(dotenv_path="../.env")
nltk.download('stopwords')

random.seed(42)
np.random.seed(42)

from dataset_utils.reader import ADIQDataset
from dataset_utils.outputs import to_basic_prompt
from utils import file_handle
from models_utils.utils.concurrency import concurrent_dict_execution

ds = ADIQDataset("datasets/simpleV3.1")

incontext_expl = file_handle.load_json(
    "incontext_expl.json"
)['example_id']

question_template = """
## Asset Description:
{asset_type}: {asset_description}

## Conditions:
{conditions}

## Conditions in Natural Language:
"""

main_template = """
Your task is to read the asset description (## Asset Description:) and conditions (## Conditions:) applied on the asset and 
write the conditions (## Conditions:) in natural language  several examples are provided, complete the last sample.

{examples}
{question}
"""

from dataset_utils.question import Question

def condition_list(q:Question) -> str:
    qlist = question_template.format(
        asset_type = q.asset_type,
        asset_description = ds.asset_descriptions[q.asset_type],
        conditions = "\n".join(q.condition_description)
    )

    return qlist

def verberlizer_incontext(q:Question, incontext_expl:str) -> str:
    _q = question_template.format(
        asset_type = q.asset_type,
        asset_description = ds.asset_descriptions.get(q.asset_type,"NONE"),
        conditions = "\n".join(q.condition_description)
    )

    _template = main_template.format(
        examples = incontext_expl,
        question = _q
    )

    return _template

import models_utils.llm.watsonx as watsonx

_examples = file_handle.load_text("condition_verberlizer_examples.txt")

def _verbalize_conditions(q:Question) -> str:
    _prompt = verberlizer_incontext(q,_examples)
    response = watsonx.get_completion_response(
        _prompt
    )

    return watsonx.clean_response(response)

param_dict = {k:[v] for k,v in enumerate(ds.questions)}

for k,v in concurrent_dict_execution(
    _verbalize_conditions,
    param_dict
):
    ds.questions[k].verberlized_conditions = v

ds.save(
    "simpleV4",
    "datasets"
)