#!/bin/bash
set -e

# Usage function
usage() { echo "Usage: $0 [-l mp4_list.txt] [-o OUTPUT_DIR] [single_video.mp4]"; exit 1; }

# Parse command line arguments
while getopts ":l:o:g:p:" opt; do
  case $opt in
    l) LIST_TXT="${OPTARG}"       ;;
    o) CUSTOM_OUTDIR="${OPTARG}"  ;;
    g) NUM_GPUS="${OPTARG}"       ;;
    p) CONCURRENCY_PER_GPU="${OPTARG}"  ;;
    *) usage ;;
  esac
done
shift $((OPTIND-1))

# Set default values for GPU configuration
NUM_GPUS=${NUM_GPUS:-1}
CONCURRENCY_PER_GPU=${CONCURRENCY_PER_GPU:-1}

# Base project paths - modify these according to your setup
base_project_path="/path/to/your/project"
SAMURAI_SCRIPTS="$base_project_path/samurai/scripts"
STABLE_ANIMATOR="/path/to/StableAnimator"

# Model paths - download these models from official sources
# SAM2.1 model: Download from https://github.com/facebookresearch/sam2
SAM_MODEL_PATH="$base_project_path/samurai/sam2/checkpoints/sam2.1_hiera_large.pt"

# Main video processing function
process_video() {
    local video_path=$1
    local gpu_id=$2
    local name=$(basename "$video_path" .mp4)
    local work_dir="$OUTPUT_DIR/$name"
    local chunk_dir="${work_dir}/chunks"

    # Check if final result already exists
    if [[ -f "$work_dir/concatenated_videos.mp4" ]]; then
      echo "Concatenated video already exists. Skipping..."
      return 0
    fi

    # Set GPU device
    export CUDA_VISIBLE_DEVICES=$gpu_id
    mkdir -p "$work_dir"

    # Check for existing detection results
    detected_jpg=( "$work_dir"/frame_*_detected.jpg )
    boxes_txt=( "$work_dir"/frame_*_boxes.txt )
    images_dir="$work_dir/images"

    # Human detection step
    if [[ -f "${detected_jpg[0]}" \
       && -f "${boxes_txt[0]}" \
       && -d "$images_dir" \
       && -n "$(ls -A "$images_dir")" ]]; then
        echo "✔️ Detection already completed (${detected_jpg[0]} & ${boxes_txt[0]} & images folder has files), skipping human detection: $name"
    else
        echo "🔍 Starting human detection: $name"
        cd "$SAMURAI_SCRIPTS" || { echo "❌ Directory not found: $SAMURAI_SCRIPTS"; exit 1; }
        if ! python humanDet0509.py \
                --input_path "$video_path" \
                --conf_threshold 0.6 \
                --frame_index 0 \
                --output_dir  "$work_dir"; then

            echo "❌ Human detection failed: $name, cleaning intermediate results"
            cd /path/to/temp/cleanup/directory
            ./cleanup_script --dryrun --deleteSrc cp "$work_dir/" ./temp/
            rm -r "$work_dir/"
            return 1
        fi
        echo "✔️ Human detection completed: generated ${detected_jpg[0]} & ${boxes_txt[0]}"
    fi

    # SAM segmentation step
    local final_result="$work_dir/masks/final_result.mp4"
    if [[ -f "$final_result" ]]; then
        echo "✔️ SAM segmentation already completed ($final_result exists), skipping demo_mul: $name"
    else
        echo "🔍 Starting SAM segmentation: $name"
        cd "$SAMURAI_SCRIPTS" || { echo "❌ Directory not found: $SAMURAI_SCRIPTS"; exit 1; }
        if ! python demo_mul-0418.py \
                --input_dir  "$work_dir" \
                --output_dir "$work_dir/masks" \
                --video_path "$video_path" \
                --sam_script "$SAMURAI_SCRIPTS/demo_mul.py" \
                --model_path  "$SAM_MODEL_PATH" \
                --device      cuda:0; then
            echo "❌ SAM segmentation failed: $name, cleaning intermediate results"
            cd /path/to/temp/cleanup/directory
            ./cleanup_script --dryrun --deleteSrc cp "$work_dir/" ./temp/
            rm -r "$work_dir/"
            return 1
        fi
        echo "✔️ SAM segmentation completed: generated $final_result"
    fi

    # Count images for validation
    images_count=$(find "$work_dir/images" -type f | wc -l)
    echo "Number of files in images folder: $images_count"

    # Pose estimation step
    if [[ -d "$work_dir/poses" && -d "$work_dir/poses/allperson" && -d "$work_dir/poses/person_0" ]]; then
        skip=true
        # Check if pose processing is complete
        for subfolder in "$work_dir/poses/allperson" "$work_dir/poses/person_0"; do
            folder_count=$(find "$subfolder" -type f | wc -l)
            echo "Number of files in subfolder $subfolder: $folder_count"
            if [ "$folder_count" -ne "$images_count" ]; then
                skip=false
                break
            fi
        done
    else
        echo "❌ Missing poses or allperson/person_0 folders, need to execute DWPose processing"
        skip=false
    fi

    if $skip; then
        echo "✔️ Skipping DWPose processing, pose folder allperson/person_0 file count matches images."
    else
        echo "🔍 Starting DWPose processing..."
        cd "$STABLE_ANIMATOR"
        python -u DWPose/visualize_dwpose_with_ids_color_dataset_face_fullpose.py \
            --video_path        "$work_dir/images" \
            --output_video_path "$work_dir/poses" \
            --output_pose_path  "$work_dir/poses.npz" \
            --mask_folder       "$work_dir/masks" \
            --num_workers 1

    fi

    python mergeAlphamask.py \
        --images_dir "$work_dir/images" \
        --alpha_dir  "$work_dir/alphamasks" \
        --output     "$work_dir/green_video.mp4" \
        --fps 30

    # Final concatenation
    concatenate_videos "$name"

    echo "► Completed: $name  (GPU $gpu_id)"
}

# Video concatenation function
concatenate_videos() {
    local name=$1
    local BASE="$OUTPUT_DIR/$name"
    local POSE_DIR="$BASE/pose_videos"
    local RESULT="$BASE/final_samurai_result.mp4"
    local GREEN="$BASE/green_video.mp4"
    local OUT="$BASE/concatenated_videos.mp4"

    [[ -f "$OUT" ]] && { echo "$OUT already exists, skipping"; return; }
    [[ ! -f "$RESULT" || ! -d "$POSE_DIR" ]] && { echo "Missing required files, skipping $name"; return; }

    local TMP; TMP=$(mktemp -d); trap 'rm -rf "$TMP"' RETURN
    local H=$(ffprobe -v error -select_streams v:0 -show_entries stream=height -of csv=p=0 "$RESULT")
    local FPS=$(ffprobe -v error -select_streams v:0 -show_entries stream=r_frame_rate -of csv=p=0 "$RESULT")
    [[ $H -gt 1080 ]] && H=1080

    # Resize main result video
    ffmpeg -i "$RESULT" -vf scale=-1:"$H" -r "$FPS" -vsync 0 -c:v libx264 -preset fast -crf 23 "$TMP/res0.mp4" -y
    local RESIZED=("$TMP/res0.mp4")

    # Resize pose videos
    shopt -s nullglob
    for v in "$POSE_DIR"/*.mp4; do
        ffmpeg -i "$v" -vf scale=-1:"$H" -r "$FPS" -vsync 0 -c:v libx264 -preset fast -crf 23 "$TMP/$(basename "$v")" -y
        RESIZED+=("$TMP/$(basename "$v")")
    done
    shopt -u nullglob

    # Add green video if exists
    if [[ -f "$GREEN" ]]; then
        ffmpeg -i "$GREEN" -vf scale=-1:"$H" -r "$FPS" -vsync 0 -c:v libx264 -preset fast -crf 23 "$TMP/green.mp4" -y
        RESIZED+=("$TMP/green.mp4")
    fi

    # Check if we have videos to concatenate
    [[ ${#RESIZED[@]} -eq 0 ]] && { echo "No videos to concatenate"; return; }

    # Build filter string for horizontal stacking
    local FILTER=""
    for i in "${!RESIZED[@]}"; do FILTER+="[$i:v]"; done
    FILTER+="hstack=inputs=${#RESIZED[@]}:shortest=1"

    # Concatenate videos horizontally
    ffmpeg $(printf ' -i %s' "${RESIZED[@]}") \
           -filter_complex "$FILTER" \
           -c:v libx264 -vsync 0 -r "$FPS" -preset fast -crf 28 \
           -pix_fmt yuv420p -max_muxing_queue_size 9999 "$OUT" -y

    # Clean up intermediate files if successful
    if [[ -f "$OUT" ]]; then
        rm -rf "$POSE_DIR"
        rm -f  "$RESULT" "$GREEN"
        echo "Cleaned up intermediate files: $POSE_DIR and $RESULT / $GREEN"
    fi
}

# Set output directory
if [[ -n "$CUSTOM_OUTDIR" ]]; then
    OUTPUT_DIR="$CUSTOM_OUTDIR"
else
    OUTPUT_DIR="${DATASET_DIR}-process"
fi
mkdir -p "$OUTPUT_DIR"

# Collect video files to process
video_files=()
if [[ -n "$LIST_TXT" ]]; then
    [[ -f "$LIST_TXT" ]] || { echo "Cannot find $LIST_TXT"; exit 1; }
    while IFS= read -r line; do
        [[ -z $line ]] && continue
        abs=$(realpath "$line"); [[ -f $abs ]] && video_files+=("$abs")
    done < "$LIST_TXT"
elif [[ $# -eq 1 ]]; then
    abs=$(realpath "$1"); [[ -f $abs ]] || { echo "$abs does not exist"; exit 1; }
    video_files+=("$abs")
else
    mapfile -t video_files < <(find "$DATASET_DIR" -maxdepth 1 -type f -name '*.mp4' | sort)
fi

(( ${#video_files[@]} > 0 )) || { echo "No video files found"; exit 1; }
echo "Total ${#video_files[@]} videos to process"

# Initialize GPU process tracking
declare -A gpu_pids; for ((g=0; g<NUM_GPUS; g++)); do gpu_pids[$g]=""; done

# Wrapper function for process_video
process_video_wrapper() { process_video "$1" "$2"; }

# Main processing loop with GPU load balancing
for idx in "${!video_files[@]}"; do
    path="${video_files[$idx]}"
    gpu=$(( idx % NUM_GPUS ))

    # Wait for available slot on this GPU
    while :; do
        active=()
        for p in ${gpu_pids[$gpu]}; do kill -0 "$p" 2>/dev/null && active+=("$p"); done
        gpu_pids[$gpu]="${active[*]}"
        (( ${#active[@]} < CONCURRENCY_PER_GPU )) && break
        sleep 5
    done

    # Start processing on selected GPU
    (process_video_wrapper "$path" "$gpu") & pid=$!
    gpu_pids[$gpu]="${gpu_pids[$gpu]} $pid"
    echo "Started $(basename "$path") → GPU$gpu (PID $pid)"
    sleep 0.05
done

# Wait for all processes to complete
wait
echo "All processing completed, results saved in: $OUTPUT_DIR"