#!/usr/bin/env bash
set -e
cd /path/to/project/eval/src

# Set paths - modify these according to your setup
GT_SEQ_ROOT=/path/to/project/eval/evaldata/ablation_GT/datapipeoutput
PRED_SEQ_ROOT=/path/to/prediction/sequences
WORK_ROOT=/path/to/working/directory

mkdir -p "$WORK_ROOT"

PY=python
CONVERT=./tools/convert_to_mot.py
EVAL=TrackEval/scripts/run_mot_challenge.py

echo "===> 1. Generate MOTChallenge txt files ..."
$PY $CONVERT --src_root "$GT_SEQ_ROOT"   --dst_root "$WORK_ROOT/gt"    --mode gt

echo "===> Converting prediction sequences ..."
$PY $CONVERT --src_root "$PRED_SEQ_ROOT" --dst_root "$WORK_ROOT/gener" --mode det

echo "===> Checking gt.txt and det.txt files ..."

# Check ground truth files
gt_count=$(find "$WORK_ROOT/gt" -name "gt.txt" | wc -l)
empty_gt=$(find "$WORK_ROOT/gt" -name "gt.txt" -size 0 | wc -l)
echo "Found $gt_count gt.txt files in $WORK_ROOT/gt, empty files: $empty_gt"
if [ "$gt_count" -ne 100 ]; then
    echo "Warning: gt.txt file count is not 100 (actual count: $gt_count)"
    missing_gt=$((100 - gt_count))
    echo "Missing $missing_gt gt.txt files"
fi
if [ "$empty_gt" -gt 0 ]; then
    echo "Warning: Found $empty_gt empty gt.txt files"
    find "$WORK_ROOT/gt" -name "gt.txt" -size 0 -exec ls -l {} \;
fi

# Check detection files
det_count=$(find "$WORK_ROOT/gener" -name "det.txt" | wc -l)
empty_det=$(find "$WORK_ROOT/gener" -name "det.txt" -size 0 | wc -l)
echo "Found $det_count det.txt files in $WORK_ROOT/gener, empty files: $empty_det"
if [ "$det_count" -ne 100 ]; then
    echo "Warning: det.txt file count is not 100 (actual count: $det_count)"
    missing_det=$((100 - det_count))
    echo "Missing $missing_det det.txt files"
fi
if [ "$empty_det" -gt 0 ]; then
    echo "Warning: Found $empty_det empty det.txt files"
    find "$WORK_ROOT/gener" -name "det.txt" -size 0 -exec ls -l {} \;
fi

# Check if files are valid before proceeding
if [ "$gt_count" -ne 100 ] || [ "$empty_gt" -gt 0 ] || [ "$det_count" -ne 100 ] || [ "$empty_det" -gt 0 ]; then
    read -p "File issues detected, continue execution? (y/n) " -n 1 -r
    echo
    if [[ ! $REPLY =~ ^[Yy]$ ]]; then
        echo "Execution terminated"
        exit 1
    fi
else
    echo "File check passed: Found 100 non-empty gt.txt and det.txt files"
fi

echo "===> 1.5 Generate seqmap ..."
SEQMAP_DIR="$WORK_ROOT/gt/seqmaps"
mkdir -p "$SEQMAP_DIR"
ls -1 "$WORK_ROOT/gt" | grep -v seqmaps > "$SEQMAP_DIR/custom-train.txt"

echo "===> 1.6 Copy missing seqinfo.ini (tracker side) ..."
for seq in $(ls -1 "$WORK_ROOT/gt" | grep -v seqmaps); do
    src="$WORK_ROOT/gt/$seq/seqinfo.ini"
    dst="$WORK_ROOT/gener/$seq/seqinfo.ini"
    [ -f "$src" ] && [ ! -f "$dst" ] && { mkdir -p "$(dirname "$dst")"; cp "$src" "$dst"; }
done

echo "===> 1.7 Ensure flat tracker files exist ..."
for seq in $(ls -1 "$WORK_ROOT/gener" | grep -v '\.txt$'); do
    [ -f "$WORK_ROOT/gener/$seq/det/det.txt" ] && \
    cp -n "$WORK_ROOT/gener/$seq/det/det.txt" "$WORK_ROOT/gener/${seq}.txt"
done

echo "===> Check and handle missing tracker files ..."

# Get all ground truth sequences
gt_sequences=$(ls -1 "$WORK_ROOT/gt" | grep -v seqmaps)

# Check which sequences have tracker files
existing_sequences=()
missing_sequences=()

for seq in $gt_sequences; do
    # Check if tracker file exists in any of the expected locations
    if [ -f "$WORK_ROOT/gener/${seq}.txt" ] || \
       [ -f "$WORK_ROOT/gener/${seq}/det.txt" ] || \
       [ -f "$WORK_ROOT/gener/${seq}/det/det.txt" ]; then
        existing_sequences+=("$seq")
        echo "Found tracker file: $seq"
    else
        missing_sequences+=("$seq")
        echo "Missing tracker file: $seq"
    fi
done

echo "Total sequences: $(echo $gt_sequences | wc -w)"
echo "Sequences with tracker: ${#existing_sequences[@]}"
echo "Missing tracker sequences: ${#missing_sequences[@]}"

# Create seqmap for available sequences
SEQMAP_DIR="$WORK_ROOT/gt/seqmaps"
mkdir -p "$SEQMAP_DIR"

if [ ${#existing_sequences[@]} -gt 0 ]; then
    printf "%s\n" "${existing_sequences[@]}" > "$SEQMAP_DIR/available-train.txt"
    echo "Created seqmap with ${#existing_sequences[@]} sequences: available-train.txt"
    SEQMAP_FILE="available-train.txt"
else
    echo "Error: No available tracker files found!"
    exit 1
fi

# Report missing sequences
if [ ${#missing_sequences[@]} -gt 0 ]; then
    echo "The following sequences will be skipped:"
    printf "  %s\n" "${missing_sequences[@]}"
fi

echo "===> Create empty tracker files for missing sequences ..."

# Get all ground truth sequences
gt_sequences=$(ls -1 "$WORK_ROOT/gt" | grep -v seqmaps)

for seq in $gt_sequences; do
    # If no tracker file exists for this sequence
    if [ ! -f "$WORK_ROOT/gener/${seq}.txt" ] && \
       [ ! -f "$WORK_ROOT/gener/${seq}/det.txt" ] && \
       [ ! -f "$WORK_ROOT/gener/${seq}/det/det.txt" ]; then
        
        echo "Creating empty tracker file for sequence $seq..."
        
        # Create sequence directory
        mkdir -p "$WORK_ROOT/gener/$seq"
        
        # Create empty det.txt
        touch "$WORK_ROOT/gener/$seq/det.txt"
        
        # Copy seqinfo.ini if exists
        if [ -f "$WORK_ROOT/gt/$seq/seqinfo.ini" ]; then
            cp "$WORK_ROOT/gt/$seq/seqinfo.ini" "$WORK_ROOT/gener/$seq/seqinfo.ini"
        fi
        
        # Create flat file as well
        touch "$WORK_ROOT/gener/${seq}.txt"
    fi
done

echo "Created empty tracker files for all missing sequences"

echo "===> 2. Run TrackEval ..."

# Run the evaluation
python TrackEval/scripts/run_mot_challenge.py \
    --BENCHMARK          custom \
    --GT_FOLDER          "$WORK_ROOT/gt" \
    --TRACKERS_FOLDER    "$WORK_ROOT" \
    --TRACKERS_TO_EVAL   gener \
    --TRACKER_SUB_FOLDER "" \
    --SKIP_SPLIT_FOL     True \
    --DO_PREPROC         False \
    --METRICS            HOTA CLEAR Identity \
    --USE_PARALLEL       True \
    --PRINT_ONLY_COMBINED True \
    --BREAK_ON_ERROR     True