#!/usr/bin/env bash

set -e

# Check if model name is provided
model_name=$1
if [ -z "$model_name" ]; then
    echo "Error: Please provide a model name as argument"
    echo "Usage: $0 <model_name>"
    exit 1
fi

# Configuration - Modify these paths according to your setup
PRED_ROOT="/path/to/project/eval/evaldata/$model_name/datapipeoutput"
GT_ROOT="/path/to/project/eval/evaldata/GT/datapipe-eval-100"
# Alternative GT path (uncomment if needed)
# GT_ROOT=/path/to/project/eval/evaldata/ablation_GT/datapipeoutput

OUT_DIR="/path/to/project/eval/extra_metrics"

# Clean previous results
rm -rf "$OUT_DIR/${model_name}"

# Create output directory
mkdir -p "$OUT_DIR/${model_name}"

# Multi-GPU configuration
NUM_WORKERS=96
GPUS="0,1,2,3,4,5,6,7"
BATCH_SIZE=100
MAX_TIME=7200  # Maximum evaluation time in seconds

echo "=================================="
echo "Multi-GPU Interaction Evaluation"
echo "=================================="
echo "Model: $model_name"
echo "GPUs: $GPUS"
echo "CPU Workers: $NUM_WORKERS"
echo "Batch Size: $BATCH_SIZE"
echo "Max Time: ${MAX_TIME}s"
echo "=================================="

# Run multi-GPU interaction coherence evaluation
python /path/to/project/eval/src/track2/eval_interaction_coherence_mgpu.py \
    --pred_root "$PRED_ROOT" \
    --gt_root "$GT_ROOT" \
    --out_csv "$OUT_DIR/${model_name}/interaction_metrics.csv" \
    --num_workers $NUM_WORKERS \
    --gpus $GPUS \
    --batch_size $BATCH_SIZE \
    --max_time $MAX_TIME \
    --resume \
    --watchdog

echo "=================================="
echo "Evaluation Complete!"
echo "Results saved to: $OUT_DIR/${model_name}/interaction_metrics.csv"
echo "=================================="