#!/bin/bash

# Color definitions for terminal output
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m'

echo -e "${GREEN}=== Optimized Video Quality Evaluation Pipeline (8xA800 GPU) ===${NC}"

# Directory configuration
BASE_DIR="path_to_evaluation_project"
WEIGHTS_DIR="path_to_downloaded_models" 
GT_ROOT="path_to_ground_truth_data/eval-100"

# Alternative GT root for ablation studies
GT_ROOT="path_to_ablation_ground_truth/output"

# Create timestamped output directory
OUTPUT_DIR="$BASE_DIR/results/optimized_$(date +%Y%m%d_%H%M%S)"

# Create output directory
mkdir -p $OUTPUT_DIR
echo -e "\nOutput directory: $OUTPUT_DIR"

# Step 1: Generate evaluation list
echo -e "\n${YELLOW}1. Generating evaluation list...${NC}"
EVAL_LIST="$OUTPUT_DIR/eval_list.txt"

if [ -f "$BASE_DIR/src/track3/generate_eval_list_multi.py" ]; then
    echo -e "Using evaluation list generator"
    python $BASE_DIR/src/track3/generate_eval_list_multi.py \
        --gt_root $GT_ROOT \
        --output $EVAL_LIST
else
    echo -e "${RED}Error: Cannot find evaluation list generation script${NC}"
    exit 1
fi

# Verify evaluation list was created successfully
if [ ! -f "$EVAL_LIST" ] || [ ! -s "$EVAL_LIST" ]; then
    echo -e "${RED}Error: Evaluation list generation failed or is empty${NC}"
    exit 1
fi

VIDEO_COUNT=$(wc -l < $EVAL_LIST)
echo -e "Found ${VIDEO_COUNT} videos in evaluation list"

# Step 2: Configure baseline methods
echo -e "\n${YELLOW}2. Configuring baseline methods...${NC}"

# Get model name from command line argument
model_name=$1

# Define baseline configurations
BASELINE_ROOTS=(
    "$model_name"
)
BASELINE_NAMES=(
    "$model_name"
)

# Verify baseline directories exist
for i in "${!BASELINE_ROOTS[@]}"; do
    if [ ! -d "${BASELINE_ROOTS[$i]}" ]; then
        echo -e "${RED}Error: Baseline directory does not exist ${BASELINE_ROOTS[$i]}${NC}"
        exit 1
    else
        VIDEO_COUNT=$(find "${BASELINE_ROOTS[$i]}" -type d -name "processed_*" | wc -l)
        echo -e "✓ ${BASELINE_NAMES[$i]}: ${BASELINE_ROOTS[$i]} (contains ${VIDEO_COUNT} videos)"
    fi
done

# Step 3: Configure GPU environment
echo -e "\n${YELLOW}3. Configuring GPU environment...${NC}"

# Detect available GPUs
GPU_COUNT=$(nvidia-smi -L | wc -l)
echo -e "Detected ${GPU_COUNT} GPUs"

# Set environment variables for optimal performance
export OMP_NUM_THREADS=6
export MKL_NUM_THREADS=6
export PYTORCH_CUDA_ALLOC_CONF=max_split_size_mb:512
export CUDA_VISIBLE_DEVICES=0,1,2,3,4,5,6,7
export CUDA_DEVICE_ORDER=PCI_BUS_ID

# Define script paths
ORIGINAL_SCRIPT="path_to_evaluation_scripts/video_evaluation_batch_multi_ind_vis_plus_CLIP.py"
OPTIMIZED_SCRIPT="path_to_evaluation_scripts/video_evaluation_batch_multi_ind_vis_plus_CLIP-opt.py"

# Create optimized script if it doesn't exist
if [ ! -f "$OPTIMIZED_SCRIPT" ]; then
    echo -e "${YELLOW}Optimized script does not exist, copying from original script...${NC}"
    cp "$ORIGINAL_SCRIPT" "$OPTIMIZED_SCRIPT"
fi

# Step 4: Start optimized evaluation
echo -e "\n${YELLOW}4. Starting optimized evaluation...${NC}"
start_time=$(date +%s)

# Note: Download required model weights from:
# - Vision models: https://download.pytorch.org/models/ 
# - CLIP models: https://github.com/openai/CLIP

python $OPTIMIZED_SCRIPT \
    --gt_list "${EVAL_LIST}" \
    --baseline_roots "${BASELINE_ROOTS[@]}" \
    --baseline_names "${BASELINE_NAMES[@]}" \
    --output "${OUTPUT_DIR}/results.json" \
    --output_table "${OUTPUT_DIR}/results_full_frame.csv" \
    --output_masked_table "${OUTPUT_DIR}/results_masked_region.csv" \
    --num_gpus 8 \
    --processes_per_gpu 4 \
    --batch_size 64 \
    --prefetch_size 64 \
    --cpu_workers 96 \
    --weights_dir "$WEIGHTS_DIR" \
    --use_mixed_precision \
    --io_threads 48 \
    2>&1 | tee "${OUTPUT_DIR}/evaluation.log"

# Check exit status
exit_code=${PIPESTATUS[0]}
if [ $exit_code -ne 0 ]; then
    echo -e "${RED}Error: Evaluation process failed with exit code: $exit_code${NC}"
    echo -e "${YELLOW}Please check the log file: ${OUTPUT_DIR}/evaluation.log${NC}"
    exit $exit_code
fi

# Calculate runtime
end_time=$(date +%s)
runtime=$((end_time-start_time))

# Display results
echo -e "\n${GREEN}=== Evaluation Completed ===${NC}"
echo -e "Total runtime: $runtime seconds ($(($runtime / 60)) minutes $(($runtime % 60)) seconds)"
echo -e "\nResult file locations:"
echo -e "- Detailed results JSON: ${OUTPUT_DIR}/results.json"
echo -e "- Full frame evaluation table: ${OUTPUT_DIR}/results_full_frame.csv"
echo -e "- Masked region evaluation table: ${OUTPUT_DIR}/results_masked_region.csv"
echo -e "- Evaluation log: ${OUTPUT_DIR}/evaluation.log"

echo -e "\n${YELLOW}GPU utilization statistics:${NC}"
nvidia-smi --query-gpu=utilization.gpu,utilization.memory,memory.used,memory.total --format=csv