import json
import math
from dataclasses import dataclass
from typing import List, Optional, Tuple, Union

import numpy as np
import cv2
import torch
import inspect
from diffusers.configuration_utils import ConfigMixin, register_to_config
from diffusers.utils import BaseOutput, logging
from diffusers.utils.torch_utils import randn_tensor
from diffusers.schedulers.scheduling_utils import KarrasDiffusionSchedulers, SchedulerMixin
from diffusers.utils.torch_utils import is_compiled_module
import torch.nn.functional as F

logger = logging.get_logger(__name__)


def denormalize(images: Union[np.ndarray, torch.Tensor]) -> Union[np.ndarray, torch.Tensor]:
    return (images / 2 + 0.5).clamp(0, 1)

def postprocess_scheduler(image: torch.Tensor):

    if not isinstance(image, torch.Tensor):
        raise ValueError(
            f"Input for postprocessing is in incorrect format: {type(image)}. We only support pytorch tensor"
        )
    image = (image / 2 + 0.5).clamp(0, 1).float()

    image = (image * 255).clamp(0, 255)
    return image

def decode_latents_scheduler(latents, num_frames, decode_chunk_size=8, vae=None, device=None):

    latents = latents.flatten(0, 1)
    latents = 1 / vae.config.scaling_factor * latents

    forward_vae_fn = vae._orig_mod.forward if is_compiled_module(vae) else vae.forward
    accepts_num_frames = "num_frames" in set(inspect.signature(forward_vae_fn).parameters.keys())


    frames = []


    vae.decoder.cpu()

    for i in range(0, latents.shape[0], decode_chunk_size):
        num_frames_in = latents[i: i + decode_chunk_size].shape[0]
        decode_kwargs = {}
        if accepts_num_frames:
            decode_kwargs["num_frames"] = num_frames_in
        frame = vae.decode(latents[i: i + decode_chunk_size], **decode_kwargs).sample


        frames.append(frame.cpu())


    frames = torch.cat(frames, dim=0)


    frames = frames.reshape(-1, num_frames, *frames.shape[1:]).permute(0, 2, 1, 3, 4)

    frames = frames.float()
    return frames

def tensor2vid_scheduler(video, processor, output_type="np"):
    batch_size, channels, num_frames, height, width = video.shape
    outputs = []
    for batch_idx in range(batch_size):
        batch_vid = video[batch_idx].permute(1, 0, 2, 3)
        batch_output = processor.postprocess(batch_vid, output_type)
        outputs.append(batch_output)
    return outputs

def tensor2vid_scheduler_new(video, processor, output_type="np"):
    batch_size, channels, num_frames, height, width = video.shape
    video = video.squeeze(0)
    video = video.permute(1, 0, 2, 3)
    outputs = postprocess_scheduler(video)
    return outputs

def decode_latents_scheduler_new(latents, num_frames, decode_chunk_size=8, vae=None, device=None):

    latents = latents.flatten(0, 1)
    latents = 1 / vae.config.scaling_factor * latents
    forward_vae_fn = vae._orig_mod.forward if is_compiled_module(vae) else vae.forward
    accepts_num_frames = "num_frames" in set(inspect.signature(forward_vae_fn).parameters.keys())


    frames = []

    for i in range(0, latents.shape[0], decode_chunk_size):
        num_frames_in = latents[i: i + decode_chunk_size].shape[0]
        decode_kwargs = {}
        if accepts_num_frames:
            decode_kwargs["num_frames"] = num_frames_in
        frame = vae.decode(latents[i: i + decode_chunk_size], **decode_kwargs).sample
        frames.append(frame)

    frames = torch.cat(frames, dim=0)


    frames = frames.reshape(-1, num_frames, *frames.shape[1:]).permute(0, 2, 1, 3, 4)

    frames = frames.float()
    return frames

@dataclass

class EulerDiscreteSchedulerProOutput(BaseOutput):
    """
    Output class for the scheduler's `step` function output.

    Args:
        prev_sample (`torch.Tensor` of shape `(batch_size, num_channels, height, width)` for images):
            Computed sample `(x_{t-1})` of previous timestep. `prev_sample` should be used as next model input in the
            denoising loop.
        pred_original_sample (`torch.Tensor` of shape `(batch_size, num_channels, height, width)` for images):
            The predicted denoised sample `(x_{0})` based on the model output from the current timestep.
            `pred_original_sample` can be used to preview progress or for guidance.
    """

    prev_sample: torch.Tensor
    pred_original_sample: Optional[torch.Tensor] = None


def betas_for_alpha_bar(
    num_diffusion_timesteps,
    max_beta=0.999,
    alpha_transform_type="cosine",
):
    """
    Create a beta schedule that discretizes the given alpha_t_bar function, which defines the cumulative product of
    (1-beta) over time from t = [0,1].

    Contains a function alpha_bar that takes an argument t and transforms it to the cumulative product of (1-beta) up
    to that part of the diffusion process.


    Args:
        num_diffusion_timesteps (`int`): the number of betas to produce.
        max_beta (`float`): the maximum beta to use; use values lower than 1 to
                     prevent singularities.
        alpha_transform_type (`str`, *optional*, default to `cosine`): the type of noise schedule for alpha_bar.
                     Choose from `cosine` or `exp`

    Returns:
        betas (`np.ndarray`): the betas used by the scheduler to step the model outputs
    """
    if alpha_transform_type == "cosine":

        def alpha_bar_fn(t):
            return math.cos((t + 0.008) / 1.008 * math.pi / 2) ** 2

    elif alpha_transform_type == "exp":

        def alpha_bar_fn(t):
            return math.exp(t * -12.0)

    else:
        raise ValueError(f"Unsupported alpha_transform_type: {alpha_transform_type}")

    betas = []
    for i in range(num_diffusion_timesteps):
        t1 = i / num_diffusion_timesteps
        t2 = (i + 1) / num_diffusion_timesteps
        betas.append(min(1 - alpha_bar_fn(t2) / alpha_bar_fn(t1), max_beta))
    return torch.tensor(betas, dtype=torch.float32)


def rescale_zero_terminal_snr(betas):
    """
    Rescales betas to have zero terminal SNR Based on https://arxiv.org/pdf/2305.08891.pdf (Algorithm 1)


    Args:
        betas (`torch.Tensor`):
            the betas that the scheduler is being initialized with.

    Returns:
        `torch.Tensor`: rescaled betas with zero terminal SNR
    """

    alphas = 1.0 - betas
    alphas_cumprod = torch.cumprod(alphas, dim=0)
    alphas_bar_sqrt = alphas_cumprod.sqrt()


    alphas_bar_sqrt_0 = alphas_bar_sqrt[0].clone()
    alphas_bar_sqrt_T = alphas_bar_sqrt[-1].clone()


    alphas_bar_sqrt -= alphas_bar_sqrt_T


    alphas_bar_sqrt *= alphas_bar_sqrt_0 / (alphas_bar_sqrt_0 - alphas_bar_sqrt_T)


    alphas_bar = alphas_bar_sqrt**2
    alphas = alphas_bar[1:] / alphas_bar[:-1]
    alphas = torch.cat([alphas_bar[0:1], alphas])
    betas = 1 - alphas

    return betas


class EulerDiscreteSchedulerPro(SchedulerMixin, ConfigMixin):
    """
    Euler scheduler.

    This model inherits from [`SchedulerMixin`] and [`ConfigMixin`]. Check the superclass documentation for the generic
    methods the library implements for all schedulers such as loading and saving.

    Args:
        num_train_timesteps (`int`, defaults to 1000):
            The number of diffusion steps to train the model.
        beta_start (`float`, defaults to 0.0001):
            The starting `beta` value of inference.
        beta_end (`float`, defaults to 0.02):
            The final `beta` value.
        beta_schedule (`str`, defaults to `"linear"`):
            The beta schedule, a mapping from a beta range to a sequence of betas for stepping the model. Choose from
            `linear` or `scaled_linear`.
        trained_betas (`np.ndarray`, *optional*):
            Pass an array of betas directly to the constructor to bypass `beta_start` and `beta_end`.
        prediction_type (`str`, defaults to `epsilon`, *optional*):
            Prediction type of the scheduler function; can be `epsilon` (predicts the noise of the diffusion process),
            `sample` (directly predicts the noisy sample`) or `v_prediction` (see section 2.4 of [Imagen
            Video](https://imagen.research.google/video/paper.pdf) paper).
        interpolation_type(`str`, defaults to `"linear"`, *optional*):
            The interpolation type to compute intermediate sigmas for the scheduler denoising steps. Should be on of
            `"linear"` or `"log_linear"`.
        use_karras_sigmas (`bool`, *optional*, defaults to `False`):
            Whether to use Karras sigmas for step sizes in the noise schedule during the sampling process. If `True`,
            the sigmas are determined according to a sequence of noise levels {σi}.
        timestep_spacing (`str`, defaults to `"linspace"`):
            The way the timesteps should be scaled. Refer to Table 2 of the [Common Diffusion Noise Schedules and
            Sample Steps are Flawed](https://huggingface.co/papers/2305.08891) for more information.
        steps_offset (`int`, defaults to 0):
            An offset added to the inference steps, as required by some model families.
        rescale_betas_zero_snr (`bool`, defaults to `False`):
            Whether to rescale the betas to have zero terminal SNR. This enables the model to generate very bright and
            dark samples instead of limiting it to samples with medium brightness. Loosely related to
            [`--offset_noise`](https://github.com/huggingface/diffusers/blob/74fd735eb073eb1d774b1ab4154a0876eb82f055/examples/dreambooth/train_dreambooth.py
        final_sigmas_type (`str`, defaults to `"zero"`):
            The final `sigma` value for the noise schedule during the sampling process. If `"sigma_min"`, the final
            sigma is the same as the last sigma in the training schedule. If `zero`, the final sigma is set to 0.
    """

    _compatibles = [e.name for e in KarrasDiffusionSchedulers]
    order = 1

    @register_to_config
    def __init__(
        self,
        num_train_timesteps: int = 1000,
        beta_start: float = 0.0001,
        beta_end: float = 0.02,
        beta_schedule: str = "linear",
        trained_betas: Optional[Union[np.ndarray, List[float]]] = None,
        prediction_type: str = "epsilon",
        interpolation_type: str = "linear",
        use_karras_sigmas: Optional[bool] = False,
        sigma_min: Optional[float] = None,
        sigma_max: Optional[float] = None,
        timestep_spacing: str = "linspace",
        timestep_type: str = "discrete",
        steps_offset: int = 0,
        rescale_betas_zero_snr: bool = False,
        final_sigmas_type: str = "zero",
    ):
        if trained_betas is not None:
            self.betas = torch.tensor(trained_betas, dtype=torch.float32)
        elif beta_schedule == "linear":
            self.betas = torch.linspace(beta_start, beta_end, num_train_timesteps, dtype=torch.float32)
        elif beta_schedule == "scaled_linear":

            self.betas = torch.linspace(beta_start**0.5, beta_end**0.5, num_train_timesteps, dtype=torch.float32) ** 2
        elif beta_schedule == "squaredcos_cap_v2":

            self.betas = betas_for_alpha_bar(num_train_timesteps)
        else:
            raise NotImplementedError(f"{beta_schedule} is not implemented for {self.__class__}")

        if rescale_betas_zero_snr:
            self.betas = rescale_zero_terminal_snr(self.betas)

        self.alphas = 1.0 - self.betas
        self.alphas_cumprod = torch.cumprod(self.alphas, dim=0)

        if rescale_betas_zero_snr:


            self.alphas_cumprod[-1] = 2**-24

        sigmas = (((1 - self.alphas_cumprod) / self.alphas_cumprod) ** 0.5).flip(0)
        timesteps = np.linspace(0, num_train_timesteps - 1, num_train_timesteps, dtype=float)[::-1].copy()
        timesteps = torch.from_numpy(timesteps).to(dtype=torch.float32)


        self.num_inference_steps = None


        if timestep_type == "continuous" and prediction_type == "v_prediction":
            self.timesteps = torch.Tensor([0.25 * sigma.log() for sigma in sigmas])
        else:
            self.timesteps = timesteps

        self.sigmas = torch.cat([sigmas, torch.zeros(1, device=sigmas.device)])

        self.is_scale_input_called = False
        self.use_karras_sigmas = use_karras_sigmas

        self._step_index = None
        self._begin_index = None
        self.sigmas = self.sigmas.to("cpu")


    def init_face_model(self, app = None, face_helper = None, handler_ante = None, face_embedding_extractor = None):
        self.app = app
        self.face_helper = face_helper
        self.handler_ante = handler_ante
        self.face_embedding_extractor = face_embedding_extractor

    @property
    def init_noise_sigma(self):

        max_sigma = max(self.sigmas) if isinstance(self.sigmas, list) else self.sigmas.max()
        if self.config.timestep_spacing in ["linspace", "trailing"]:
            return max_sigma

        return (max_sigma**2 + 1) ** 0.5

    @property
    def step_index(self):
        """
        The index counter for current timestep. It will increase 1 after each scheduler step.
        """
        return self._step_index

    @property
    def begin_index(self):
        """
        The index for the first timestep. It should be set from pipeline with `set_begin_index` method.
        """
        return self._begin_index


    def set_begin_index(self, begin_index: int = 0):
        """
        Sets the begin index for the scheduler. This function should be run from pipeline before the inference.

        Args:
            begin_index (`int`):
                The begin index for the scheduler.
        """
        self._begin_index = begin_index

    def scale_model_input(self, sample: torch.Tensor, timestep: Union[float, torch.Tensor]) -> torch.Tensor:
        """
        Ensures interchangeability with schedulers that need to scale the denoising model input depending on the
        current timestep. Scales the denoising model input by `(sigma**2 + 1) ** 0.5` to match the Euler algorithm.

        Args:
            sample (`torch.Tensor`):
                The input sample.
            timestep (`int`, *optional*):
                The current timestep in the diffusion chain.

        Returns:
            `torch.Tensor`:
                A scaled input sample.
        """
        if self.step_index is None:
            self._init_step_index(timestep)

        sigma = self.sigmas[self.step_index]
        sample = sample / ((sigma**2 + 1) ** 0.5)

        self.is_scale_input_called = True
        return sample

    def set_timesteps(
        self,
        num_inference_steps: int = None,
        device: Union[str, torch.device] = None,
        timesteps: Optional[List[int]] = None,
        sigmas: Optional[List[float]] = None,
    ):
        """
        Sets the discrete timesteps used for the diffusion chain (to be run before inference).

        Args:
            num_inference_steps (`int`):
                The number of diffusion steps used when generating samples with a pre-trained model.
            device (`str` or `torch.device`, *optional*):
                The device to which the timesteps should be moved to. If `None`, the timesteps are not moved.
            timesteps (`List[int]`, *optional*):
                Custom timesteps used to support arbitrary timesteps schedule. If `None`, timesteps will be generated
                based on the `timestep_spacing` attribute. If `timesteps` is passed, `num_inference_steps` and `sigmas`
                must be `None`, and `timestep_spacing` attribute will be ignored.
            sigmas (`List[float]`, *optional*):
                Custom sigmas used to support arbitrary timesteps schedule schedule. If `None`, timesteps and sigmas
                will be generated based on the relevant scheduler attributes. If `sigmas` is passed,
                `num_inference_steps` and `timesteps` must be `None`, and the timesteps will be generated based on the
                custom sigmas schedule.
        """

        if timesteps is not None and sigmas is not None:
            raise ValueError("Only one of `timesteps` or `sigmas` should be set.")
        if num_inference_steps is None and timesteps is None and sigmas is None:
            raise ValueError("Must pass exactly one of `num_inference_steps` or `timesteps` or `sigmas.")
        if num_inference_steps is not None and (timesteps is not None or sigmas is not None):
            raise ValueError("Can only pass one of `num_inference_steps` or `timesteps` or `sigmas`.")
        if timesteps is not None and self.config.use_karras_sigmas:
            raise ValueError("Cannot set `timesteps` with `config.use_karras_sigmas = True`.")
        if (
            timesteps is not None
            and self.config.timestep_type == "continuous"
            and self.config.prediction_type == "v_prediction"
        ):
            raise ValueError(
                "Cannot set `timesteps` with `config.timestep_type = 'continuous'` and `config.prediction_type = 'v_prediction'`."
            )

        if num_inference_steps is None:
            num_inference_steps = len(timesteps) if timesteps is not None else len(sigmas) - 1
        self.num_inference_steps = num_inference_steps

        if sigmas is not None:
            log_sigmas = np.log(np.array(((1 - self.alphas_cumprod) / self.alphas_cumprod) ** 0.5))
            sigmas = np.array(sigmas).astype(np.float32)
            timesteps = np.array([self._sigma_to_t(sigma, log_sigmas) for sigma in sigmas[:-1]])

        else:
            if timesteps is not None:
                timesteps = np.array(timesteps).astype(np.float32)
            else:

                if self.config.timestep_spacing == "linspace":
                    timesteps = np.linspace(
                        0, self.config.num_train_timesteps - 1, num_inference_steps, dtype=np.float32
                    )[::-1].copy()
                elif self.config.timestep_spacing == "leading":
                    step_ratio = self.config.num_train_timesteps // self.num_inference_steps


                    timesteps = (
                        (np.arange(0, num_inference_steps) * step_ratio).round()[::-1].copy().astype(np.float32)
                    )
                    timesteps += self.config.steps_offset
                elif self.config.timestep_spacing == "trailing":
                    step_ratio = self.config.num_train_timesteps / self.num_inference_steps


                    timesteps = (
                        (np.arange(self.config.num_train_timesteps, 0, -step_ratio)).round().copy().astype(np.float32)
                    )
                    timesteps -= 1
                else:
                    raise ValueError(
                        f"{self.config.timestep_spacing} is not supported. Please make sure to choose one of 'linspace', 'leading' or 'trailing'."
                    )

            sigmas = np.array(((1 - self.alphas_cumprod) / self.alphas_cumprod) ** 0.5)
            log_sigmas = np.log(sigmas)
            if self.config.interpolation_type == "linear":
                sigmas = np.interp(timesteps, np.arange(0, len(sigmas)), sigmas)
            elif self.config.interpolation_type == "log_linear":
                sigmas = torch.linspace(np.log(sigmas[-1]), np.log(sigmas[0]), num_inference_steps + 1).exp().numpy()
            else:
                raise ValueError(
                    f"{self.config.interpolation_type} is not implemented. Please specify interpolation_type to either"
                    " 'linear' or 'log_linear'"
                )

            if self.config.use_karras_sigmas:
                sigmas = self._convert_to_karras(in_sigmas=sigmas, num_inference_steps=self.num_inference_steps)
                timesteps = np.array([self._sigma_to_t(sigma, log_sigmas) for sigma in sigmas])

            if self.config.final_sigmas_type == "sigma_min":
                sigma_last = ((1 - self.alphas_cumprod[0]) / self.alphas_cumprod[0]) ** 0.5
            elif self.config.final_sigmas_type == "zero":
                sigma_last = 0
            else:
                raise ValueError(
                    f"`final_sigmas_type` must be one of 'zero', or 'sigma_min', but got {self.config.final_sigmas_type}"
                )

            sigmas = np.concatenate([sigmas, [sigma_last]]).astype(np.float32)

        sigmas = torch.from_numpy(sigmas).to(dtype=torch.float32, device=device)


        if self.config.timestep_type == "continuous" and self.config.prediction_type == "v_prediction":
            self.timesteps = torch.Tensor([0.25 * sigma.log() for sigma in sigmas[:-1]]).to(device=device)
        else:
            self.timesteps = torch.from_numpy(timesteps.astype(np.float32)).to(device=device)

        self._step_index = None
        self._begin_index = None
        self.sigmas = sigmas.to("cpu")

    def _sigma_to_t(self, sigma, log_sigmas):

        log_sigma = np.log(np.maximum(sigma, 1e-10))


        dists = log_sigma - log_sigmas[:, np.newaxis]


        low_idx = np.cumsum((dists >= 0), axis=0).argmax(axis=0).clip(max=log_sigmas.shape[0] - 2)
        high_idx = low_idx + 1

        low = log_sigmas[low_idx]
        high = log_sigmas[high_idx]


        w = (low - log_sigma) / (low - high)
        w = np.clip(w, 0, 1)


        t = (1 - w) * low_idx + w * high_idx
        t = t.reshape(sigma.shape)
        return t


    def _convert_to_karras(self, in_sigmas: torch.Tensor, num_inference_steps) -> torch.Tensor:
        """Constructs the noise schedule of Karras et al. (2022)."""


        if hasattr(self.config, "sigma_min"):
            sigma_min = self.config.sigma_min
        else:
            sigma_min = None

        if hasattr(self.config, "sigma_max"):
            sigma_max = self.config.sigma_max
        else:
            sigma_max = None

        sigma_min = sigma_min if sigma_min is not None else in_sigmas[-1].item()
        sigma_max = sigma_max if sigma_max is not None else in_sigmas[0].item()

        rho = 7.0
        ramp = np.linspace(0, 1, num_inference_steps)
        min_inv_rho = sigma_min ** (1 / rho)
        max_inv_rho = sigma_max ** (1 / rho)
        sigmas = (max_inv_rho + ramp * (min_inv_rho - max_inv_rho)) ** rho
        return sigmas

    def index_for_timestep(self, timestep, schedule_timesteps=None):
        if schedule_timesteps is None:
            schedule_timesteps = self.timesteps

        indices = (schedule_timesteps == timestep).nonzero()


        pos = 1 if len(indices) > 1 else 0

        return indices[pos].item()

    def _init_step_index(self, timestep):
        if self.begin_index is None:
            if isinstance(timestep, torch.Tensor):
                timestep = timestep.to(self.timesteps.device)
            self._step_index = self.index_for_timestep(timestep)
        else:
            self._step_index = self._begin_index


    def step(
        self,
        model_output: torch.Tensor,
        timestep: Union[float, torch.Tensor],
        sample: torch.Tensor,
        s_churn: float = 0.0,
        s_tmin: float = 0.0,
        s_tmax: float = float("inf"),
        s_noise: float = 1.0,
        generator: Optional[torch.Generator] = None,
        return_dict: bool = True,
        refine_face=False,
        learning_rate=None,
        num_optimization_iter=3,
        refine_reference_face_embeddings0=None,
        refine_reference_face_embeddings1=None,
        vae=None,
        decode_chunk_size=4,
        image_processor=None,
        output_type=None,
        device=None,
        target_face_mask_list0=None,
        target_face_mask_list1=None,
        noise_aug_strength=None,
    ) -> Union[EulerDiscreteSchedulerProOutput, Tuple]:
        """
        Predict the sample from the previous timestep by reversing the SDE. This function propagates the diffusion
        process from the learned model outputs (most often the predicted noise).
        """

        if isinstance(timestep, (int, torch.IntTensor, torch.LongTensor)):
            raise ValueError(
                (
                    "Passing integer indices (e.g. from `enumerate(timesteps)`) as timesteps to"
                    " `EulerDiscreteScheduler.step()` is not supported. Make sure to pass"
                    " one of the `scheduler.timesteps` as a timestep."
                ),
            )

        if not self.is_scale_input_called:
            logger.warning(
                "The `scale_model_input` function should be called before `step` to ensure correct denoising. "
                "See `StableDiffusionPipeline` for a usage example."
            )

        if self.step_index is None:
            self._init_step_index(timestep)


        sample = sample.to(torch.float32)

        sigma = self.sigmas[self.step_index]

        gamma = min(s_churn / (len(self.sigmas) - 1), 2**0.5 - 1) if s_tmin <= sigma <= s_tmax else 0.0

        noise = randn_tensor(
            model_output.shape, dtype=model_output.dtype, device=model_output.device, generator=generator
        )

        eps = noise * s_noise
        sigma_hat = sigma * (gamma + 1)

        if gamma > 0:
            sample = sample + eps * (sigma_hat**2 - sigma**2) ** 0.5


        if self.config.prediction_type == "original_sample" or self.config.prediction_type == "sample":
            pred_original_sample = model_output
        elif self.config.prediction_type == "epsilon":
            pred_original_sample = sample - sigma_hat * model_output
        elif self.config.prediction_type == "v_prediction":
            pred_original_sample = model_output * (-sigma / (sigma**2 + 1) ** 0.5) + (sample / (sigma**2 + 1))
        else:
            raise ValueError(
                f"prediction_type given as {self.config.prediction_type} must be one of `epsilon`, or `v_prediction`"
            )


        if refine_face:
            print("DEBUG: Starting face refinement...")
            print(f"DEBUG: refine_reference_face_embeddings0 is not None: {refine_reference_face_embeddings0 is not None}")
            print(f"DEBUG: refine_reference_face_embeddings1 is not None: {refine_reference_face_embeddings1 is not None}")
            print(f"DEBUG: target_face_mask_list0 is not None: {target_face_mask_list0 is not None}")
            print(f"DEBUG: target_face_mask_list1 is not None: {target_face_mask_list1 is not None}")
            
            if refine_reference_face_embeddings0 is not None:
                print(f"DEBUG: refine_reference_face_embeddings0.shape: {refine_reference_face_embeddings0.shape}")
            if refine_reference_face_embeddings1 is not None:
                print(f"DEBUG: refine_reference_face_embeddings1.shape: {refine_reference_face_embeddings1.shape}")
            if target_face_mask_list0 is not None:
                print(f"DEBUG: target_face_mask_list0 length: {len(target_face_mask_list0)}")
                if len(target_face_mask_list0) > 0:
                    print(f"DEBUG: target_face_mask_list0[0]: {target_face_mask_list0[0]}")
            if target_face_mask_list1 is not None:
                print(f"DEBUG: target_face_mask_list1 length: {len(target_face_mask_list1)}")
                if len(target_face_mask_list1) > 0:
                    print(f"DEBUG: target_face_mask_list1[0]: {target_face_mask_list1[0]}")
            
            self.face_embedding_extractor.to(device=device)
            self.face_embedding_extractor.eval()
            

            face_embeddings = []
            target_face_mask_lists = []
            

            if refine_reference_face_embeddings0 is not None and target_face_mask_list0 is not None:
                print("DEBUG: Adding person 0 for refinement")
                org_face_embedding0 = refine_reference_face_embeddings0.repeat(sample.size()[1], 1)
                face_embeddings.append(org_face_embedding0)
                target_face_mask_lists.append(target_face_mask_list0)
                
            if refine_reference_face_embeddings1 is not None and target_face_mask_list1 is not None:
                print("DEBUG: Adding person 1 for refinement")
                org_face_embedding1 = refine_reference_face_embeddings1.repeat(sample.size()[1], 1)
                face_embeddings.append(org_face_embedding1)
                target_face_mask_lists.append(target_face_mask_list1)
            
            print(f"DEBUG: Number of faces to refine: {len(face_embeddings)}")
            
            if not face_embeddings:
                print("Warning: refine_face=True but no valid face embeddings provided")
            else:
                z0 = pred_original_sample.clone().detach()
                optimizer = torch.optim.Adam([z0], lr=learning_rate)
                cosine_loss = torch.nn.CosineSimilarity(dim=1)
                num_frames = sample.size()[1]
                z0.requires_grad = True
                
                print(f"DEBUG: num_frames: {num_frames}")
                print(f"DEBUG: z0.shape: {z0.shape}")
                
                for idx in range(num_optimization_iter):
                    print(f"Current iteration: {idx}")
                    total_face_loss = None
                    

                    pred_frames = decode_latents_scheduler_new(
                        latents=z0, 
                        num_frames=num_frames, 
                        decode_chunk_size=decode_chunk_size, 
                        vae=vae, 
                        device=device
                    )
                    pred_frames = tensor2vid_scheduler_new(pred_frames, processor=image_processor, output_type=output_type)
                    pred_frames = pred_frames.div_(255).sub_(0.5).div_(0.5)
                    
                    print(f"DEBUG: pred_frames.shape: {pred_frames.shape}")
                    print(f"DEBUG: pred_frames type: {type(pred_frames)}")
                    

                    for person_idx, (org_face_embedding, target_face_mask_list) in enumerate(zip(face_embeddings, target_face_mask_lists)):
                        print(f"DEBUG: Processing person {person_idx}")
                        print(f"DEBUG: target_face_mask_list length: {len(target_face_mask_list)}")
                        print(f"DEBUG: org_face_embedding.shape: {org_face_embedding.shape}")
                        
                        pred_cropped_faces = []
                        

                        for k in range(num_frames):
                            print(f"DEBUG: Processing frame {k}/{num_frames}")
                            if k < len(target_face_mask_list):
                                mask_coords = target_face_mask_list[k]
                                print(f"DEBUG: Frame {k} mask_coords: {mask_coords}")
                                
                                if mask_coords is not None and len(mask_coords) >= 4:
                                    target_y_min = int(mask_coords[0])
                                    target_y_max = int(mask_coords[1])
                                    target_x_min = int(mask_coords[2])
                                    target_x_max = int(mask_coords[3])
                                    
                                    print(f"DEBUG: Frame {k} crop coords: y_min={target_y_min}, y_max={target_y_max}, x_min={target_x_min}, x_max={target_x_max}")
                                    

                                    if (target_y_max > target_y_min and target_x_max > target_x_min and 
                                        target_y_min >= 0 and target_x_min >= 0 and
                                        target_y_max < pred_frames.shape[2] and target_x_max < pred_frames.shape[3]):
                                        
                                        pred_frame = pred_frames[k][:, target_y_min:target_y_max + 1, target_x_min:target_x_max + 1]
                                        pred_frame = pred_frame.unsqueeze(0)
                                        pred_frame = F.interpolate(pred_frame, size=(112, 112), mode='bilinear', align_corners=False)
                                        pred_cropped_faces.append(pred_frame)
                                        print(f"DEBUG: Frame {k} successfully cropped, shape: {pred_frame.shape}")
                                    else:
                                        print(f"DEBUG: Frame {k} invalid coordinates: y_min={target_y_min}, y_max={target_y_max}, x_min={target_x_min}, x_max={target_x_max}")
                                        print(f"DEBUG: pred_frames.shape: {pred_frames.shape}")
                                else:
                                    print(f"DEBUG: Frame {k} invalid mask_coords: {mask_coords}")
                            else:
                                print(f"DEBUG: Frame {k} >= target_face_mask_list length ({len(target_face_mask_list)})")
                        
                        print(f"DEBUG: Person {person_idx} extracted {len(pred_cropped_faces)} valid faces")
                        
                        if pred_cropped_faces:
                            pred_cropped_faces = torch.cat(pred_cropped_faces, dim=0)
                            print(f"DEBUG: pred_cropped_faces.shape: {pred_cropped_faces.shape}")
                            
                            pred_face_embedding = self.face_embedding_extractor(pred_cropped_faces)
                            print(f"DEBUG: pred_face_embedding.shape: {pred_face_embedding.shape}")
                            

                            target_embedding = org_face_embedding[:pred_face_embedding.size(0)]
                            print(f"DEBUG: target_embedding.shape: {target_embedding.shape}")
                            
                            face_loss = (1 - cosine_loss(pred_face_embedding, target_embedding)).abs().mean()
                            
                            if total_face_loss is None:
                                total_face_loss = face_loss
                            else:
                                total_face_loss += face_loss
                            
                            print(f"Person {person_idx} face loss: {face_loss.item():.6f}")
                        else:
                            print(f"DEBUG: Person {person_idx} has no valid cropped faces")
                    
                    if total_face_loss is not None:
                        print(f"Total face loss: {total_face_loss.item():.6f}")
                        

                        optimizer.zero_grad()
                        total_face_loss.backward(retain_graph=True)
                        optimizer.step()
                    else:
                        print("No valid faces found for optimization")
                
                pred_original_sample = z0.detach().to(device=device)
                torch.cuda.empty_cache()


        derivative = (sample - pred_original_sample) / sigma_hat

        dt = self.sigmas[self.step_index + 1] - sigma_hat

        prev_sample = sample + derivative * dt


        prev_sample = prev_sample.to(model_output.dtype)


        self._step_index += 1

        if not return_dict:
            return (prev_sample,)

        return EulerDiscreteSchedulerProOutput(prev_sample=prev_sample, pred_original_sample=pred_original_sample)
        
    def add_noise(
        self,
        original_samples: torch.Tensor,
        noise: torch.Tensor,
        timesteps: torch.Tensor,
    ) -> torch.Tensor:

        sigmas = self.sigmas.to(device=original_samples.device, dtype=original_samples.dtype)
        if original_samples.device.type == "mps" and torch.is_floating_point(timesteps):

            schedule_timesteps = self.timesteps.to(original_samples.device, dtype=torch.float32)
            timesteps = timesteps.to(original_samples.device, dtype=torch.float32)
        else:
            schedule_timesteps = self.timesteps.to(original_samples.device)
            timesteps = timesteps.to(original_samples.device)


        if self.begin_index is None:
            step_indices = [self.index_for_timestep(t, schedule_timesteps) for t in timesteps]
        elif self.step_index is not None:

            step_indices = [self.step_index] * timesteps.shape[0]
        else:

            step_indices = [self.begin_index] * timesteps.shape[0]

        sigma = sigmas[step_indices].flatten()
        while len(sigma.shape) < len(original_samples.shape):
            sigma = sigma.unsqueeze(-1)

        noisy_samples = original_samples + noise * sigma
        return noisy_samples

    def get_velocity(self, sample: torch.Tensor, noise: torch.Tensor, timesteps: torch.Tensor) -> torch.Tensor:
        if (
            isinstance(timesteps, int)
            or isinstance(timesteps, torch.IntTensor)
            or isinstance(timesteps, torch.LongTensor)
        ):
            raise ValueError(
                (
                    "Passing integer indices (e.g. from `enumerate(timesteps)`) as timesteps to"
                    " `EulerDiscreteScheduler.get_velocity()` is not supported. Make sure to pass"
                    " one of the `scheduler.timesteps` as a timestep."
                ),
            )

        if sample.device.type == "mps" and torch.is_floating_point(timesteps):

            schedule_timesteps = self.timesteps.to(sample.device, dtype=torch.float32)
            timesteps = timesteps.to(sample.device, dtype=torch.float32)
        else:
            schedule_timesteps = self.timesteps.to(sample.device)
            timesteps = timesteps.to(sample.device)

        step_indices = [self.index_for_timestep(t, schedule_timesteps) for t in timesteps]
        alphas_cumprod = self.alphas_cumprod.to(sample)
        sqrt_alpha_prod = alphas_cumprod[step_indices] ** 0.5
        sqrt_alpha_prod = sqrt_alpha_prod.flatten()
        while len(sqrt_alpha_prod.shape) < len(sample.shape):
            sqrt_alpha_prod = sqrt_alpha_prod.unsqueeze(-1)

        sqrt_one_minus_alpha_prod = (1 - alphas_cumprod[step_indices]) ** 0.5
        sqrt_one_minus_alpha_prod = sqrt_one_minus_alpha_prod.flatten()
        while len(sqrt_one_minus_alpha_prod.shape) < len(sample.shape):
            sqrt_one_minus_alpha_prod = sqrt_one_minus_alpha_prod.unsqueeze(-1)

        velocity = sqrt_alpha_prod * noise - sqrt_one_minus_alpha_prod * sample
        return velocity

    def __len__(self):
        return self.config.num_train_timesteps