""" 3D ResNet backbone adapted from https://github.com/Tencent/MedicalNet/blob/master/models/resnet.py. """
import torch
import torch.nn as nn
import torch.nn.functional as F
from torch.autograd import Variable
import math
from functools import partial

__all__ = [
    'ResNet', 'resnet10', 'resnet18', 'resnet34', 'resnet50', 'resnet101',
    'resnet152', 'resnet200'
]


def conv3x3x3(in_planes, out_planes, stride=1, dilation=1):
    # 3x3x3 convolution with padding
    return nn.Conv3d(
        in_planes,
        out_planes,
        kernel_size=3,
        dilation=dilation,
        stride=stride,
        padding=dilation,
        bias=False)


def downsample_basic_block(x, planes, stride, no_cuda=False):
    out = F.avg_pool3d(x, kernel_size=1, stride=stride)
    zero_pads = torch.Tensor(
        out.size(0), planes - out.size(1), out.size(2), out.size(3),
        out.size(4)).zero_()
    if not no_cuda:
        if isinstance(out.data, torch.cuda.FloatTensor):
            zero_pads = zero_pads.cuda()

    out = Variable(torch.cat([out.data, zero_pads], dim=1))

    return out


class BasicBlock(nn.Module):
    expansion = 1

    def __init__(self, inplanes, planes, stride=1, dilation=1, downsample=None):
        super(BasicBlock, self).__init__()
        self.conv1 = conv3x3x3(inplanes, planes, stride=stride, dilation=dilation)
        self.bn1 = nn.BatchNorm3d(planes)
        self.relu = nn.ReLU(inplace=True)
        self.conv2 = conv3x3x3(planes, planes, dilation=dilation)
        self.bn2 = nn.BatchNorm3d(planes)
        self.downsample = downsample
        self.stride = stride
        self.dilation = dilation

    def forward(self, x):
        residual = x

        out = self.conv1(x)
        out = self.bn1(out)
        out = self.relu(out)
        out = self.conv2(out)
        out = self.bn2(out)

        if self.downsample is not None:
            residual = self.downsample(x)

        out += residual
        out = self.relu(out)

        return out


class Bottleneck(nn.Module):
    expansion = 4

    def __init__(self, inplanes, planes, stride=1, dilation=1, downsample=None):
        super(Bottleneck, self).__init__()
        self.conv1 = nn.Conv3d(inplanes, planes, kernel_size=1, bias=False)
        self.bn1 = nn.BatchNorm3d(planes)
        self.conv2 = nn.Conv3d(
            planes, planes, kernel_size=3, stride=stride, dilation=dilation, padding=dilation, bias=False)
        self.bn2 = nn.BatchNorm3d(planes)
        self.conv3 = nn.Conv3d(planes, planes * 4, kernel_size=1, bias=False)
        self.bn3 = nn.BatchNorm3d(planes * 4)
        self.relu = nn.ReLU(inplace=True)
        self.downsample = downsample
        self.stride = stride
        self.dilation = dilation

    def forward(self, x):
        residual = x

        out = self.conv1(x)
        out = self.bn1(out)
        out = self.relu(out)

        out = self.conv2(out)
        out = self.bn2(out)
        out = self.relu(out)

        out = self.conv3(out)
        out = self.bn3(out)

        if self.downsample is not None:
            residual = self.downsample(x)

        out += residual
        out = self.relu(out)

        return out


class ResNet3D(nn.Module):
    """ResNet-50"""
    def __init__(self, cfg, debug=False):
        ######
        block=Bottleneck,
        layers=cfg.get('layers', [3, 4, 6, 3])
        self.num_output_maps = len(cfg.get('out_fmaps', ['P5', 'P4', 'P3']))
        shortcut_type='B'
        no_cuda = False
        #######
        self.no_max_pool = False
        self.inplanes = cfg.get('start_channels', 64)
        inplanes = cfg.get('start_channels', 64)
        self.no_cuda = no_cuda
        super(ResNet3D, self).__init__()
        self.conv1 = nn.Conv3d(
            1,
            inplanes,
            kernel_size=5,
            stride=(2, 2, 2),
            padding=(2, 2, 2),
            bias=False)

        self.bn1 = nn.BatchNorm3d(inplanes)
        self.relu = nn.ReLU(inplace=True)
        self.maxpool = nn.MaxPool3d(kernel_size=(3, 3, 3), stride=2, padding=1)
        self.layer1 = self._make_layer(block, inplanes , layers[0], shortcut_type)
        self.layer2 = self._make_layer(block, inplanes * 2, layers[1], shortcut_type, stride=2)
        self.layer3 = self._make_layer(block, inplanes * 4, layers[2], shortcut_type, stride=2, dilation=2)
        self.layer4 = self._make_layer(block, inplanes * 8, layers[3], shortcut_type, stride=2, dilation=4)

        self._out_feature_strides = {}
        #self._out_features = cfg.MODEL.SEM_SEG_HEAD.IN_FEATURES
        self._out_channels = cfg.get('fpn_channels', 258)
        groups = int(self._out_channels//6)
        self._out3 = nn.Sequential(nn.Conv3d(inplanes * 2 * 4, self._out_channels, kernel_size=1),nn.GroupNorm(groups, self._out_channels))
        self._out2 = nn.Sequential(nn.Conv3d(inplanes * 4 * 4, self._out_channels, kernel_size=1),nn.GroupNorm(groups, self._out_channels))
        self._out1 = nn.Sequential(nn.Conv3d(inplanes * 8 * 4, self._out_channels, kernel_size=1),nn.GroupNorm(groups, self._out_channels))
        """for Plevel in self._out_features:
            self._out_feature_strides[Plevel] = 2"""

        for m in self.modules():
            if isinstance(m, nn.Conv3d):
                m.weight = nn.init.kaiming_normal(m.weight, mode='fan_out')
            elif isinstance(m, nn.BatchNorm3d):
                m.weight.data.fill_(1)
                m.bias.data.zero_()
    
    def _make_layer(self, block, planes, blocks, shortcut_type, stride=1, dilation=1):
        downsample = None
        block = block[-1]
        if stride != 1 or self.inplanes != planes * block.expansion:
            if shortcut_type == 'A':
                downsample = partial(
                    downsample_basic_block,
                    planes=planes * block.expansion,
                    stride=stride,
                    no_cuda=self.no_cuda)
            else:
                downsample = nn.Sequential(
                    nn.Conv3d(
                        self.inplanes,
                        planes * block.expansion,
                        kernel_size=1,
                        stride=stride,
                        bias=False), nn.BatchNorm3d(planes * block.expansion))

        layers = []
        layers.append(block(self.inplanes, planes, stride=stride, dilation=dilation, downsample=downsample))
        self.inplanes = planes * block.expansion
        for i in range(1, blocks):
            layers.append(block(self.inplanes, planes, dilation=dilation))

        return nn.Sequential(*layers)

    def forward(self, x):
        outputs = {} 
        x = self.conv1(x)
        x = self.bn1(x)
        x = self.relu(x)

        if self.num_output_maps > 4:
            outputs[1] = x
        if not self.no_max_pool:
            x = self.maxpool(x)



        x = self.layer1(x)
        if self.num_output_maps > 3:
            outputs[2] = x
        
        x = self.layer2(x)
        if self.num_output_maps > 2:
            outputs[3] = self._out3(x)

        x = self.layer3(x)
        if self.num_output_maps > 1:
            outputs[4] = self._out2(x)

        x = self.layer4(x)
        outputs[5] = self._out1(x)
        


        #x = self.avgpool(x)
        #x = x.view(x.size(0), -1)
        #x = self.fc(x)
        
        return outputs


def resnet50(**kwargs):
    """Constructs a ResNet-50 model.
    """
    model = ResNet3D(Bottleneck, [3, 4, 6, 3], **kwargs)
    return model
