# Copyright (c) Facebook, Inc. and its affiliates.
#
# This source code is licensed under the MIT license found in the
# LICENSE file in the root directory of this source tree.

import faiss
import argparse
import numpy as np

from densephrases.scripts.reconstruct_server import rpc

############################################################
# Server implementation
############################################################


class ReconstructServer(rpc.Server):
    """ Assign version that can be exposed via RPC """

    def __init__(self, s: int, index: faiss.Index):
        rpc.Server.__init__(self, s)
        self.index = index

    def get_ntotal(self) -> int:
        return self.index.ntotal

    def reconstruct(self, keys):
        return np.stack(list(map(self.index.reconstruct, keys)))



def run_index_server(index: faiss.Index, port: int, v6: bool = False):
    """ serve requests for that index forerver """
    rpc.run_server(
        lambda s: ReconstructServer(s, index),
        port, v6=v6)

def get_args():
    parser = argparse.ArgumentParser()
    parser.add_argument('--index_path')
    parser.add_argument('--port', type=int)
    parser.add_argument('--v6', default=False, action='store_true')
    return parser.parse_args()


def main():
    args = get_args()
    index = faiss.read_index(args.index_path)
    run_index_server(index, args.port, args.v6)



if __name__ == '__main__':
    main()
