# Libraries and functions ----
library(ADRecommender)
suppressMessages(library(tidyverse))
library(scmamp)
library(furrr)
library(here)
library(rsample)
library(microbenchmark)
source(here('paper_scripts', 'utils', 'utils.R'))


prepare_parallelization()
seed_num <- c(7777, 7778, 7779, 7780, 7781)

# Data preparation ----
dat <- get_data_stratified_kfolds_cv(mfs_metaod = FALSE, metric = 'pr_auc', mfs_scaled = T)
split <- dat$data$splits[[1]]
training_perf <- get_train_perf_from_split(split)
training_mfs <- get_train_mfs_from_split(split)
testing_perf <- get_test_perf_from_split(split)
testing_mfs <- get_test_mfs_from_split(split)

# Fitting computation time ----

fitting_times <- microbenchmark(
  recommender('global_best') %>%
    fit(training_perf, training_mfs),
  recommender('orthus') %>%
    fit(training_perf, training_mfs),
  recommender('regression') %>%
    fit(training_perf, training_mfs),
  recommender('nearest_neighbor') %>%
    fit(training_perf, training_mfs)
  , unit = 'seconds', times = 10)

fit_df <- tribble(~"Recommender", ~"Average", ~"Min", ~"Max", ~"Height",
       "Regression", 15.75, 14.35, 18.73, 4,
       "Orthus", 6.62, 5.68, 8.89, 3,
       "Global Best", 0.03, 0.025 , 0.07, 2,
       "Nearest Neighbor", 0.005, 0.004, 0.006, 1)

p_fit <- ggplot(fit_df) +
  geom_segment(aes(x=Min, xend=Max, y = Height, yend = Height), color="black") +
  geom_point( aes(x=Min, y=Height)) +
  geom_point( aes(x=Max, y=Height)) +
  #coord_flip()+
  theme_light() +
  #theme(
  #  legend.position = "none",
  #) +
  xlab("Fitting time (s) for 83 datasets") +
  ylab("Recommender") +
  scale_y_continuous(labels = c('Nearest Neighbor', 'Global Best', 'Orthus', 'Regression')) +
  scale_x_log10()

ggsave(filename = 'computation_time_fitting.pdf',
       plot = p_fit,
       device = 'pdf',
       path = here('paper_scripts', 'appendices'),
       units = 'cm',
       width = 10,
       height = 5
       )


# Recommendation computation time

rec_gb <- recommender('global_best') %>%
  fit(training_perf, training_mfs)
rec_orthus <- recommender('orthus') %>%
  fit(training_perf, training_mfs)
rec_reg <- recommender('regression') %>%
  fit(training_perf, training_mfs)
rec_nn <- recommender('nearest_neighbor') %>%
  fit(training_perf, training_mfs)

rec_times <- microbenchmark(
  rec_gb %>%
    predict(testing_mfs),
  rec_orthus %>%
    predict(testing_mfs),
  rec_reg %>%
    predict(testing_mfs),
  rec_nn %>%
    predict(testing_mfs)
  , unit = 'ms', times = 10)


rec_df <- tribble(~"Recommender", ~"Min", ~"Max", ~"Height",
                  "Regression", 4764, 4953, 3,
                  "Orthus", 11293, 12563, 4,
                  "Global Best", 0.04, 0.14, 1,
                  "Nearest Neighbor", 97, 140, 2)

p_rec <- ggplot(rec_df) +
  geom_segment(aes(x=Min, xend=Max, y = Height, yend = Height), color="black") +
  geom_point( aes(x=Min, y=Height)) +
  geom_point( aes(x=Max, y=Height)) +
  #coord_flip()+
  theme_light() +
  #theme(
  #  legend.position = "none",
  #) +
  xlab("Recommendation time (ms) for 11 datasets") +
  ylab("Recommender") +
  scale_y_continuous(labels = c('Global Best', 'Nearest Neighbor', 'Regression', 'Orthus')) +
  scale_x_log10() +
  theme(axis.title = element_text(size = 10))

ggsave(filename = 'computation_time_recommendation.pdf',
       plot = p_rec,
       device = 'pdf',
       path = here('paper_scripts', 'appendices'),
       units = 'cm',
       width = 10,
       height = 5
)


