import torch
import torch.nn as nn

import copy
from ..utils.misc import stack, logmeanexp
from torch.distributions import kl_divergence
from .modules import build_mlp
from .modules import TTPoolingEncoder_Dim
from .attention import SelfAttn
import math
import numpy as np

import ipdb 
import torch.nn.functional as F
from torch.distributions import kl_divergence

import uncertainty_toolbox as uct

from torch.distributions.normal import Normal
from attrdict import AttrDict
from ..utils.misc import stack, logmeanexp

class PositionalEncoding(nn.Module):
    def __init__(self, d_model, max_seq_len):
        super(PositionalEncoding, self).__init__()
        position = torch.arange(0, max_seq_len).unsqueeze(1)
        div_term = torch.exp(torch.arange(0, d_model, 2)*(-math.log(10000.0)/d_model))
        pe_1 = torch.zeros(max_seq_len, d_model)
        pe_1[:, 0::2] = torch.sin(position * div_term)
        pe_1[:, 1::2] = torch.cos(position * div_term)
        self.register_buffer('pe_1', pe_1)

        pe_2 = torch.zeros(max_seq_len, d_model)

        pe_2[:, 0::2] = torch.cos(position * div_term)
        pe_2[:, 1::2] = torch.sin(position * div_term)
        self.register_buffer('pe_2', pe_2)

    def forward(self, x, y_dim):
        pe = torch.cat([self.pe_1[:x.size(-2)-y_dim, :], self.pe_2[:y_dim, :]])
        x = x + pe
        return x

class DimensionAggregator(nn.Module):
    def __init__(self, dim_hid, dim_out, max_seq_len=101):
        super(DimensionAggregator, self).__init__()
        self.dim_hid = dim_hid
        self.dim_out = dim_out
        self.positional_encoding = PositionalEncoding(self.dim_hid, max_seq_len)
        self.linear = nn.Linear(1, self.dim_hid)
        self.selfattention = SelfAttn(self.dim_hid, self.dim_out)

    def forward(self, data_xy, y_dim):
        data_xy_unsqueeze = data_xy.unsqueeze(-1) 
        # [B, num_data, dim_x+dim_y, 1]
        data_xy_linear = self.linear(data_xy_unsqueeze) 
        # [B, num_data, dim_x+dim_y, dim_hid]
        data_xy_positional = self.positional_encoding(data_xy_linear, y_dim) 
        # [B, num_data, dim_x+dim_y, dim_hid]
        
        data_xy_selfattn = self.selfattention(data_xy_positional.reshape(-1,\
            data_xy_positional.shape[-2], data_xy_positional.shape[-1])) 
        # [B * num_data, dim_x+dim_y, dim_out]

        data_xy_selfattn = data_xy_selfattn.reshape(data_xy_positional.shape[0],\
            data_xy_positional.shape[1], data_xy_positional.shape[2],
            data_xy_selfattn.shape[-1]) 
        # [B, num_data, dim_x+dim_y, dim_out]
        
        data_x_selfattn, data_y_selfattn = \
            data_xy_selfattn.split([data_xy_positional.shape[2]-y_dim,y_dim],
                                   dim=-2) 
        # [B, num_data, dim_x, dim_out], [B, num_data, dim_y, dim_out]
        
        data_x_agg = data_x_selfattn.mean(dim=-2, keepdim=True)
        data_x_expanded = data_x_agg.expand(-1, -1, y_dim, -1)

        data_xy_combined = torch.cat([data_x_expanded, data_y_selfattn], dim=-1)
        return data_xy_combined
    
class DTANP_Y_UNCERTAINTY_base(nn.Module):
    def __init__(
        self,
        dim_x,
        dim_y,
        d_model,
        emb_depth,
        dim_feedforward,
        nhead,
        dropout,
        num_layers,
        bound_std
    ):
        super(DTANP_Y_UNCERTAINTY_base, self).__init__()
        self.dim_agg = DimensionAggregator(int(d_model/2), int(d_model/2))

        encoder_layer = nn.TransformerEncoderLayer(d_model, nhead, dim_feedforward, dropout, batch_first=True)
        self.encoder1 = nn.TransformerEncoder(encoder_layer, num_layers)

        encoder_layer = nn.TransformerEncoderLayer(d_model, nhead, dim_feedforward, dropout, batch_first=True)
        self.encoder2 = nn.TransformerEncoder(encoder_layer, 2)

        self.bound_std = bound_std
        self.lenc = TTPoolingEncoder_Dim(
                dim_x=int(d_model/2),
                dim_y=int(d_model/2),
                dim_hid=d_model,
                dim_lat=dim_feedforward,
                self_attn=True,
                pre_depth=4,
                post_depth=2)

    def construct_input(self, batch, autoreg=False):
        x_y_ctx = torch.cat((batch.xc, batch.yc), dim=-1)
        x_0_tar = torch.cat((batch.xt, torch.zeros_like(batch.yt)), dim=-1)

        inp = torch.cat((x_y_ctx, x_0_tar), dim=1)

        return inp

    def create_mask(self, batch, y_dim, autoreg=False):
        num_ctx = batch.xc.shape[1]
        num_tar = batch.xt.shape[1]
        num_all = num_ctx + num_tar
        mask = torch.zeros(y_dim * num_all, y_dim * num_all, device='cuda').fill_(float('-inf'))
        mask[:, :y_dim * num_ctx] = 0.0

        return mask, num_tar 
    
    def encode(self, batch, z=None, num_samples=None, autoreg=False):
        y_dim = batch.yt.shape[-1]
        inp = self.construct_input(batch, autoreg)
        mask, num_tar = self.create_mask(batch, y_dim, autoreg)
        embeddings = self.dim_agg(inp, y_dim) 
            
        embeddings = embeddings.view(embeddings.shape[0], -1, embeddings.shape[-1])
        out = stack(self.encoder1(embeddings, mask), num_samples)

        if z is None:
            context_embeddings = embeddings[:, :batch.xc.shape[1]*y_dim]
            pz = self.encoder2(context_embeddings) 
            pz = self.lenc(pz)
            z = pz.rsample() if num_samples is None \
                    else pz.rsample([num_samples])
        
        z = stack(z, inp.shape[-2], -2)
        z = z.repeat(1, 1, y_dim, 1).view(num_samples, embeddings.shape[0], -1, z.shape[-1])
        out = torch.cat([out, z], dim=-1)
        out = out.view(*out.shape[:2], -1, y_dim, out.shape[-1])

        return out[:, :, -num_tar:,:]

    def lencode(self, batch, autoreg=False):
        
        inp = self.construct_input(batch, autoreg)
        embeddings = self.dim_agg(inp, batch.y.shape[-1])
        
        num_context = batch.xc.shape[1]
        num_total = batch.x.shape[1]
        
        embeddings = torch.mean(embeddings, dim=-2)
        
        # context_embeddings = embeddings[:, :num_context].reshape(-1, num_context * batch.y.shape[-1], embeddings.shape[-1])
        # total_embeddings = embeddings[:, :num_total].reshape(-1, num_total * batch.y.shape[-1], embeddings.shape[-1])
        
        context_embeddings = embeddings[:, :num_context]
        total_embeddings = embeddings[:, :num_total]
        
        pz = self.lenc(context_embeddings)
        qz = self.lenc(total_embeddings)
        return pz, qz

class DTANP_Y_UNCERTAINTY(DTANP_Y_UNCERTAINTY_base):
    def __init__(
        self,
        dim_x,
        dim_y,
        d_model,
        emb_depth,
        dim_feedforward,
        nhead,
        dropout,
        num_layers,
        bound_std=True
    ):
        super(DTANP_Y_UNCERTAINTY, self).__init__(
            dim_x,
            dim_y,
            d_model,
            emb_depth,
            dim_feedforward,
            nhead,
            dropout,
            num_layers,
            bound_std
        )

        self.predictor = nn.Sequential(
            nn.Linear(dim_feedforward+d_model, dim_feedforward),
            nn.ReLU(),
            nn.Linear(dim_feedforward, 2)
        )
    
    def forward(self, batch, num_samples, reduce_ll=True):
        outs = AttrDict()
        if self.training:
            pz, qz = self.lencode(batch)

            z = qz.rsample() if num_samples is None else \
                    qz.rsample([num_samples])

            py = self.predict(batch.xc, batch.yc, batch.x, z=z, num_samples=num_samples)

            if num_samples > 1 :
                recon = py.log_prob(stack(batch.y, num_samples)).sum(-1)
                log_qz = qz.log_prob(z).sum(-1)
                log_pz = pz.log_prob(z).sum(-1)

                log_w = recon.sum(-1) + log_pz - log_qz
                outs.loss = -logmeanexp(log_w).mean() / batch.x.shape[-2]
            else:
                outs.recon = py.log_prob(batch.y).sum(-1).mean()
                outs.kld = kl_divergence(qz, pz).sum(-1).mean()
                outs.loss = -outs.recon + outs.kld / batch.x.shape[-2]
        else:
            py = self.predict(batch.xc, batch.yc, batch.x, num_samples=num_samples)
            if num_samples is None:
                ll = py.log_prob(batch.y).sum(-1)
            else:
                y = torch.stack([batch.y]*num_samples)
                if reduce_ll:
                    ll = logmeanexp(py.log_prob(y).sum(-1))
                else:
                    ll = py.log_prob(y).sum(-1)
            num_ctx = batch.xc.shape[-2]
            
            if reduce_ll:
                outs.ctx_ll = ll[...,:num_ctx].mean()
                outs.tar_ll = ll[...,num_ctx:].mean()
            else:
                outs.ctx_ll = ll[...,:num_ctx].mean()
                outs.tar_ll = ll[...,num_ctx:].mean()
        return outs

    def predict(self, xc, yc, xt, z=None, num_samples=None):
        batch = AttrDict()
        batch.xc = xc
        batch.yc = yc
        batch.xt = xt
        batch.yt = torch.zeros((xt.shape[0], xt.shape[1], yc.shape[2]), device='cuda')

        z_target = self.encode(batch, z=z, num_samples=num_samples, autoreg=False)

        out = self.predictor(z_target)
        mean, std = torch.chunk(out, 2, dim=-1)
        mean, std = mean.reshape((*mean.shape[:-2],-1)), std.reshape((*std.shape[:-2],-1))
        if self.bound_std:
            std = 0.1 + 0.9 * F.softplus(std)
        else:
            std = torch.exp(std)

        return Normal(mean, std)

    def calculate_crps(self, y_true, means, stds):
        y_true = y_true.squeeze(-1)
        means = means.squeeze(-1)
        stds = stds.squeeze(-1)

        z = (y_true - means) / stds
        #cdf_z = 0.5 * (1 + torch.stack([torch.erf(z[i, ...]/torch.sqrt(torch.tensor(2.0, device=z.device))) for i in range(50)]).mean(dim=0))

        cdf_z = 0.5 * (1 + torch.erf(z/torch.sqrt(torch.tensor(2.0, device=z.device))).mean(dim=0))

        pdf_z = (torch.exp(-0.5 * z**2) / torch.sqrt(torch.tensor(2 * torch.pi, device=z.device))).mean(dim=0)

        crps = stds * (z * (2 * cdf_z - 1) + 2 * pdf_z - 1 / torch.sqrt(torch.tensor(torch.pi, device=z.device)))
        return crps.mean(dim=-1)
    
    def crps(self, batch, num_samples=None):
        outs = AttrDict()

        if num_samples is None:
            y = batch.y.unsqueeze(-1)
        else:
            y = torch.stack([batch.y]*num_samples)
            
        py = self.predict(batch.xc, batch.yc, batch.x, num_samples=num_samples)
        
        num_ctx = batch.xc.shape[-2]

        means = py.loc; stds = py.scale 

        ctx_means, tar_means = means[..., :num_ctx, :], means[..., num_ctx:, :]
        ctx_stds, tar_stds = stds[..., :num_ctx, :], stds[..., num_ctx:, :]
        y_ctx, y_tar = y[..., :num_ctx, :], y[..., num_ctx:, :]
                
        ctx_crps = self.calculate_crps(y_ctx, ctx_means, ctx_stds)
        tar_crps = self.calculate_crps(y_tar, tar_means, tar_stds)
        
        means = means.mean(dim=0)
        stds = torch.sqrt((stds**2).mean(dim=0) + (py.loc**2).mean(dim=0) - (py.loc.mean(dim=0)**2))
        
        z_score = Normal(0, 1).icdf(torch.tensor([(1 + 0.68) / 2])).to(means.device)
        ctx_means, tar_means = means[..., :num_ctx, :], means[..., num_ctx:, :]
        ctx_stds, tar_stds = stds[..., :num_ctx, :], stds[..., num_ctx:, :]
        y_ctx, y_tar = y[..., :num_ctx, :], y[..., num_ctx:, :]

        lower_bounds_ctx = ctx_means - z_score * ctx_stds
        upper_bounds_ctx = ctx_means + z_score * ctx_stds
        lower_bounds_tar = tar_means - z_score * tar_stds
        upper_bounds_tar = tar_means + z_score * tar_stds

        outs.ctx_ci = ((y_ctx >= lower_bounds_ctx) & (y_ctx <= upper_bounds_ctx)).float().mean()
        outs.tar_ci = ((y_tar >= lower_bounds_tar) & (y_tar <= upper_bounds_tar)).float().mean()
        
        outs.ctx_crps = ctx_crps.mean()
        outs.tar_crps = tar_crps.mean()
        
        return outs
    
    def uncertainty(self, batch, num_samples=None):
        outs = AttrDict()

        if num_samples is None:
            y = batch.y.unsqueeze(-1)
        else:
            y = torch.stack([batch.y]*num_samples)
            
        py = self.predict(batch.xc, batch.yc, batch.x, num_samples=num_samples)
        
        num_ctx = batch.xc.shape[-2]

        means = py.loc; stds = py.scale 
        
        means = means.mean(dim=0)
        stds = torch.sqrt((stds**2).mean(dim=0) + (py.loc**2).mean(dim=0) - (py.loc.mean(dim=0)**2))
        
        z_score = Normal(0, 1).icdf(torch.tensor([(1 + 0.68) / 2])).to(means.device)
        ctx_means, tar_means = means[..., :num_ctx, :], means[..., num_ctx:, :]
        ctx_stds, tar_stds = stds[..., :num_ctx, :], stds[..., num_ctx:, :]
        y_ctx, y_tar = y[..., :num_ctx, :], y[..., num_ctx:, :]

        ctx_metrics = uct.metrics.get_all_metrics(ctx_means.cpu().numpy()[0,:,0], ctx_stds.cpu().numpy()[0,:,0], y_ctx.cpu().numpy()[0,0,:,0])
        tar_metrics = uct.metrics.get_all_metrics(tar_means.cpu().numpy()[0,:,0], tar_stds.cpu().numpy()[0,:,0], y_tar.cpu().numpy()[0,0,:,0])

        outs.ctx_mae = ctx_metrics['accuracy']['mae']
        outs.ctx_rmse = ctx_metrics['accuracy']['rmse']
        outs.ctx_mdae = ctx_metrics['accuracy']['mdae']
        outs.ctx_marpd = ctx_metrics['accuracy']['marpd']
        outs.ctx_r2 = ctx_metrics['accuracy']['r2']
        outs.ctx_corr = ctx_metrics['accuracy']['corr']
        outs.ctx_rms_cal = ctx_metrics['avg_calibration']['rms_cal']
        outs.ctx_ma_cal = ctx_metrics['avg_calibration']['ma_cal']
        outs.ctx_miscal_area = ctx_metrics['avg_calibration']['miscal_area']
        
        outs.tar_mae = tar_metrics['accuracy']['mae']
        outs.tar_rmse = tar_metrics['accuracy']['rmse']
        outs.tar_mdae = tar_metrics['accuracy']['mdae']
        outs.tar_marpd = tar_metrics['accuracy']['marpd']
        outs.tar_r2 = tar_metrics['accuracy']['r2']
        outs.tar_corr = tar_metrics['accuracy']['corr']
        outs.tar_rms_cal = tar_metrics['avg_calibration']['rms_cal']
        outs.tar_ma_cal = tar_metrics['avg_calibration']['ma_cal']
        outs.tar_miscal_area = tar_metrics['avg_calibration']['miscal_area']


        return outs