import torch
import torch.nn as nn

import copy
from ..utils.misc import stack, logmeanexp
from torch.distributions import kl_divergence
from .modules import build_mlp
from .modules import TTPoolingEncoder_Dim
from .attention import SelfAttn
import math

import ipdb 
import torch.nn.functional as F
from torch.distributions import kl_divergence

from torch.distributions.normal import Normal
from attrdict import AttrDict
from ..utils.misc import stack, logmeanexp

class PositionalEncoding(nn.Module):
    def __init__(self, d_model, max_seq_len):
        super(PositionalEncoding, self).__init__()
        position = torch.arange(0, max_seq_len).unsqueeze(1)
        div_term = torch.exp(torch.arange(0, d_model, 2)*(-math.log(10000.0)/d_model))
        pe_1 = torch.zeros(max_seq_len, d_model)
        pe_1[:, 0::2] = torch.sin(position * div_term)
        pe_1[:, 1::2] = torch.cos(position * div_term)
        self.register_buffer('pe_1', pe_1)

        pe_2 = torch.zeros(max_seq_len, d_model)

        pe_2[:, 0::2] = torch.cos(position * div_term)
        pe_2[:, 1::2] = torch.sin(position * div_term)
        self.register_buffer('pe_2', pe_2)

    def forward(self, x, y_dim):
        pe = torch.cat([self.pe_1[:x.size(-2)-y_dim, :], self.pe_2[:y_dim, :]])
        x = x + pe
        return x

class DimensionAggregator(nn.Module):
    def __init__(self, dim_hid, dim_out, max_seq_len=101):
        super(DimensionAggregator, self).__init__()
        self.dim_hid = dim_hid
        self.dim_out = dim_out
        self.positional_encoding = PositionalEncoding(self.dim_hid, max_seq_len)
        self.linear = nn.Linear(1, self.dim_hid)
        self.selfattention = SelfAttn(self.dim_hid, self.dim_out)

    def forward(self, data_xy, y_dim):
        data_xy_unsqueeze = data_xy.unsqueeze(-1) 
        # [B, num_data, dim_x+dim_y, 1]
        data_xy_linear = self.linear(data_xy_unsqueeze) 
        # [B, num_data, dim_x+dim_y, dim_hid]
        data_xy_positional = self.positional_encoding(data_xy_linear, y_dim) 
        # [B, num_data, dim_x+dim_y, dim_hid]
        
        data_xy_selfattn = self.selfattention(data_xy_positional.reshape(-1,\
            data_xy_positional.shape[-2], data_xy_positional.shape[-1])) 
        # [B * num_data, dim_x+dim_y, dim_out]
        
        data_xy_selfattn = data_xy_selfattn.reshape(data_xy_positional.shape[0],\
            data_xy_positional.shape[1], data_xy_positional.shape[2],
            data_xy_selfattn.shape[-1]) 
        # [B, num_data, dim_x+dim_y, dim_out]
        
        data_x_selfattn, data_y_selfattn = \
            data_xy_selfattn.split([data_xy_positional.shape[2]-y_dim,y_dim],
                                   dim=-2) 
        # [B, num_data, dim_x, dim_out], [B, num_data, dim_y, dim_out]
        
        data_x_agg = data_x_selfattn.mean(dim=-2, keepdim=True)
        data_x_expanded = data_x_agg.expand(-1, -1, y_dim, -1)

        data_xy_combined = torch.cat([data_x_expanded, data_y_selfattn], dim=-1)
        return data_xy_combined
    
class DTANP_Y_base(nn.Module):
    def __init__(
        self,
        dim_x,
        dim_y,
        d_model,
        emb_depth,
        dim_feedforward,
        nhead,
        dropout,
        num_layers,
        bound_std
    ):
        super(DTANP_Y_base, self).__init__()
        self.dim_agg = DimensionAggregator(int(d_model/2), int(d_model/2))

        encoder_layer = nn.TransformerEncoderLayer(d_model, nhead, dim_feedforward, dropout, batch_first=True)
        self.encoder1 = nn.TransformerEncoder(encoder_layer, num_layers)

        encoder_layer = nn.TransformerEncoderLayer(d_model, nhead, dim_feedforward, dropout, batch_first=True)
        self.encoder2 = nn.TransformerEncoder(encoder_layer, 2)

        self.bound_std = bound_std
        self.lenc = TTPoolingEncoder_Dim(
                dim_x=int(d_model/2),
                dim_y=int(d_model/2),
                dim_hid=d_model,
                dim_lat=dim_feedforward,
                self_attn=True,
                pre_depth=4,
                post_depth=2)

    def construct_input(self, batch, autoreg=False):
        x_y_ctx = torch.cat((batch.xc, batch.yc), dim=-1)
        x_0_tar = torch.cat((batch.xt, torch.zeros_like(batch.yt)), dim=-1)

        inp = torch.cat((x_y_ctx, x_0_tar), dim=1)

        return inp

    def create_mask(self, batch, y_dim, autoreg=False):
        num_ctx = batch.xc.shape[1]
        num_tar = batch.xt.shape[1]
        num_all = num_ctx + num_tar
        mask = torch.zeros(y_dim * num_all, y_dim * num_all, device='cuda').fill_(float('-inf'))
        mask[:, :y_dim * num_ctx] = 0.0

        return mask, num_tar 
    
    def encode(self, batch, z=None, num_samples=None, autoreg=False):
        y_dim = batch.yt.shape[-1]
        inp = self.construct_input(batch, autoreg)
        mask, num_tar = self.create_mask(batch, y_dim, autoreg)
        embeddings = self.dim_agg(inp, y_dim) 
            
        embeddings = embeddings.view(embeddings.shape[0], -1, embeddings.shape[-1])
        out = stack(self.encoder1(embeddings, mask), num_samples)
        
        if z is None:
            context_embeddings = embeddings[:, :batch.xc.shape[1]*y_dim]
            pz = self.encoder2(context_embeddings) 
            pz = self.lenc(pz)
            z = pz.rsample() if num_samples is None \
                    else pz.rsample([num_samples])
        
        z = stack(z, inp.shape[-2], -2)
        z = z.repeat(1, 1, y_dim, 1).view(num_samples, embeddings.shape[0], -1, z.shape[-1])
        out = torch.cat([out, z], dim=-1)
        out = out.view(*out.shape[:2], -1, y_dim, out.shape[-1])

        return out[:, :, -num_tar:,:]

    def lencode(self, batch, autoreg=False):
        
        inp = self.construct_input(batch, autoreg)
        embeddings = self.dim_agg(inp, batch.y.shape[-1])
        
        num_context = batch.xc.shape[1]
        num_total = batch.x.shape[1]
        
        embeddings = torch.mean(embeddings, dim=-2)
        
        # context_embeddings = embeddings[:, :num_context].reshape(-1, num_context * batch.y.shape[-1], embeddings.shape[-1])
        # total_embeddings = embeddings[:, :num_total].reshape(-1, num_total * batch.y.shape[-1], embeddings.shape[-1])
        
        context_embeddings = embeddings[:, :num_context]
        total_embeddings = embeddings[:, :num_total]
        
        pz = self.lenc(context_embeddings)
        qz = self.lenc(total_embeddings)
        return pz, qz

class DTANP_YN(DTANP_Y_base):
    def __init__(
        self,
        dim_x,
        dim_y,
        d_model,
        emb_depth,
        dim_feedforward,
        nhead,
        dropout,
        num_layers,
        bound_std=True
    ):
        super(DTANP_YN, self).__init__(
            dim_x,
            dim_y,
            d_model,
            emb_depth,
            dim_feedforward,
            nhead,
            dropout,
            num_layers,
            bound_std
        )

        self.predictor = nn.Sequential(
            nn.Linear(dim_feedforward+d_model, dim_feedforward),
            nn.ReLU(),
            nn.Linear(dim_feedforward, dim_y*2)
        )
    
    def forward(self, batch, num_samples, reduce_ll=True):
        outs = AttrDict()
        if self.training:
            x_mean = batch.xc.mean(dim=-2).unsqueeze(dim=-2)
            x_stdv = batch.xc.std(dim=-2).unsqueeze(dim=-2)
            y_mean = batch.yc.mean(dim=-2).unsqueeze(dim=-2)
            y_stdv = batch.yc.std(dim=-2).unsqueeze(dim=-2)
            batch.xc = (batch.xc-x_mean)/x_stdv
            batch.yc = (batch.yc-y_mean)/y_stdv
            batch.xt = (batch.xt-x_mean)/x_stdv
            batch.yt = (batch.yt-y_mean)/y_stdv
            batch.x = (batch.x-x_mean)/x_stdv
            batch.y = (batch.y-y_mean)/y_stdv
            pz, qz = self.lencode(batch)

            z = qz.rsample() if num_samples is None else \
                    qz.rsample([num_samples])

            py = self.predict(batch.xc, batch.yc, batch.x, z=z, num_samples=num_samples)

            py.loc, py.scale = py.loc * y_stdv.unsqueeze(dim=0) + y_mean.unsqueeze(dim=0), py.scale * y_stdv.unsqueeze(dim=0)

            if num_samples > 1 :
                recon = py.log_prob(stack(batch.y * y_stdv + y_mean, num_samples)).sum(-1)
                log_qz = qz.log_prob(z).sum(-1)
                log_pz = pz.log_prob(z).sum(-1)

                log_w = recon.sum(-1) + log_pz - log_qz
                outs.loss = -logmeanexp(log_w).mean() / batch.x.shape[-2]
            else:
                outs.recon = py.log_prob(batch.y).sum(-1).mean()
                outs.kld = kl_divergence(qz, pz).sum(-1).mean()
                outs.loss = -outs.recon + outs.kld / batch.x.shape[-2]
        else:
            x_mean = batch.xc.mean(dim=-2).unsqueeze(dim=-2)
            x_stdv = batch.xc.std(dim=-2).unsqueeze(dim=-2)
            y_mean = batch.yc.mean(dim=-2).unsqueeze(dim=-2)
            y_stdv = batch.yc.std(dim=-2).unsqueeze(dim=-2)
            batch.xc = (batch.xc-x_mean)/x_stdv
            batch.yc = (batch.yc-y_mean)/y_stdv
            batch.xt = (batch.xt-x_mean)/x_stdv
            batch.yt = (batch.yt-y_mean)/y_stdv
            batch.x = (batch.x-x_mean)/x_stdv
            batch.y = (batch.y-y_mean)/y_stdv
            py = self.predict(batch.xc, batch.yc, batch.x, num_samples=num_samples)
            py.loc, py.scale = py.loc * y_stdv.unsqueeze(dim=0) + y_mean.unsqueeze(dim=0), py.scale * y_stdv.unsqueeze(dim=0)

            if num_samples is None:
                ll = py.log_prob(batch.y * y_stdv + y_mean).sum(-1)
            else:
                y = torch.stack([batch.y * y_stdv + y_mean]*num_samples)
                if reduce_ll:
                    ll = logmeanexp(py.log_prob(y).sum(-1))
                else:
                    ll = py.log_prob(y).sum(-1)
            num_ctx = batch.xc.shape[-2]
            
            if reduce_ll:
                outs.ctx_ll = ll[...,:num_ctx].mean()
                outs.tar_ll = ll[...,num_ctx:].mean()
            else:
                outs.ctx_ll = ll[...,:num_ctx].mean()
                outs.tar_ll = ll[...,num_ctx:].mean()
        return outs

    def predict(self, xc, yc, xt, z=None, num_samples=None):
        batch = AttrDict()
        batch.xc = xc
        batch.yc = yc
        batch.xt = xt
        batch.yt = torch.zeros((xt.shape[0], xt.shape[1], yc.shape[2]), device='cuda')

        z_target = self.encode(batch, z=z, num_samples=num_samples, autoreg=False)
        out = self.predictor(z_target)
        mean, std = torch.chunk(out, 2, dim=-1)
        mean, std = mean.reshape((*mean.shape[:-2],-1)), std.reshape((*std.shape[:-2],-1))
        if self.bound_std:
            std = 0.1 + 0.9 * F.softplus(std)
        else:
            std = torch.exp(std)

        return Normal(mean, std)