import torch
import torch.nn as nn
from attrdict import AttrDict
import math

from .modules import CrossAttnEncoder, Decoder, PoolingEncoder, CrossAttnEncoder_Dim, Decoder_Dim
from .attention import SelfAttn

import os
import sys
sys.path.append(os.path.dirname(os.path.abspath(os.path.dirname(__file__))))

from .neuraloperator.neuralop.models import FNO1d

class PositionalEncoding(nn.Module):
    def __init__(self, d_model, max_seq_len):
        super(PositionalEncoding, self).__init__()
        position = torch.arange(0, max_seq_len).unsqueeze(1)
        div_term = torch.exp(torch.arange(0, d_model, 2)*(-math.log(10000.0)/d_model))
        pe = torch.zeros(max_seq_len, d_model)
        pe[:, 0::2] = torch.sin(position * div_term)
        pe[:, 1::2] = torch.cos(position * div_term)
        # pe = pe.unsqueeze(0).transpose(0, 1)
        self.register_buffer('pe', pe)

    def forward(self, x):
        pe = torch.cat([self.pe[:x.size(-2)-1, :], self.pe[-1, :].unsqueeze(0)])
        x = x + pe
        return x

class DimensionAggregator(nn.Module):
    def __init__(self, dim_hid, dim_out, fno, max_seq_len=101):
        super(DimensionAggregator, self).__init__()
        self.dim_hid = dim_hid
        self.dim_out = dim_out
        self.positional_encoding = PositionalEncoding(self.dim_hid, max_seq_len)
        self.linear = nn.Linear(1, self.dim_hid)
        self.selfattention = SelfAttn(self.dim_hid, self.dim_out)
        self.sfno = fno

    def forward(self, data_xy):
        data_xy_unsqueeze = data_xy.unsqueeze(-1) # [B, num_data, dim_x+1, 1]
        data_xy_linear = self.linear(data_xy_unsqueeze) # [B, num_data, dim_x+1, dim_hid]
        # data_xy_positional = data_xy_linear # [B, num_data, dim_x+1, dim_hid]
        data_xy_positional = self.positional_encoding(data_xy_linear) # [B, num_data, dim_x+1, dim_hid]
        data_xy_positional = torch.swapaxes(data_xy_positional,-1,-2)
        shape = data_xy_positional.shape
        data_xy_positional = data_xy_positional.reshape(-1,*shape[-2:])
        data_xy_positional = torch.swapaxes(self.sfno(data_xy_positional).reshape([*shape]),-1,-2)
        data_xy_selfattn = self.selfattention(data_xy_positional.reshape(-1, data_xy_positional.shape[-2], data_xy_positional.shape[-1])) # [B * num_data, dim_x+1, dim_out]
        data_xy_selfattn = data_xy_selfattn.reshape(data_xy_positional.shape[0], data_xy_positional.shape[1], data_xy_positional.shape[2], data_xy_selfattn.shape[-1]) # [B, num_data, dim_x+1, dim_out]
        data_x_selfattn, data_y_selfattn = data_xy_selfattn.split([data_xy_positional.shape[2]-1,1],dim=-2) # [B, num_data, dim_x, dim_out], [B, num_data, 1, dim_out]
        data_x_agg = data_x_selfattn.mean(dim=-2, keepdim=True) # [B, num_data, 1, dim_out]
        data_xy_agg = torch.cat([data_x_agg, data_y_selfattn], dim=-2) # [B, num_data, 2, dim_out]])
        return data_xy_agg

class NORDCANP(nn.Module):
    def __init__(self,
            # dim_x=1,
            dim_y=1,
            dim_hid=128,
            # enc_v_depth=4,
            # enc_qk_depth=2,
            # enc_pre_depth=4,
            # enc_post_depth=2,
            dec_depth=3):

        super().__init__()

        dim_out_chan = 32
        projection_chan = 32
        token_len = 2
        self.learnable_token = nn.Parameter(torch.randn(token_len))
        self.sfno = FNO1d(n_modes_height=8, in_channels=dim_hid, out_channels=dim_hid, hidden_channels=projection_chan, projection_channels=projection_chan, factorization='dense')
        self.dim_out_chan = dim_out_chan

        self.enc1 = CrossAttnEncoder_Dim(dim_hid=dim_hid)

        # self.enc2 = PoolingEncoder(
        #         dim_x=dim_x,
        #         dim_y=dim_y,
        #         dim_hid=dim_hid,
        #         self_attn=True,
        #         pre_depth=enc_pre_depth,
        #         post_depth=enc_post_depth)

        self.dec = Decoder_Dim(
                dim_y=dim_y,
                dim_enc=dim_hid,
                dim_hid=dim_hid,
                depth=dec_depth)

        self.dim_agg = DimensionAggregator(dim_hid, dim_hid, self.sfno)
        self.encoded = None
        self.query   = None

    def predict(self, xc, yc, xt, num_samples=None):
        yt_fake = torch.zeros([*xt.shape[:-1],1]).to(xt.device)
        target_xy = torch.cat([xt, yt_fake], dim=-1) # [B, num_target, dim_x+1]
        context_xy = torch.cat([xc, yc], dim=-1) # [B, num_context, dim_x+1]
        # na_space_tar = self.learnable_token.expand([*target_xy.shape[:-1],len(self.learnable_token)])
        # na_space_con = self.learnable_token.expand([*context_xy.shape[:-1],len(self.learnable_token)])
        # context_xy = torch.cat([na_space_con, context_xy])
        # target_xy = torch.cat([na_space_tar, target_xy])
        target_xy_agg = self.dim_agg(target_xy) # [B, num_target, 2, dim_hid]
        context_xy_agg = self.dim_agg(context_xy) # [B, num_context, 2, dim_hid]
        query = target_xy_agg[:, :, 0, :] # [B, num_target, dim_hid]
        key = context_xy_agg[:, :, 0, :] # [B, num_context, dim_hid]
        value = context_xy_agg[:, :, 1, :] # [B, num_context, dim_hid]
        encoded = self.enc1(query, key, value)
        # theta2 = self.enc2(xc, yc)
        # encoded = torch.cat([theta1,
        #     torch.stack([theta2]*xt.shape[-2], -2)], -1)
        self.encoded = encoded
        self.query   = query

        return self.dec(encoded, query)

    def forward(self, batch, num_samples=None, reduce_ll=True):
        outs = AttrDict()
        py = self.predict(batch.xc, batch.yc, batch.x)
        ll = py.log_prob(batch.y).sum(-1)

        if self.training:
            outs.loss = -ll.mean()
        else:
            num_ctx = batch.xc.shape[-2]
            if reduce_ll:
                outs.ctx_ll = ll[...,:num_ctx].mean()
                outs.tar_ll = ll[...,num_ctx:].mean()
            else:
                outs.ctx_ll = ll[...,:num_ctx]
                outs.tar_ll = ll[...,num_ctx:]

        return outs
