import torch
import torch.nn as nn
from torch.distributions import kl_divergence
from attrdict import AttrDict

from torch.distributions import Normal
from torch.distributions.relaxed_categorical import RelaxedOneHotCategorical

from ..utils.misc import stack, logmeanexp, log_w_weighted_sum_exp
from ..utils.sampling import sample_subset
from .modules import PoolingEncoder, Decoder
import torch.nn.init as init

class ForwardTransition(nn.Module):
    def __init__(self, d_model):
        super(ForwardTransition, self).__init__()
        self.predictor = nn.Sequential(
            nn.Linear(2*d_model, 2*d_model),
            nn.ReLU(),
            nn.Linear(2*d_model, 2*d_model),
            nn.ReLU(),
            nn.Linear(2*d_model, 2*d_model)
        )
    
    def forward(self, psi):
        out = self.predictor(psi)
        mean, std = torch.chunk(out, 2, dim=-1)
        # std = 0.01+0.09*torch.sigmoid(std)
        std = 0.01*torch.exp(std)
        return Normal(torch.chunk(psi, 2, dim=-1)[0]+mean, std)


class BackwardTransition(nn.Module):
    def __init__(self, d_model):
        super(BackwardTransition, self).__init__()
        self.predictor = nn.Sequential(
            nn.Linear(2*d_model, 2*d_model),
            nn.ReLU(),
            nn.Linear(2*d_model, 2*d_model),
            nn.ReLU(),
            nn.Linear(2*d_model, 2*d_model)
        )
    
    def forward(self, psi):
        out = self.predictor(psi)
        mean, std = torch.chunk(out, 2, dim=-1)
        std = 0.1*torch.exp(std)
        return Normal(torch.chunk(psi, 2, dim=-1)[0]+mean, std)

class Gradeint_estimator(nn.Module):
    def __init__(self, d_model):
        super(Gradeint_estimator,self).__init__()
        self.predictor = nn.Sequential(
            nn.Linear(2*d_model, 2*d_model),
            nn.ReLU(),
            nn.Linear(2*d_model, 2*d_model),
            nn.ReLU(),
            nn.Linear(2*d_model, d_model)
        )

        self.apply(self._init_weights)
    
    def _init_weights(self, m):
        if isinstance(m, nn.Linear):
            init.zeros_(m.weight)
            init.zeros_(m.bias)
    
    def forward(self, psi_data):
        return self.predictor(psi_data)


class NP_TTS_LANGEVIN_ADDING(nn.Module):
    def __init__(self,
            dim_x=1,
            dim_y=1,
            dim_hid=128,
            dim_lat=128,
            enc_pre_depth=4,
            enc_post_depth=2,
            dec_depth=3):

        super().__init__()

        self.denc = PoolingEncoder(
                dim_x=dim_x,
                dim_y=dim_y,
                dim_hid=dim_hid,
                pre_depth=enc_pre_depth,
                post_depth=enc_post_depth)

        self.lenc = PoolingEncoder(
                dim_x=dim_x,
                dim_y=dim_y,
                dim_hid=dim_hid,
                dim_lat=dim_lat,
                pre_depth=enc_pre_depth,
                post_depth=enc_post_depth)

        self.dec = Decoder(
                dim_x=dim_x,
                dim_y=dim_y,
                dim_enc=dim_hid+dim_lat,
                dim_hid=dim_hid,
                depth=dec_depth)
        
        self.gradient_estimator = Gradeint_estimator(dim_lat)
        self.forward_transition = ForwardTransition(d_model=dim_lat)
        self.backward_transition = BackwardTransition(d_model=dim_lat)

    def compute_gradient_log_pi(self, T, t, z, pz, xc, yc, theta, sigma=1., num_samples=50):
        z_=z.clone().detach().requires_grad_(True)
        # z_ = z.clone().detach()
        # z_.requires_grad_(True)

        py = self.predict(xc, yc, xc, z=z_, num_samples=num_samples)

        loss = py.log_prob(yc).sum(dim=-2).squeeze(dim=-1).sum()

        loss.backward()

        grad = self.gradient_estimator(torch.cat([z,theta],-1))

        mean = torch.tensor([ 0.0035,  0.0329, -0.0217,  0.0685,  0.0085,  0.0802,  0.0094, -0.0053,                       
         0.0154,  0.0173,  0.0005, -0.0114,  0.0192, -0.0110, -0.0035, -0.0059,                       
         0.0022,  0.0025, -0.0092, -0.0198, -0.0027, -0.0241,  0.0131, -0.0127,                       
        -0.0102,  0.0126,  0.0166,  0.0050,  0.0295, -0.0017,  0.0237,  0.0085,                       
         0.0142, -0.0185,  0.0042, -0.0075, -0.0658, -0.0104,  0.0168, -0.0154,                       
         0.0285, -0.0804, -0.0086, -0.0448,  0.0079, -0.0033, -0.0253,  0.0258,                       
        -0.0269, -0.0404, -0.0144,  0.0116, -0.0096,  0.0048, -0.0148,  0.0047,                       
        -0.0087, -0.0548, -0.0114, -0.0086, -0.0177,  0.0636,  0.0108,  0.0046,                       
         0.0662, -0.0095, -0.0090, -0.0103,  0.0073, -0.0098, -0.0126,  0.0266,                       
        -0.0137, -0.0051,  0.0372,  0.0004, -0.0083, -0.0031,  0.0235,  0.0250,                       
         0.0023,  0.0105, -0.0083, -0.0135,  0.0251,  0.0396, -0.0095,  0.0310,                       
        -0.0178, -0.0228,  0.0218, -0.0258, -0.0213, -0.0021,  0.0142,  0.0090,                       
         0.0312,  0.0388,  0.0042, -0.0212,  0.0134, -0.0185,  0.0193,  0.0225,                       
         0.0094,  0.0041, -0.0236,  0.0229, -0.0221, -0.0253, -0.0075,  0.0377,                       
         0.0067,  0.0101, -0.0183,  0.0275, -0.0107, -0.0019,  0.0203,  0.0423,                       
        -0.0245,  0.0094,  0.0055, -0.0072, -0.0098, -0.0138, -0.0429,  0.0542], device='cuda:0').unsqueeze(0).expand(16,-1)
        # std = torch.tensor([0.0157, 0.0305, 0.0372, 0.0807, 0.0179, 0.1770, 0.0645, 0.0328, 0.0138,                       
        # 0.0140, 0.0696, 0.0304, 0.0716, 0.0174, 0.0208, 0.0094, 0.0253, 0.0485,                       
        # 0.0087, 0.0183, 0.0328, 0.0059, 0.0088, 0.0172, 0.0090, 0.0953, 0.0582,                       
        # 0.0083, 0.0180, 0.0153, 0.0221, 0.0170, 0.0241, 0.0778, 0.0210, 0.2163,                       
        # 0.0808, 0.0073, 0.0237, 0.0133, 0.0689, 0.0755, 0.0097, 0.1630, 0.0161,                       
        # 0.0206, 0.0731, 0.0617, 0.0163, 0.0311, 0.0170, 0.0152, 0.1277, 0.0075,                       
        # 0.0102, 0.0828, 0.2070, 0.0539, 0.0123, 0.0143, 0.0750, 0.0463, 0.0086,                       
        # 0.0097, 0.1265, 0.0472, 0.0176, 0.0472, 0.0227, 0.0074, 0.0833, 0.0411,                       
        # 0.0436, 0.0832, 0.0638, 0.0104, 0.0193, 0.0094, 0.0150, 0.0443, 0.0090,                       
        # 0.0155, 0.0577, 0.0337, 0.1094, 0.0405, 0.0130, 0.0579, 0.0268, 0.0214,                       
        # 0.0251, 0.0423, 0.0172, 0.0119, 0.0067, 0.0372, 0.0474, 0.0311, 0.0123,                       
        # 0.0388, 0.0366, 0.0459, 0.0161, 0.0097, 0.0083, 0.0235, 0.0159, 0.0595,                       
        # 0.2393, 0.0430, 0.0371, 0.0405, 0.0060, 0.0248, 0.0253, 0.0488, 0.0191,                       
        # 0.0161, 0.0243, 0.0661, 1.1723, 0.0086, 0.0296, 0.0407, 0.0130, 0.1904,                       
        # 0.0613, 0.2425], device='cuda:0').unsqueeze(0).expand(16,-1)
        # gradient = z_.grad * t/T - (1-t/T)*(z_-pz.loc)/pz.scale**2 - t/T*z_/sigma**2
        # gradient = z_.grad * t/T - (1-t/T)*(z_-pz.loc) - t/T*z_/sigma**2
        gradient = (z_.grad+grad) * t/T - (1-t/T)*(z_-pz.loc)/pz.scale**2 - t/T*(z_-mean)/pz.scale**2
        return gradient

    def test_time_scaling(self, xc, yc, pz=None, z=None, w=None, theta=None, T=10, cut_T=None, num_samples=50):
        log_w = torch.log(w).cuda()
        log_w_list = []
        for t in range(cut_T):
            forward_normal = self.forward_transition(torch.cat([z,theta], -1))

            new_z = forward_normal.rsample()
            backward_normal = self.backward_transition(torch.cat([new_z,theta], -1))
            update_log_pi   = self.compute_log_pi(T, t+1, new_z, pz, xc, yc, num_samples)
            previous_log_pi = self.compute_log_pi(T, t, z, pz, xc, yc, num_samples)
            
            log_backward = backward_normal.log_prob(z).sum(dim=-1).cuda()
            log_forward  = forward_normal.log_prob(new_z).sum(dim=-1).cuda()
            updated_new_log_w = log_w + update_log_pi + log_backward - previous_log_pi - log_forward

            z = new_z
            log_w = updated_new_log_w - torch.max(updated_new_log_w, dim=0,keepdim=True)[0]
            log_w_list.append(log_w)
            # shifted_w = torch.exp(updated_new_log_w - torch.max(updated_new_log_w, dim=0,keepdim=True)[0])
            # w = shifted_w / shifted_w.sum(dim=0, keepdim=True)
            # z = new_z
            # log_w = torch.log(w).cuda()
        log_w_diff = torch.max(log_w[:,0])-torch.min(log_w[:,0])
        update_log_pi_diff = torch.max(update_log_pi[:,0])-torch.min(update_log_pi[:,0])
        previous_log_pi_diff = torch.max(previous_log_pi[:,0])-torch.min(previous_log_pi[:,0])
        log_backward_diff = torch.max(log_backward[:,0])-torch.min(log_backward[:,0])
        log_forward_diff = torch.max(log_forward[:,0])-torch.min(log_forward[:,0])
        return z, log_w, log_w_list, log_w_diff, update_log_pi_diff, previous_log_pi_diff, log_backward_diff, log_forward_diff

    def test_time_scaling_resampling(self, xc, yc, pz=None, z=None, w=None, theta=None, T=10, cut_T=None, num_samples=50, threshold_rate=.5):
        log_w = torch.log(w).cuda()
        log_w_list = []
        # z_list = []
        ess_threshold = num_samples*threshold_rate
        # ess_threshold = 0.
        learning_rate = torch.tensor(0.003)
        sigma = 1.
        noise_scale = 0.1
        for t in range(cut_T):
            forward_gradient = self.compute_gradient_log_pi(T, t+1, z, pz, xc, yc, theta, sigma=sigma, num_samples=num_samples)
            # forward_normal = Normal(z+(cut_T-t)/cut_T*learning_rate*forward_gradient, torch.sqrt(torch.tensor(2.))*torch.sqrt(learning_rate)*torch.ones_like(z).cuda())
            # forward_normal = Normal(z+learning_rate*forward_gradient, 0.001*torch.sqrt(torch.tensor(2.))*torch.sqrt(learning_rate)*torch.ones_like(z).cuda())
            forward_normal = Normal(z+learning_rate*forward_gradient, noise_scale*torch.sqrt(torch.tensor(2.))*torch.sqrt(learning_rate)*torch.ones_like(z).cuda())
            # forward_normal = self.forward_transition(torch.cat([z,theta], -1))

            new_z = forward_normal.rsample()
            backward_gradient = self.compute_gradient_log_pi(T, t, new_z, pz, xc, yc, theta, sigma=sigma, num_samples=num_samples)
            # backward_normal = Normal(new_z+(cut_T-t+1)/cut_T*learning_rate*backward_gradient, torch.sqrt(torch.tensor(2.))*torch.sqrt(learning_rate)*torch.ones_like(new_z).cuda())
            # backward_normal = Normal(new_z+learning_rate*backward_gradient, 0.001*torch.sqrt(torch.tensor(2.))*torch.sqrt(learning_rate)*torch.ones_like(new_z).cuda())
            backward_normal = Normal(new_z+learning_rate*backward_gradient, noise_scale*torch.sqrt(torch.tensor(2.))*torch.sqrt(learning_rate)*torch.ones_like(new_z).cuda())
            # backward_normal = self.backward_transition(torch.cat([new_z,theta], -1))
            update_log_pi   = self.compute_log_pi(T, t+1, new_z, pz, xc, yc, num_samples)
            previous_log_pi = self.compute_log_pi(T, t, z, pz, xc, yc, num_samples)
            
            log_backward = backward_normal.log_prob(z).sum(dim=-1).cuda()
            log_forward  = forward_normal.log_prob(new_z).sum(dim=-1).cuda()
            updated_new_log_w = log_w + update_log_pi + log_backward - previous_log_pi - log_forward

            z = new_z
            # z_list.append(z)
            log_w = updated_new_log_w - torch.max(updated_new_log_w, dim=0,keepdim=True)[0]
            log_w_list.append(log_w)
            ess_now = self.stable_ess(log_w)
            # print('ESS before resampling')
            # print(ess_now)

            # print(ess_now)
            # print(torch.where(ess_now<ess_threshold))
            resample_index = torch.where(ess_now<ess_threshold)[0]
            resample_mask = ess_now<ess_threshold

            # print(resample_index)
            if len(resample_index) == 0:
                # print('No resampling')
                pass
            else:
                w = torch.exp(log_w - torch.max(log_w, dim=0,keepdim=True)[0])
                prob = w / w.sum(dim=0, keepdim=True)
                # resampling with hard gumbel softmax
                # make num_samples number of sampled_indices using hard gumbel softmax to ensure backpropagation

                sampled_indices = RelaxedOneHotCategorical(1, prob.T).rsample(sample_shape=(num_samples,)) # [num_samples(replace dimension), batch_size, num_samples(gumbel softmax dimension)]
                hard_sampled_indices = torch.argmax(sampled_indices, dim=-1)
                hard_sampled_indices = torch.nn.functional.one_hot(hard_sampled_indices, num_classes=prob.shape[0]).float() # [num_samples(replace dimension), batch_size, num_samples(gumbel softmax dimension)]
                st_sampled_indices = (hard_sampled_indices-sampled_indices).detach()+sampled_indices # [num_samples(replace dimension), batch_size, num_samples(gumbel softmax dimension)]
                
                resampled_z = z.clone() # [batch_size, num_samples, dim_lat]
                resampled_log_w = log_w.clone() # [batch_size, num_samples]
                # Here, we have to make resampled_z by multiplying st_sampled_indices and z where mask is True
                st_z = torch.einsum('ijk,kjd->ijd', st_sampled_indices, z)

                resampled_z[:,resample_mask,:] = st_z[:,resample_mask,:]
                resampled_log_w[:,resample_mask] = torch.zeros_like(log_w[:,resample_mask])
                # print(resampled_z[:10,:3,:10])
                z = resampled_z
                log_w = resampled_log_w

                # sampled_indices = RelaxedOneHotCategorical(1, prob.T).rsample().T
                # resampled_z = z.clone()
                # resampled_log_w = log_w.clone()
            # print('ESS after resampling')
            # print(self.stable_ess(log_w))
        
        log_w_diff = torch.max(log_w[:,0])-torch.min(log_w[:,0])
        update_log_pi_diff = torch.max(update_log_pi[:,0])-torch.min(update_log_pi[:,0])
        previous_log_pi_diff = torch.max(previous_log_pi[:,0])-torch.min(previous_log_pi[:,0])
        log_backward_diff = torch.max(log_backward[:,0])-torch.min(log_backward[:,0])
        log_forward_diff = torch.max(log_forward[:,0])-torch.min(log_forward[:,0])
        # return z, log_w, z_list, log_w_list, log_w_diff, update_log_pi_diff, previous_log_pi_diff, log_backward_diff, log_forward_diff
        return z, log_w, log_w_list, log_w_diff, update_log_pi_diff, previous_log_pi_diff, log_backward_diff, log_forward_diff

    def compute_log_pi(self, T, t, z, pz, xc, yc, num_samples):
        py = self.predict(xc, yc, xc, z=z, num_samples=num_samples)
        mean = torch.tensor([ 0.0035,  0.0329, -0.0217,  0.0685,  0.0085,  0.0802,  0.0094, -0.0053,                       
         0.0154,  0.0173,  0.0005, -0.0114,  0.0192, -0.0110, -0.0035, -0.0059,                       
         0.0022,  0.0025, -0.0092, -0.0198, -0.0027, -0.0241,  0.0131, -0.0127,                       
        -0.0102,  0.0126,  0.0166,  0.0050,  0.0295, -0.0017,  0.0237,  0.0085,                       
         0.0142, -0.0185,  0.0042, -0.0075, -0.0658, -0.0104,  0.0168, -0.0154,                       
         0.0285, -0.0804, -0.0086, -0.0448,  0.0079, -0.0033, -0.0253,  0.0258,                       
        -0.0269, -0.0404, -0.0144,  0.0116, -0.0096,  0.0048, -0.0148,  0.0047,                       
        -0.0087, -0.0548, -0.0114, -0.0086, -0.0177,  0.0636,  0.0108,  0.0046,                       
         0.0662, -0.0095, -0.0090, -0.0103,  0.0073, -0.0098, -0.0126,  0.0266,                       
        -0.0137, -0.0051,  0.0372,  0.0004, -0.0083, -0.0031,  0.0235,  0.0250,                       
         0.0023,  0.0105, -0.0083, -0.0135,  0.0251,  0.0396, -0.0095,  0.0310,                       
        -0.0178, -0.0228,  0.0218, -0.0258, -0.0213, -0.0021,  0.0142,  0.0090,                       
         0.0312,  0.0388,  0.0042, -0.0212,  0.0134, -0.0185,  0.0193,  0.0225,                       
         0.0094,  0.0041, -0.0236,  0.0229, -0.0221, -0.0253, -0.0075,  0.0377,                       
         0.0067,  0.0101, -0.0183,  0.0275, -0.0107, -0.0019,  0.0203,  0.0423,                       
        -0.0245,  0.0094,  0.0055, -0.0072, -0.0098, -0.0138, -0.0429,  0.0542], device='cuda:0').unsqueeze(0).expand(16,-1)
        # std = torch.tensor([0.0157, 0.0305, 0.0372, 0.0807, 0.0179, 0.1770, 0.0645, 0.0328, 0.0138,                       
        # 0.0140, 0.0696, 0.0304, 0.0716, 0.0174, 0.0208, 0.0094, 0.0253, 0.0485,                       
        # 0.0087, 0.0183, 0.0328, 0.0059, 0.0088, 0.0172, 0.0090, 0.0953, 0.0582,                       
        # 0.0083, 0.0180, 0.0153, 0.0221, 0.0170, 0.0241, 0.0778, 0.0210, 0.2163,                       
        # 0.0808, 0.0073, 0.0237, 0.0133, 0.0689, 0.0755, 0.0097, 0.1630, 0.0161,                       
        # 0.0206, 0.0731, 0.0617, 0.0163, 0.0311, 0.0170, 0.0152, 0.1277, 0.0075,                       
        # 0.0102, 0.0828, 0.2070, 0.0539, 0.0123, 0.0143, 0.0750, 0.0463, 0.0086,                       
        # 0.0097, 0.1265, 0.0472, 0.0176, 0.0472, 0.0227, 0.0074, 0.0833, 0.0411,                       
        # 0.0436, 0.0832, 0.0638, 0.0104, 0.0193, 0.0094, 0.0150, 0.0443, 0.0090,                       
        # 0.0155, 0.0577, 0.0337, 0.1094, 0.0405, 0.0130, 0.0579, 0.0268, 0.0214,                       
        # 0.0251, 0.0423, 0.0172, 0.0119, 0.0067, 0.0372, 0.0474, 0.0311, 0.0123,                       
        # 0.0388, 0.0366, 0.0459, 0.0161, 0.0097, 0.0083, 0.0235, 0.0159, 0.0595,                       
        # 0.2393, 0.0430, 0.0371, 0.0405, 0.0060, 0.0248, 0.0253, 0.0488, 0.0191,                       
        # 0.0161, 0.0243, 0.0661, 1.1723, 0.0086, 0.0296, 0.0407, 0.0130, 0.1904,                       
        # 0.0613, 0.2425], device='cuda:0').unsqueeze(0).expand(16,-1)
        # prior = Normal(mean,std)    
        prior = Normal(mean,pz.scale*torch.ones([16,128]).cuda().cuda())    
        log_pi = t/T*py.log_prob(yc).sum(dim=-2).squeeze(dim=-1) + t/T*prior.log_prob(z).sum(dim=-1)+(T-t)/T*pz.log_prob(z).sum(dim=-1)
        # print(py.log_prob(yc).sum(dim=-2).squeeze(dim=-1)[:,0])
        # print(prior.log_prob(z).sum(dim=-1)[:,0])
        # print(pz.log_prob(z).sum(dim=-1)[:,0])
        # print(log_pi[:,0])
        return log_pi


    def compute_ess_loss(self, log_w):
        ess = torch.exp(torch.logsumexp(2*log_w, dim=0)-2*torch.logsumexp(log_w, dim=0))
        return ess.mean()

    def stable_ess(self, log_weights):
        log_w_max = torch.max(log_weights, dim=0, keepdim=True)[0]
        normalized_weights = torch.exp(log_weights - log_w_max)
        sum_weights = torch.sum(normalized_weights, dim=0)
        sum_weights_squared = torch.sum(normalized_weights ** 2, dim=0)
        ess = (sum_weights ** 2) / sum_weights_squared
        return ess

    def predict(self, xc, yc, xt, z=None, num_samples=None):

        theta = stack(self.denc(xc, yc), num_samples)
        if z is None:
            pz = self.lenc(xc, yc)
            z = pz.rsample() if num_samples is None \
                    else pz.rsample([num_samples])

        encoded = torch.cat([theta, z], -1)

        encoded = stack(encoded, xt.shape[-2], -2)

        return self.dec(encoded, stack(xt, num_samples))

    def tts_predict(self, xc, yc, xt, z=None, num_samples=None, T=10, cut_T=None, resampling=False, threshold_rate=.5):
        theta = stack(self.denc(xc, yc), num_samples)
        if cut_T == None:
            cut_T = T
        if z is None:
            pz = self.lenc(xc, yc)
            z = pz.rsample() if num_samples is None \
                    else pz.rsample([num_samples])

        w = torch.ones([z.shape[0], z.shape[1]])/z.shape[0]
        if resampling == False:
            z, log_w, log_w_list, log_w_diff, update_log_pi_diff, previous_log_pi_diff, log_backward_diff, log_forward_diff = self.test_time_scaling(xc, yc, pz, z, w, theta, num_samples=num_samples,T=T,cut_T=cut_T)
        else:
            # z, log_w, z_list, log_w_list, log_w_diff, update_log_pi_diff, previous_log_pi_diff, log_backward_diff, log_forward_diff = self.test_time_scaling_resampling(xc, yc, pz, z, w, theta, num_samples=num_samples,T=T,cut_T=cut_T,threshold_rate=threshold_rate)
            z, log_w, log_w_list, log_w_diff, update_log_pi_diff, previous_log_pi_diff, log_backward_diff, log_forward_diff = self.test_time_scaling_resampling(xc, yc, pz, z, w, theta, num_samples=num_samples,T=T,cut_T=cut_T,threshold_rate=threshold_rate)
        encoded_list = []
        encoded = torch.cat([theta, z], -1)
        encoded = stack(encoded, xt.shape[-2], -2)
        # for i in range(cut_T):
        #     encoded = torch.cat([theta, z_list[i]], -1)
        #     encoded = stack(encoded, xt.shape[-2], -2)
        #     encoded_list.append(encoded)

        py_list = []
        py = self.dec(encoded, stack(xt, num_samples))
        # for i in range(cut_T):
        #     py_list.append(self.dec(encoded_list[i], stack(xt, num_samples)))

        if z.get_device() == '-1':
            log_w = stack(log_w, xt.shape[-2],-1).cpu()
        else:
            log_w = stack(log_w, xt.shape[-2],-1).cuda()

        # return py_list, log_w, log_w_list, log_w_diff, update_log_pi_diff, previous_log_pi_diff, log_backward_diff, log_forward_diff
        return py, log_w, log_w_list, log_w_diff, update_log_pi_diff, previous_log_pi_diff, log_backward_diff, log_forward_diff

    def sample(self, xc, yc, xt, z=None, num_samples=None):
        pred_dist = self.predict(xc, yc, xt, z, num_samples)
        return pred_dist.loc

    def forward(self, batch, num_samples=None, reduce_ll=True, test_time_scaling=False, ess=False, T=10, ess_lambda=1., threshold_rate=.5):
        outs = AttrDict()
        if test_time_scaling == False:
            if self.training:
                pz = self.lenc(batch.xc, batch.yc)
                qz = self.lenc(batch.x, batch.y)
                z = qz.rsample() if num_samples is None else \
                        qz.rsample([num_samples])
                py = self.predict(batch.xc, batch.yc, batch.x,
                        z=z, num_samples=num_samples)

                if num_samples > 1:
                    # K * B * N
                    recon = py.log_prob(stack(batch.y, num_samples)).sum(-1)
                    # K * B
                    log_qz = qz.log_prob(z).sum(-1)
                    log_pz = pz.log_prob(z).sum(-1)

                    # K * B
                    log_w = recon.sum(-1) + log_pz - log_qz

                    outs.loss = -logmeanexp(log_w).mean() / batch.x.shape[-2]
                else:
                    outs.recon = py.log_prob(batch.y).sum(-1).mean()
                    outs.kld = kl_divergence(qz, pz).sum(-1).mean()
                    outs.loss = -outs.recon + outs.kld / batch.x.shape[-2]

            else:
                py = self.predict(batch.xc, batch.yc, batch.x, num_samples=num_samples)
                if num_samples is None:
                    ll = py.log_prob(batch.y).sum(-1)
                else:
                    y = torch.stack([batch.y]*num_samples)
                    if reduce_ll:
                        ll = logmeanexp(py.log_prob(y).sum(-1))
                    else:
                        ll = py.log_prob(y).sum(-1)
                num_ctx = batch.xc.shape[-2]
                if reduce_ll:
                    outs.ctx_ll = ll[...,:num_ctx].mean()
                    outs.tar_ll = ll[...,num_ctx:].mean()
                else:
                    outs.ctx_ll = ll[...,:num_ctx]
                    outs.tar_ll = ll[...,num_ctx:]
            return outs
        else:
            if self.training:
                break_t = T
                # py_list, log_w, log_w_list, log_w_diff, update_log_pi_diff, previous_log_pi_diff, log_backward_diff, log_forward_diff = self.tts_predict(batch.xc, batch.yc, batch.x, num_samples=num_samples,T=T, cut_T=break_t, resampling=True, threshold_rate=threshold_rate)
                py, log_w, log_w_list, log_w_diff, update_log_pi_diff, previous_log_pi_diff, log_backward_diff, log_forward_diff = self.tts_predict(batch.xc, batch.yc, batch.x, num_samples=num_samples,T=T, cut_T=break_t, resampling=True, threshold_rate=threshold_rate)

                if num_samples > 1:
                    # K * B * N
                    outs.loss = 0
                    # outs.recon_loss = 0
                    # for i in range(break_t):
                    #     recon = py_list[i].log_prob(stack(batch.y, num_samples)).sum(-1)
                    #     # K * B
                    #     now_recon_loss = -log_w_weighted_sum_exp(recon, stack(log_w_list[i], batch.x.shape[-2],-1).cuda()).mean()
                    #     outs.loss += now_recon_loss
                    #     outs.recon_loss += now_recon_loss
                    # outs.recon_loss /= break_t
                    recon = py.log_prob(stack(batch.y, num_samples)).sum(-1)
                    # # K * B

                    outs.loss = -log_w_weighted_sum_exp(recon, log_w).mean()
                    
                    if ess ==True:
                        ess_loss_list = []
                        for i in range(break_t):
                            ess_loss_list.append(self.compute_ess_loss(log_w_list[i]))
                            outs.loss += ess_lambda*self.compute_ess_loss(log_w_list[i])
                        outs.ess_loss = sum(ess_loss_list)/break_t
                        # outs.ess_loss = self.compute_ess_loss(log_w)
                        # outs.loss += ess_lambda*outs.ess_loss
                    # outs.loss = -log_weighted_sum_exp(log_w, w) / batch.x.shape[-2]
                    outs.log_w_diff = log_w_diff
                    outs.update_log_pi_diff = update_log_pi_diff
                    outs.previous_log_pi_diff = previous_log_pi_diff
                    outs.log_backward_diff = log_backward_diff
                    outs.log_forward_diff = log_forward_diff
            else:
                # py_list, log_w, log_w_list, log_w_diff, update_log_pi_diff, previous_log_pi_diff, log_backward_diff, log_forward_diff = self.tts_predict(batch.xc, batch.yc, batch.x, num_samples=num_samples,T=T, resampling=True, threshold_rate=threshold_rate)
                py, log_w, log_w_list, log_w_diff, update_log_pi_diff, previous_log_pi_diff, log_backward_diff, log_forward_diff = self.tts_predict(batch.xc, batch.yc, batch.x, num_samples=num_samples,T=T, resampling=True, threshold_rate=threshold_rate)
                # print(self.stable_ess(log_w[...,0]))
                # exit()
                y = torch.stack([batch.y]*num_samples)
                if reduce_ll:
                    ll = log_w_weighted_sum_exp(py.log_prob(y).sum(-1), log_w)
                else:
                    ll = py.log_prob(y).sum(-1)
                num_ctx = batch.xc.shape[-2]
                if reduce_ll:
                    outs.ctx_ll = ll[...,:num_ctx].mean()
                    outs.tar_ll = ll[...,num_ctx:].mean()
                else:
                    outs.ctx_ll = ll[...,:num_ctx]
                    outs.tar_ll = ll[...,num_ctx:]
            return outs