import os, hashlib, random

class LLMClient:
    """Text generation interface.
    - stub: deterministic, offline, reproducible output
    - openai: uses OpenAI Chat Completions when OPENAI_API_KEY is set
    """
    def __init__(self, mode: str = None, temperature: float = 0.3):
        self.mode = mode or os.getenv("LLM4R_MODE", "stub")
        self.temperature = temperature
        if self.mode not in ("stub", "openai"):
            self.mode = "stub"

    def _seed_from_text(self, text: str) -> int:
        digest = hashlib.sha256(text.encode("utf-8")).hexdigest()
        return int(digest[:8], 16)

    def generate(self, prompt: str, max_tokens: int = 512) -> str:
        if self.mode == "openai":
            try:
                from openai import OpenAI
                client = OpenAI()
                resp = client.chat.completions.create(
                    model=os.getenv("LLM4R_MODEL", "gpt-4o-mini"),
                    messages=[{"role":"user","content":prompt}],
                    temperature=self.temperature,
                    max_tokens=max_tokens,
                )
                return resp.choices[0].message.content
            except Exception as e:
                return f"[openai_error] {e}\n[falling_back_to_stub]\n" + self._stub_output(prompt, max_tokens)
        return self._stub_output(prompt, max_tokens)

    def _stub_output(self, prompt: str, max_tokens: int) -> str:
        rnd = random.Random(self._seed_from_text(prompt))
        words = [w for w in prompt.split() if w.isalpha()]
        bag = words[:30] or ["agent","review","system","score","plan"]
        out = []
        for _ in range(min(120, max_tokens)):
            out.append(rnd.choice(bag))
            if rnd.random() < 0.12: out.append(".")
        return " ".join(out)
