function a = graphPlot(g, outputFile, varargin)
%GRAPHPLOT Plot a graph using SFDP.
%   A = GRAPHPLOT(G, OUTPUTFILE, OPTIONS) generates a PNG image of the graph G,
%   saved to OUTPUTFILE. If a return value A is specified, the image is
%   also loaded into A. The plot is generated by writing G to a temporary
%   file in dot format, and running the SFDP graph plotter on the file.
%
%   OPTIONS contains plot options: node circle height, width and fontSize;
%   graphType (GraphType.UNDIRECTED/GraphType.DIRECTED - determines whether
%   edge weights are showen as arc labels).
%
%   You will need to install GRAPHVIZ (including the SFDP plotter). See
%   http://www.graphviz.org/Download..php for installation instructions.
%
%   See also: WRITERDOT, GraphType.

global GLOBAL_VARS;

% Acquire writer instance. Could be in principle cached for all calls of
% this script.
writerFactory   = graph.writer.WriterFactory();
writerDot       = writerFactory.newInstance(graph.api.GraphFormat.DOT);

% Write dot file
%[dummy, fileName, ext] = fileparts(outputFile);
%dotFile = [tempdir '/' fileName ext '.dot'];
dotFile = tempname;
writerDot.write(g, dotFile, varargin{:});
if ((nargin < 2) || (isempty(outputFile)))
    outputFile = tempname;
end

% Generate graph plot using the SFDP utility (a system call)
status = system([GLOBAL_VARS.sfdp_exec ' -q2 -s300 -T png -Ksfdp -o"' outputFile '" ' ...
    dotFile]);
if (status ~= 0)
    error('MATLAB:plotGraph:sfdp', 'SFDP command failed to generate plot from dot file');
end

% Clean up
delete(dotFile);

if (nargout >= 1)
    % Load image into workspace
    a = imread(outputFile);
end

end
