# This source code is licensed under the MIT license found in the
# LICENSE file in the root directory of this source tree.
import random
from collections import defaultdict
import copy
import json
import os
from os.path import exists, join, isdir
from dataclasses import dataclass, field
import sys
from typing import Optional, Dict, Sequence
import numpy as np
from tqdm import tqdm
import logging
import bitsandbytes as bnb
import pandas as pd
import importlib
from packaging import version
from read_json import sort_dict_by_value, print_dict, divide_dict
from packaging.version import parse

import torch
import transformers
from transformers.models.llama.modeling_llama import LlamaRMSNorm
from transformers.utils.import_utils import is_sagemaker_mp_enabled
from torch.nn.utils.rnn import pad_sequence
import argparse
from transformers import (
    AutoTokenizer,
    AutoModelForCausalLM,
    set_seed,
    Seq2SeqTrainer,
    BitsAndBytesConfig,
    LlamaTokenizer,
)
from datasets import load_dataset, Dataset, load_from_disk
import evaluate

from peft import (
    prepare_model_for_kbit_training,
    LoraConfig,
    get_peft_model,
    PeftModel
)
from peft.tuners.lora import LoraLayer
from transformers.trainer_utils import PREFIX_CHECKPOINT_DIR

import matplotlib.pyplot as plt

IGNORE_INDEX = -100
DEFAULT_PAD_TOKEN = "<pad>"

@dataclass
class ModelArguments:
    # Optional[str] is equivalent to Union[str, None].
    model_name_or_path: Optional[str] = field(
        default="EleutherAI/pythia-12b"
    )
    trust_remote_code: Optional[bool] = field(
        default=False,
        metadata={"help": "Enable unpickling of arbitrary code in AutoModelForCausalLM#from_pretrained."}
    )
    use_auth_token: Optional[bool] = field(
        default=False,
        metadata={"help": "Enables using Huggingface auth token from Git Credentials."}
    )

@dataclass
class DataArguments:
    eval_dataset_size: int = field(
        default=1024, metadata={"help": "Size of validation dataset."}
    )
    max_train_samples: Optional[int] = field(
        default=None,
        metadata={
            "help": "For debugging purposes or quicker training, truncate the number of training examples to this "
            "value if set."
        },
    )
    max_eval_samples: Optional[int] = field(
        default=None,
        metadata={
            "help": "For debugging purposes or quicker training, truncate the number of evaluation examples to this "
            "value if set."
        },
    )
    source_max_len: int = field(
        default=1024,
        metadata={"help": "Maximum source sequence length. Sequences will be right padded (and possibly truncated)."},
    )
    target_max_len: int = field(
        default=256,
        metadata={"help": "Maximum target sequence length. Sequences will be right padded (and possibly truncated)."},
    )
    dataset: str = field(
        default='alpaca',
        metadata={"help": "Which dataset to finetune on. See datamodule for options."}
    )
    dataset_format: Optional[str] = field(
        default=None,
        metadata={"help": "Which dataset format is used. [alpaca|chip2|self-instruct|hh-rlhf]"}
    )

@dataclass
class TrainingArguments(transformers.Seq2SeqTrainingArguments):
    cache_dir: Optional[str] = field(
        default=None
    )
    train_on_source: Optional[bool] = field(
        default=False,
        metadata={"help": "Whether to train on the input in addition to the target text."}
    )
    mmlu_split: Optional[str] = field(
        default='eval',
        metadata={"help": "The MMLU split to run on"}
    )
    mmlu_dataset: Optional[str] = field(
        default='mmlu-fs',
        metadata={"help": "MMLU dataset to use: options are `mmlu-zs` for zero-shot or `mmlu-fs` for few shot."}
    )
    do_mmlu_eval: Optional[bool] = field(
        default=False,
        metadata={"help": "Whether to run the MMLU evaluation."}
    )
    max_mmlu_samples: Optional[int] = field(
        default=None,
        metadata={"help": "If set, only evaluates on `max_mmlu_samples` of the MMMLU dataset."}
    )
    mmlu_source_max_len: int = field(
        default=2048,
        metadata={"help": "Maximum source sequence length for mmlu."}
    )
    full_finetune: bool = field(
        default=False,
        metadata={"help": "Finetune the entire model without adapters."}
    )
    adam8bit: bool = field(
        default=False,
        metadata={"help": "Use 8-bit adam."}
    )
    double_quant: bool = field(
        default=True,
        metadata={"help": "Compress the quantization statistics through double quantization."}
    )
    quant_type: str = field(
        default="nf4",
        metadata={"help": "Quantization data type to use. Should be one of `fp4` or `nf4`."}
    )
    bits: int = field(
        default=4,
        metadata={"help": "How many bits to use."}
    )
    lora_r: int = field(
        default=64,
        metadata={"help": "Lora R dimension."}
    )
    lora_alpha: float = field(
        default=16,
        metadata={"help": " Lora alpha."}
    )
    lora_dropout: float = field(
        default=0.0,
        metadata={"help":"Lora dropout."}
    )
    max_memory_MB: int = field(
        default=80000,
        metadata={"help": "Free memory per gpu."}
    )
    report_to: str = field(
        default='none',
        metadata={"help": "To use wandb or something else for reporting."}
    )
    run_name: str = field(
        default='none',
        metadata={"help": "To use wandb, and the running name."}
    )
    whether_quantize: bool = field(default=False, metadata={"help": 'Whether to quantize the model.'})
    gamma_learning_ratio: float = field(default=1, metadata={"help": 'The learning rate for the gamma parameters'})

    output_dir: str = field(default='./output', metadata={"help": 'The output dir for logs and checkpoints'})
    optim: str = field(default='paged_adamw_32bit', metadata={"help": 'The optimizer to be used'})
    per_device_train_batch_size: int = field(default=1, metadata={"help": 'The training batch size per GPU. Increase for better speed.'})
    per_device_eval_batch_size: int = field(default=1, metadata={"help": 'The evaluation batch size per GPU. Increase for better speed.'})
    gradient_accumulation_steps: int = field(default=16, metadata={"help": 'How many gradients to accumulate before to perform an optimizer step'})
    max_steps: int = field(default=10000, metadata={"help": 'How many optimizer update steps to take'})
    weight_decay: float = field(default=0.0, metadata={"help": 'The L2 weight decay rate of AdamW'}) # use lora dropout instead for regularization if needed
    learning_rate: float = field(default=0.0002, metadata={"help": 'The learnign rate'})
    remove_unused_columns: bool = field(default=False, metadata={"help": 'Removed unused columns. Needed to make this codebase work.'})
    max_grad_norm: float = field(default=0.3, metadata={"help": 'Gradient clipping max norm. This is tuned and works well for all models tested.'})
    gradient_checkpointing: bool = field(default=True, metadata={"help": 'Use gradient checkpointing. You want to use this.'})
    do_train: bool = field(default=True, metadata={"help": 'To train or not to train, that is the question?'})
    lr_scheduler_type: str = field(default='constant', metadata={"help": 'Learning rate schedule. Constant a bit better than cosine, and has advantage for analysis'})
    warmup_ratio: float = field(default=0.03, metadata={"help": 'Fraction of steps to do a warmup for'})
    logging_steps: int = field(default=10, metadata={"help": 'The frequency of update steps after which to log the loss'})
    group_by_length: bool = field(default=True, metadata={"help": 'Group sequences into batches with same length. Saves memory and speeds up training considerably.'})
    save_strategy: str = field(default='steps', metadata={"help": 'When to save checkpoints'})
    save_steps: int = field(default=250, metadata={"help": 'How often to save a model'})
    save_total_limit: int = field(default=40, metadata={"help": 'How many checkpoints to save before the oldest is overwritten'})

    whether_localization: Optional[bool] = field(
        default=False,
        metadata={"help": "Whether to run localization."}
    )

    block_wise: Optional[bool] = field(
        default=False,
        metadata={"help": "Whether to use block-wise for localization."}
    )

@dataclass
class GenerationArguments:
    # For more hyperparameters check:
    # https://huggingface.co/docs/transformers/main_classes/text_generation#transformers.GenerationConfig
    # Length arguments
    max_new_tokens: Optional[int] = field(
        default=256,
        metadata={"help": "Maximum number of new tokens to be generated in evaluation or prediction loops"
                          "if predict_with_generate is set."}
    )
    min_new_tokens : Optional[int] = field(
        default=None,
        metadata={"help": "Minimum number of new tokens to generate."}
    )

    # Generation strategy
    do_sample: Optional[bool] = field(default=False)
    num_beams: Optional[int] = field(default=1)
    num_beam_groups: Optional[int] = field(default=1)
    penalty_alpha: Optional[float] = field(default=None)
    use_cache: Optional[bool] = field(default=True)

    # Hyperparameters for logit manipulation
    temperature: Optional[float] = field(default=1.0)
    top_k: Optional[int] = field(default=50)
    top_p: Optional[float] = field(default=1.0)
    typical_p: Optional[float] = field(default=1.0)
    diversity_penalty: Optional[float] = field(default=0.0)
    repetition_penalty: Optional[float] = field(default=1.0)
    length_penalty: Optional[float] = field(default=1.0)
    no_repeat_ngram_size: Optional[int] = field(default=0)

@dataclass
class ExtraArguments:
    project_name: Optional[str] = field(
        default='PEFT'
    )
    mmlu_freq: Optional[int] = field(
        default=200,
        metadata={"help": "Frequency of MMLU evaluation."}
    )
    lora_drop_prob: Optional[float] = field(
        default=0.0,
        metadata={"help": "Stocastic drop LoRA Layers."}
    )

    layer_json_file: Optional[str] = field(
        default=None,
        metadata={"help": "Layer json file for localization."}
    )

    layer_score_file: Optional[str] = field(
        default=None,
        metadata={"help": "Layer json file for localization."}
    )
    only_permutation: Optional[bool] = field(
        default=True,
        metadata={"help": "Whether to use only permutation."}
    )
    desired_std_ratio: Optional[float] = field(
        default=1.0,
        metadata={"help": "Desired std for lora score."}
    )
    max_learning_rate_ratio: Optional[float] = field(
        default=1.2,
        metadata={"help": "Max learning rate ratio for lora score."}
    )
    min_learning_rate_ratio: Optional[float] = field(
        default=0.5,
        metadata={"help": "Max learning rate ratio for lora score."}
    )
    desired_mean: Optional[float] = field(
        default=1.0,
        metadata={"help": "Desired mean for lora score."}
    )
    dynamic_lr: Optional[bool] = field(
        default=False,
        metadata={"help": "Whether to use dynamic learning rate."}
    )

    topK: Optional[int] = field(
        default=0,
        metadata={"help": "TopK for localization."}
    )
    divide_ratio: Optional[float] = field(
        default=0.0,
        metadata={"help": "Divide ratio for localization. If 0.0, use topK."}
    )
    total_steps: Optional[int] = field(
        default=2000,
        metadata={"help": "Total steps for localization."}
    )
    whether_unfreeze_normal: Optional[bool] = field(
        default=False,
        metadata={"help": "Whether to freeze normal layers."}
    )
    interval: Optional[int] = field(
        default=100,
        metadata={"help": "Interval for training lora"}
    )
    lora_gamma_interval: Optional[int] = field(
        default=1,
        metadata={"help": "Interval for training lora gamma."}
    )
    upperbound: Optional[float] = field(
        default=1.0
    )
    default_localization: Optional[bool] = field(
        default=False,
        metadata={"help": "Whether to use default localization."}
    )
    Random: Optional[bool] = field(
        default=False,
        metadata={"help": "Whether to use random Selection task-specific layers."}
    )
    start_iter: Optional[int] = field(
        default=0,
        metadata={"help": "Start Iteration for training lora"}
    )
    all_layers: Optional[str] = field(
        default='ALL',
        metadata={"help": "Whether to use all layers. ALL, ATT, FFN"}
    )
    ratio: Optional[float] = field(
        default=32.0,
        metadata={"help": "Ratio for Norm Lora layers."}
    )
    use_sigmoid: Optional[bool] = field(
        default=False,
        metadata={"help": "Whether to use sigmoid."}
    )
    sigmoid_init_value: Optional[float] = field(
        default=5.0,
        metadata={"help": "Sigmoid init value."}
    )


def print_trainable_parameters(args, model):
    """
    Prints the number of trainable parameters in the model.
    """
    trainable_params = 0
    all_param = 0
    for _, param in model.named_parameters():
        all_param += param.numel()
        if param.requires_grad:
            trainable_params += param.numel()
    if args.bits == 4: trainable_params /= 2
    print(
        f"trainable params: {trainable_params} || "
        f"all params: {all_param} || "
        f"trainable: {100 * trainable_params / all_param}"
    )

def smart_tokenizer_and_embedding_resize(
    special_tokens_dict: Dict,
    tokenizer: transformers.PreTrainedTokenizer,
    model: transformers.PreTrainedModel,
):
    """Resize tokenizer and embedding.

    Note: This is the unoptimized version that may make your embedding size not be divisible by 64.
    """
    num_new_tokens = tokenizer.add_special_tokens(special_tokens_dict)
    model.resize_token_embeddings(len(tokenizer))
    
    if num_new_tokens > 0:
        input_embeddings_data = model.get_input_embeddings().weight.data
        output_embeddings_data = model.get_output_embeddings().weight.data

        input_embeddings_avg = input_embeddings_data[:-num_new_tokens].mean(dim=0, keepdim=True)
        output_embeddings_avg = output_embeddings_data[:-num_new_tokens].mean(dim=0, keepdim=True)

        input_embeddings_data[-num_new_tokens:] = input_embeddings_avg
        output_embeddings_data[-num_new_tokens:] = output_embeddings_avg

    model.config.pad_token_id = tokenizer.pad_token_id

@dataclass
class DataCollatorForCausalLM(object):
    tokenizer: transformers.PreTrainedTokenizer
    source_max_len: int
    target_max_len: int
    train_on_source: bool
    predict_with_generate: bool

    def __call__(self, instances: Sequence[Dict]) -> Dict[str, torch.Tensor]:
        # Extract elements
        sources = [example['input'] for example in instances]
        targets = [example['output'] for example in instances]
        # Tokenize
        tokenized_sources_with_prompt = self.tokenizer(
            sources,
            add_special_tokens=False,
        )
        tokenized_targets = self.tokenizer(
            targets,
            add_special_tokens=False,
        )
        # Build the input and labels for causal LM
        input_ids = []
        labels = []
        for tokenized_source, tokenized_target in zip(
            tokenized_sources_with_prompt['input_ids'],
            tokenized_targets['input_ids']
        ):
            input_ids.append(torch.tensor(tokenized_source))
            labels.append(torch.tensor(tokenized_target))

        data_dict = {
            'input_ids': input_ids,
            'labels': labels
        }

        return data_dict

UNNATURAL_INSTRUCTIONS_PROMPT_DICT = {
    "prompt_input": (
        "Below is an instruction that describes a task, paired with an input that provides further context. "
        "Write a response that appropriately completes the request.\n\n"
        "### Instruction:\n{instruction}\n\n### Input:\n{input}\n\n### Response: "
    )
}

def extract_unnatural_instructions_data(examples):
    out = {
        'input': None,
        'output': None,
    }
    for example_instances in examples['instances']:
        prompt_format = UNNATURAL_INSTRUCTIONS_PROMPT_DICT["prompt_input"]
        input={
            "instruction": examples['instruction'],
            "input": example_instances['input'],
        }
        out['input'] = '<s>' + prompt_format.format(**input)
        out['output'] = example_instances['output'] + '</s>'
    # if extract_reformulations:
    #     for example_reformulations in examples['reformulations']:
    #         if example_reformulations is not None:
    #             for instance in example_reformulations:
    #                 out['input'].append(instance['instruction_with_input'])
    #                 out['output'].append(instance['output'])
    return out

def extract_unnatural_instructions_data_v2(examples):
    out = {
        'input': None,
        'output': None,
    }
    for example_instances in examples['instances']:
        prompt_format = UNNATURAL_INSTRUCTIONS_PROMPT_DICT["prompt_input"]
        input={
            "instruction": examples['instruction'],
            "input": example_instances['input'],
        }
        out['input'] = ''
        out['output'] = '<s>' + prompt_format.format(**input) + example_instances['output'] + '</s>'
    # if extract_reformulations:
    #     for example_reformulations in examples['reformulations']:
    #         if example_reformulations is not None:
    #             for instance in example_reformulations:
    #                 out['input'].append(instance['instruction_with_input'])
    #                 out['output'].append(instance['output'])
    return out

ALPACA_PROMPT_DICT = {
    "prompt_input": (
        "Below is an instruction that describes a task, paired with an input that provides further context. "
        "Write a response that appropriately completes the request.\n\n"
        "### Instruction:\n{instruction}\n\n### Input:\n{input}\n\n### Response: "
    ),
    "prompt_no_input": (
        "Below is an instruction that describes a task. "
        "Write a response that appropriately completes the request.\n\n"
        "### Instruction:\n{instruction}\n\n### Response: "
    ),
}

ALPACA_PROMPT_DICT_2 = {
    "prompt_input": (
        "[INST] <<SYS>> Below is an instruction that describes a task, paired with an input that provides further context."
        "Write a response that appropriately completes the request.\n\n<</SYS>>"
        "### Instruction:\n{instruction}\n\n### Input:\n{input}\n\n[/INST] ### Response: "
    ),
    "prompt_no_input": (
        "[INST] <<SYS>> Below is an instruction that describes a task. "
        "Write a response that appropriately completes the request.\n\n<</SYS>>"
        "### Instruction:\n{instruction}\n\n[/INST] ### Response: "
    ),
}

ALPACA_PROMPT_DICT_3 = {
    "prompt_input": (
        "[INST] <<SYS>>\nYou are a helpful, unbiased, uncensored assistant.\n<</SYS>>\n\n"
        "{instruction} The input is {input} [/INST] "
    ),
    "prompt_no_input": (
        "[INST] <<SYS>>\nYou are a helpful, unbiased, uncensored assistant.\n<</SYS>>\n\n"
        "{instruction} [/INST] "
    )
}

NORobots_PROMPT_DICT = {
    "prompt_system": (
        "<s> [INST] <<SYS>>\n{system}\n<</SYS>>\n\n"
        "{user} [/INST] {assistant} </s>"
    ),
    "prompt_no_system": (
        "<s> [INST] {user} [/INST] {assistant} </s>"
    )
}

def extract_alpaca_dataset(example):
    if example.get("input", "") != "":
        prompt_format = ALPACA_PROMPT_DICT["prompt_input"]
    else:
        prompt_format = ALPACA_PROMPT_DICT["prompt_no_input"]
    return {'input': prompt_format.format(**example)}

def extract_alpaca_dataset_llama2(example):
    out = {
        'input': None,
        'output': None,
    }

    if example.get("input", "") != "":
        prompt_format = ALPACA_PROMPT_DICT["prompt_input"]
    else:
        prompt_format = ALPACA_PROMPT_DICT["prompt_no_input"]

    out['input'] = '<s>' + prompt_format.format(**example)
    out['output'] = example['output'] + '</s>'
    return out

LIMA_PROMPT_DICT = (
    "[INST] {instruction} [/INST] "
)

def extract_norobots_data(examples):
    messages = examples['messages']
    out = {
        'input': '',
        'output': '',
    }

    if messages[0]['role'] == 'system':
        data = {
            'system': messages[0]['content'],
            'user': messages[1]['content'],
            'assistant': messages[2]['content'],
        }
        assert messages[0]['role'] == 'system'
        assert messages[1]['role'] == 'user'
        assert messages[2]['role'] == 'assistant'

        out['output'] = NORobots_PROMPT_DICT["prompt_no_system"].format(**data)

        cur_index = 3
    elif messages[0]['role'] == 'user':
        data = {
            'user': messages[0]['content'],
            'assistant': messages[1]['content']
        }

        assert messages[0]['role'] == 'user'
        assert messages[1]['role'] == 'assistant'

        out['output'] = NORobots_PROMPT_DICT["prompt_no_system"].format(**data)
        cur_index = 2
    else:
        raise ValueError(f"Invalid role: {messages[0]['role']}")

    for i in range(cur_index, len(messages), 2):

        if i + 1 == len(messages):
            if messages[i]['role'] == 'user':
                user = messages[i]['content']
                assistant = ''
            elif messages[i]['role'] == 'assistant':
                user = ''
                assistant = messages[i]['content']
            else:
                raise ValueError(f"Invalid role: {messages[i]['role']}")
        else:
            user = messages[i]['content']
            assistant = messages[i + 1]['content']
            assert messages[i]['role'] == 'user'
            assert messages[i + 1]['role'] == 'assistant'

        out['output'] += f'<s>[INST] {user} [/INST] {assistant} </s>'

    return out

def extract_lima_data(examples):
    if len(examples['conversations']) == 1:
        Input = {'instruction': examples['conversations'][0]}
        out = {
            'input': None,
            'output': '',
        }
        out['input'] = '<s>' + LIMA_PROMPT_DICT.format(**Input)
    else:
        output = ''
        p = int(len(examples['conversations']) / 2)
        for i in range(p):
            start = examples['conversations'][2 * i]
            end = examples['conversations'][2 * i + 1]
            output += f'<s>[INST] {start} [/INST] {end} </s>'

        out = {
            'input': '',
            'output': output,
        }

    return out


def extract_alpaca_dataset_llama2_v2(example):
    out = {
        'input': "",
        'output': None,
    }

    if example.get("input", "") != "":
        prompt_format = ALPACA_PROMPT_DICT["prompt_input"]
    else:
        prompt_format = ALPACA_PROMPT_DICT["prompt_no_input"]

    input = '<s>' + prompt_format.format(**example)
    output = example['output'] + '</s>'

    out['output'] = input + output

    return out

def local_dataset(dataset_name):
    if dataset_name.endswith('.json') or dataset_name.endswith('.jsonl'):
        full_dataset = Dataset.from_json(path_or_paths=dataset_name)
    elif dataset_name.endswith('.csv'):
        full_dataset = Dataset.from_pandas(pd.read_csv(dataset_name))
    elif dataset_name.endswith('.tsv'):
        full_dataset = Dataset.from_pandas(pd.read_csv(dataset_name, delimiter='\t'))
    else:
        raise ValueError(f"Unsupported dataset format: {dataset_name}")

    split_dataset = full_dataset.train_test_split(test_size=0.1)
    return split_dataset



def make_data_module(tokenizer: transformers.PreTrainedTokenizer, args) -> Dict:
    """
    Make dataset and collator for supervised fine-tuning.
    Datasets are expected to have the following columns: { `input`, `output` }

    Available datasets to be selected with `dataset` argument:
        - alpaca, 52002 examples
        - alpaca cleaned, 51942 examples
        - chip2 (OIG), 210289 examples
        - self-instruct, 82612 examples
        - hh-rlhf (Anthropic), 160800 examples
        - longform, 23.7k examples
        - oasst1 (OpenAssistant) primary message tree only, 9,846 examples

    Coming soon:
        - unnatural instructions core, 66010 examples
        - unnatural instructions full, 240670 examples
        - alpaca-gpt4, 52002 examples
        - unnatural-instructions-gpt4, 9000 examples
        - supernatural-instructions, 69624 examples (same as paper with 100 ex/task more can be used)
        - flan (FLAN v2), up to 20M examples available
        - vicuna
    """
    def load_data(dataset_name):
        if dataset_name == 'alpaca':
            return load_from_disk("tatsu-lab/alpaca")
            # return load_from_disk('./datasets/alpaca')
        elif dataset_name == 'alpaca-gpt4':
            return load_from_disk('./dataset/alpaca-gpt4')
            # return load_dataset("vicgalle/alpaca-gpt4")
        elif dataset_name == 'LIMA':
            return load_from_disk("./dataset/lima")
            # return load_dataset("GAIR/lima")
        elif dataset_name == 'no_robots':
            return load_from_disk("./dataset/no_robots")
        elif dataset_name == 'vicuna':
            raise NotImplementedError("Vicuna data was not released.")
        else:
            if os.path.exists(dataset_name):
                try:
                    args.dataset_format = args.dataset_format if args.dataset_format else "input-output"
                    full_dataset = local_dataset(dataset_name)
                    return full_dataset
                except:
                    raise ValueError(f"Error loading dataset from {dataset_name}")
            else:
                raise NotImplementedError(f"Dataset {dataset_name} not implemented yet.")

    def format_dataset(dataset, dataset_format):
        if (
            dataset_format == 'alpaca' or dataset_format == 'alpaca-clean' or dataset_format == 'alpaca-gpt4' or
            (dataset_format is None and args.dataset in ['alpaca', 'alpaca-clean', 'alpaca-gpt4'])
        ):
            # dataset = dataset.map(extract_alpaca_dataset, remove_columns=['instruction'])
            dataset = dataset.map(extract_alpaca_dataset_llama2, remove_columns=['instruction'])
        elif dataset_format == 'chip2' or (dataset_format is None and args.dataset == 'chip2'):
            dataset = dataset.map(lambda x: {
                'input': x['text'].split('\n<bot>: ')[0].replace('<human>: ', ''),
                'output': x['text'].split('\n<bot>: ')[1],
            })
        elif dataset_format == 'self-instruct' or (dataset_format is None and args.dataset == 'self-instruct'):
            for old, new in [["prompt", "input"], ["completion", "output"]]:
                dataset = dataset.rename_column(old, new)
        elif dataset_format == 'hh-rlhf' or (dataset_format is None and args.dataset == 'hh-rlhf'):
            dataset = dataset.map(lambda x: {
                'input': '',
                'output': x['chosen']
            })
        elif dataset_format == 'LIMA' or (dataset_format is None and args.dataset == 'LIMA'):
            dataset = dataset.map(extract_lima_data)
        elif dataset_format == 'no_robots' or (dataset_format is None and args.dataset == 'no_robots'):
            dataset = dataset.map(extract_norobots_data)
            dataset['train'] = dataset['train_sft']
            dataset['test'] = dataset['test_sft']
        elif dataset_format == 'oasst1' or (dataset_format is None and args.dataset == 'oasst1'):
            dataset = dataset.map(lambda x: {
                'input': '',
                'output': '<s>' + x['text'] + '</s>',
            })
        elif dataset_format == 'unnatural-instructions-core' or (dataset_format is None and args.dataset == 'unnatural-instructions-core'):
            dataset = dataset.map(extract_unnatural_instructions_data)
        elif dataset_format == 'input-output':
            # leave as is
            pass
        # Remove unused columns.
        dataset = dataset.remove_columns(
            [col for col in dataset.column_names['train'] if col not in ['input', 'output']]
        )
        return dataset

     # Load dataset.
    dataset = load_data(args.dataset)
    dataset = format_dataset(dataset, args.dataset_format)
    for i in range(4):
        print(dataset['train'][i])
        print(dataset['train'][-i])
    print(len(dataset['train']))
    # print(dataset[0])

    # Split train/eval, reduce size
    if args.do_eval or args.do_predict:
        if 'eval' in dataset:
            eval_dataset = dataset['eval']
        else:
            print('Splitting train dataset in train and validation according to `eval_dataset_size`')
            dataset = dataset["train"].train_test_split(
                test_size=args.eval_dataset_size, shuffle=True, seed=42
            )
            eval_dataset = dataset['test']
        if args.max_eval_samples is not None and len(eval_dataset) > args.max_eval_samples:
            eval_dataset = eval_dataset.select(range(args.max_eval_samples))
        if args.group_by_length:
            eval_dataset = eval_dataset.map(lambda x: {'length': len(x['input']) + len(x['output'])})
    if args.do_train:
        train_dataset = dataset['train']
        # train_dataset = train_dataset[:args.max_train_samples]
        if args.max_train_samples is not None and len(train_dataset) > args.max_train_samples:
            train_dataset = train_dataset.select(range(args.max_train_samples))
        if args.group_by_length:
            train_dataset = train_dataset.map(lambda x: {'length': len(x['input']) + len(x['output'])})

    data_collator = DataCollatorForCausalLM(
        tokenizer=tokenizer,
        source_max_len=args.source_max_len,
        target_max_len=args.target_max_len,
        train_on_source=args.train_on_source,
        predict_with_generate=args.predict_with_generate,
    )
    return dict(
        train_dataset=train_dataset if args.do_train else None,
        eval_dataset=eval_dataset if args.do_eval else None,
        predict_dataset=eval_dataset if args.do_predict else None,
        data_collator=data_collator
    )


def plot_data_lengths(input_ids, labels):
    lengths_source = [len(x) for x in input_ids]
    lengths_target = [len(x) for x in labels]

    # Plotting the histogram
    plt.figure(figsize=(10, 6))
    plt.hist(lengths_source, bins=20, alpha=0.7, color='blue')
    plt.xlabel('Length of input_ids')
    plt.ylabel('Frequency')
    plt.title('Distribution of Lengths of input_ids')
    plt.show()

    plt.figure(figsize=(10, 6))
    plt.hist(lengths_target, bins=20, alpha=0.7, color='blue')
    plt.xlabel('Length of targets')
    plt.ylabel('Frequency')
    plt.title('Distribution of Lengths of input_ids')
    plt.show()

    ratio = 0.01
    num_data = len(lengths_source)
    pos = int(num_data * ratio)

    lengths_source = sorted(lengths_source)
    lengths_target = sorted(lengths_target)

    print(f"{ratio}: {lengths_source[-pos]}")
    print(f"{ratio}: {lengths_target[-pos]}")

    print(f"max: {max(lengths_source)}")
    print(f"max: {max(lengths_target)}")
    print(f"length: {len(lengths_source)}")

def train():
    hfparser = transformers.HfArgumentParser((
        ModelArguments, DataArguments, TrainingArguments, GenerationArguments, ExtraArguments
    ))
    model_args, data_args, training_args, generation_args, project_args, extra_args = \
        hfparser.parse_args_into_dataclasses(return_remaining_strings=True)
    training_args.generation_config = transformers.GenerationConfig(**vars(generation_args))
    args = argparse.Namespace(
        **vars(model_args), **vars(data_args), **vars(training_args), **vars(project_args)
    )
    print(args)

    print('loaded model')
    set_seed(args.seed)

    tokenizer = AutoTokenizer.from_pretrained(
        args.model_name_or_path,
        add_eos_token=True,
        add_bos_token=True,
    )

    data_module = make_data_module(tokenizer=tokenizer, args=args)

    train_dataset = data_module['train_dataset']
    eval_dataset = data_module['eval_dataset']
    predict_dataset = data_module['predict_dataset']

    data_collator = data_module['data_collator']

    train_dataset_dict = data_collator(train_dataset)

    input_ids = train_dataset_dict['input_ids']
    labels = train_dataset_dict['labels']

    # for input, output in zip(input_ids, labels):
    #     print(tokenizer.decode(input, skip_special_tokens=True))
    #     print(tokenizer.decode(output, skip_special_tokens=True))

    plot_data_lengths(input_ids, labels)

    model = AutoModelForCausalLM.from_pretrained(
        args.model_name_or_path,
        device_map='cpu',
    )

    if tokenizer._pad_token is None:
        smart_tokenizer_and_embedding_resize(
            special_tokens_dict=dict(pad_token=DEFAULT_PAD_TOKEN),
            tokenizer=tokenizer,
            model=model,
        )



if __name__ == "__main__":
    train()
