#!/bin/bash

# Define text colors
GREEN='\033[0;32m'
BLUE='\033[0;34m'
YELLOW='\033[1;33m'
CYAN='\033[0;36m'
NC='\033[0m' # No Color

# Help function
show_help() {
  echo -e "${CYAN}===== Structured Report Generation Script =====${NC}"
  echo "Usage: ./run.sh [options]"
  echo ""
  echo "Options:"
  echo "  --help                Display this help message"
  echo "  --toy_set             Enable toy dataset mode"
  echo "  --mode [mode]         Set processing mode (study_80, etc.)"
  echo "  --model [name]        Specify model name"
  echo "  --api_key [key]       Specify API key for the selected model"
  echo "  --retrieval [num]     Set number of retrievals"
  echo "  --format [format]     Set output format"
  echo "  --usage [value]       Set candidate usage value (0.0-1.0)"
  echo ""
  echo "Examples:"
  echo "  ./run.sh --toy_set --mode study_80 --model gpt-4.1 --api_key sk-xxxx"
  echo "  ./run.sh --mode full --model qwen3-235b-a22b --retrieval 10 --api_key sk-xxxx"
  echo ""
}

# Initialize variables
toy_set=false
mode="test_2studies"
model="gpt-4.1"
api_key=""
n_retrieval=5
output_format="SROSRO"
candidate_usage=0.8

# Process command line arguments
while [[ $# -gt 0 ]]; do
  case $1 in
    --help)
      show_help
      exit 0
      ;;
    --toy_set)
      toy_set=true
      shift
      ;;
    --mode)
      mode="$2"
      shift 2
      ;;
    --model)
      model="$2"
      shift 2
      ;;
    --api_key)
      api_key="$2"
      shift 2
      ;;
    --retrieval)
      n_retrieval="$2"
      shift 2
      ;;
    --format)
      output_format="$2"
      shift 2
      ;;
    --usage)
      candidate_usage="$2"
      shift 2
      ;;
    *)
      echo -e "${YELLOW}Warning: Unknown parameter: $1${NC}"
      shift
      ;;
  esac
done

# Display banner
echo -e "${CYAN}┌────────────────────────────────────────────────┐${NC}"
echo -e "${CYAN}│       Structured Report Generation             │${NC}"
echo -e "${CYAN}└────────────────────────────────────────────────┘${NC}"

# Model selection
echo -e "\n${BLUE}Model Selection:${NC}"
echo "  1) gpt-4.1 (default)"
echo "  2) qwen3-235b-a22b"
echo "  3) deepseek-v3-0324"
echo "  4) llama4-maverick-instruct-basic"
read -p "Select model (default: 1): " model_choice

case ${model_choice:-1} in
  1) model="gpt-4.1" ;;
  2) model="qwen3-235b-a22b" ;;
  3) model="deepseek-v3-0324" ;;
  4) model="llama4-maverick-instruct-basic" ;;
  *) echo -e "${YELLOW}Invalid choice. Setting to default (gpt-4.1).${NC}"; model="gpt-4.1" ;;
esac

# API Key input
if [ -z "$api_key" ]; then
  read -p "Enter API key for $model: " api_key
  if [ -z "$api_key" ]; then
    echo -e "${YELLOW}No API key provided. Exiting.${NC}"
    exit 1
  fi
fi

# Mode selection
echo -e "\n${BLUE}Processing Mode Selection:${NC}"
echo "  1) test_80studies"
echo "  2) test_2studies (default)"
echo "  3) rexval"
echo "  4) gold_eval"
echo "  5) maira_cascade"
echo "  6) maira"
echo "  7) rexerr"
echo "  8) medversa"
echo "  9) rgrg"
echo "  10) cvt2distilgpt2"
read -p "Select mode (default: 2): " mode_choice

case ${mode_choice:-2} in
  1) mode="test_80studies" ;;
  2) mode="test_2studies" ;;
  3) mode="rexval" ;;
  4) mode="gold_eval" ;;
  5) mode="maira_cascade" ;;
  6) mode="maira" ;;
  7) mode="rexerr" ;;
  8) mode="medversa" ;;
  9) mode="rgrg" ;;
  10) mode="cvt2distilgpt2" ;;
  *) echo -e "${YELLOW}Invalid choice. Setting to default (test_2studies).${NC}"; mode="test_2studies" ;;
esac

# Retrieval count
echo -e "\n${BLUE}Number of Retrievals:${NC}"
read -p "Enter number of retrievals (default: 5): " retrieval_input
n_retrieval=${retrieval_input:-5}

# Candidate usage
echo -e "\n${BLUE}Candidate Usage:${NC}"
read -p "Enter candidate usage value (0.0-1.0, default: 1.0): " usage_input
candidate_usage=${usage_input:-1.0}

# Holdout devset option
echo -e "\n${BLUE}Holdout Devset:${NC}"
read -p "Use holdout devset? (y/N): " holdout_choice
use_holdout=false
if [[ -n $holdout_choice ]] && [[ $holdout_choice =~ ^[Yy]$ ]]; then
  use_holdout=true
fi

# Build command
cmd="python singleSR/run.py"

if [ "$toy_set" = true ]; then
  cmd="$cmd --toy_set"
fi

if [ "$use_holdout" = true ]; then
  cmd="$cmd --holdout_devset"
fi

cmd="$cmd --mode $mode --deployment_name $model --api_key $api_key --n_retrieval $n_retrieval --output_format $output_format --candidate_usage $candidate_usage"

# Display execution summary
echo -e "\n${GREEN}=== Execution Settings ====${NC}"
echo -e "${BLUE}Dataset:${NC} $([ "$toy_set" = true ] && echo "Toy Dataset" || echo "Full Dataset")"
echo -e "${BLUE}Mode:${NC} $mode"
echo -e "${BLUE}Model:${NC} $model"
echo -e "${BLUE}Retrievals:${NC} $n_retrieval"
echo -e "${BLUE}Output Format:${NC} $output_format"
echo -e "${BLUE}Candidate Usage:${NC} $candidate_usage"
echo -e "${BLUE}Holdout Devset:${NC} $([ "$use_holdout" = true ] && echo "Yes" || echo "No")"
echo -e "${BLUE}Entity Types:${NC} rf,re,ri,rm"

# Ask for confirmation
echo -e "\n${YELLOW}Executing the following command:${NC}"
echo -e "${CYAN}$cmd${NC}"
read -p "Continue? (Y/n): " confirm

if [[ -n $confirm ]] && [[ $confirm =~ ^[Nn]$ ]]; then
  echo -e "\n${YELLOW}Operation cancelled.${NC}"
  exit 0
fi

# Execute command
echo -e "\n${GREEN}Executing command...${NC}"
eval $cmd

# Completion message
echo -e "\n${GREEN}Operation completed!${NC}"