#!/bin/bash

# Define text colors
GREEN='\033[0;32m'
BLUE='\033[0;34m'
YELLOW='\033[1;33m'
CYAN='\033[0;36m'
RED='\033[0;31m'
NC='\033[0m' # No Color

# Display banner
echo -e "${CYAN}┌────────────────────────────────────────────┐${NC}"
echo -e "${CYAN}│          Metric Evaluation Script          │${NC}"
echo -e "${CYAN}└────────────────────────────────────────────┘${NC}"

# Show initial guidance
echo -e "\n${YELLOW}=== IMPORTANT INFORMATION ===${NC}"
echo -e "This script computes only LunguageScore via lunguagescore.py for single, sequential, or ReXVal."
echo -e ""
echo -e "You will need to provide paths to:"
echo -e "  1. Ground truth report file (CSV format)"
echo -e "  2. Prediction report file (CSV format)"
echo -e ""
echo -e "${RED}NOTE:${NC} Without running the appropriate runner script first, this evaluation may not work correctly."
echo -e "${YELLOW}===========================${NC}"

# Help function
show_help() {
  echo -e "${CYAN}===== LunguageScore Runner =====${NC}"
  echo "Usage: ./3.metric_runner.sh [options]"
  echo ""
  echo "Options:"
  echo "  --help                         Display this help message"
  echo "  --mode [mode]                  Mode: single | sequential | rexval"
  echo "  --model_name [name]            Model name for single/sequential: maira_standard|maira_cascade|rgrg|cvt2distilgpt2|medversa|medgemma|lingshu"
  echo "  --gt_path [path]               Path to gold SR CSV (default: ./dataset/gold_SR.csv)"
  echo "  --benchmark_sr_dir [dir]       Dir with <model_name>_SR.csv (default: ./benchmark_SR)"
  echo "  --rexval_path [path]           Path to structured ReXVal CSV (default: ./benchmark/rexval.csv)"
  echo "  --output [path]                Output CSV (default: auto under ./scores)"
  echo "  --print_head                   Print head of result"
  echo ""
  echo "Examples:"
  echo "  ./3.metric_runner.sh --mode single --model_name medgemma"
  echo "  ./3.metric_runner.sh --mode sequential --model_name medgemma --output scores/custom.csv"
  echo "  ./3.metric_runner.sh --mode rexval --rexval_path benchmark/rexval.csv --print_head"
  echo ""
}

# Initialize variables
gt_path=""
benchmark_sr_dir=""
rexval_path=""
output_path=""
mode=""
model_name=""
print_head=false

# Process command line arguments
while [[ $# -gt 0 ]]; do
  case $1 in
    --help)
      show_help
      exit 0
      ;;
    --gt_path)
      gt_path="$2"
      shift 2
      ;;
    --benchmark_sr_dir)
      benchmark_sr_dir="$2"
      shift 2
      ;;
    --rexval_path)
      rexval_path="$2"
      shift 2
      ;;
    --mode)
      mode="$2"
      shift 2
      ;;
    --model_name)
      model_name="$2"
      shift 2
      ;;
    --output)
      output_path="$2"
      shift 2
      ;;
    --print_head)
      print_head=true
      shift 1
      ;;
    *)
      echo -e "${YELLOW}Warning: Unknown parameter: $1${NC}"
      shift
      ;;
  esac
done

# Ask for mode if not provided
if [ -z "$mode" ]; then
  echo -e "\n${BLUE}Evaluation Mode Selection:${NC}"
  echo "  1) single"
  echo "  2) sequential"
  echo "  3) rexval"
  
  read -p "Select mode number (1-3): " mode_choice
  
  case $mode_choice in
    1) mode="single" ;;
    2) mode="sequential" ;;
    3) mode="rexval" ;;
    *) 
      echo -e "${RED}Invalid choice. Please select 1-3.${NC}"
      exit 1
      ;;
  esac
fi

# Defaults and validation per mode
if [[ "$mode" == "single" || "$mode" == "sequential" ]]; then
  gt_path=${gt_path:-"./dataset/gold_SR.csv"}
  benchmark_sr_dir=${benchmark_sr_dir:-"./benchmark_SR"}
  if [ -z "$model_name" ]; then
    echo -e "\n${BLUE}Select model_name:${NC}"
    echo "  1) maira_standard"
    echo "  2) maira_cascade"
    echo "  3) rgrg"
    echo "  4) cvt2distilgpt2"
    echo "  5) medversa"
    echo "  6) medgemma"
    echo "  7) lingshu"
    read -p "Select number (1-7): " model_choice
    case $model_choice in
      1) model_name="maira_standard";;
      2) model_name="maira_cascade";;
      3) model_name="rgrg";;
      4) model_name="cvt2distilgpt2";;
      5) model_name="medversa";;
      6) model_name="medgemma";;
      7) model_name="lingshu";;
      *) echo -e "${RED}Invalid choice.${NC}"; exit 1;;
    esac
  fi
  cmd="python lunguagescore.py --mode $mode --model_name $model_name --gt_path \"$gt_path\" --benchmark_sr_dir \"$benchmark_sr_dir\""
elif [[ "$mode" == "rexval" ]]; then
  rexval_path=${rexval_path:-"./benchmark/rexval.csv"}
  cmd="python lunguagescore.py --mode rexval --rexval_path \"$rexval_path\""
else
  echo -e "${RED}Unsupported mode: $mode${NC}"
  exit 1
fi

# Append optional output and print_head
if [ -n "$output_path" ]; then
  cmd+=" --output \"$output_path\""
fi
if [ "$print_head" = true ]; then
  cmd+=" --print_head"
fi

# Display execution summary
echo -e "\n${GREEN}=== Execution Settings ====${NC}"
echo -e "${BLUE}Mode:${NC} $mode"
if [[ "$mode" == "single" || "$mode" == "sequential" ]]; then
  echo -e "${BLUE}Model:${NC} $model_name"
  echo -e "${BLUE}GT Path:${NC} $gt_path"
  echo -e "${BLUE}Benchmark SR Dir:${NC} $benchmark_sr_dir"
else
  echo -e "${BLUE}ReXVal Path:${NC} $rexval_path"
fi
if [ -n "$output_path" ]; then
  echo -e "${BLUE}Output:${NC} $output_path"
fi

# Ask for confirmation
echo -e "\n${YELLOW}Executing the following command:${NC}"
echo -e "${CYAN}$cmd${NC}"
read -p "Continue? (Y/n): " confirm

if [[ -n $confirm ]] && [[ $confirm =~ ^[Nn]$ ]]; then
  echo -e "\n${YELLOW}Operation cancelled.${NC}"
  exit 0
fi

# Execute command
echo -e "\n${GREEN}Executing command...${NC}"
eval $cmd

# Completion message
echo -e "\n${GREEN}Operation completed!${NC}"