import hashlib
import secrets
from abc import abstractmethod


class CryptoRandomGenerator:
    @abstractmethod
    def randbelow(self, n: int) -> int:
        pass

    @abstractmethod
    def randbits(self, k: int) -> int:
        pass

    def randbytes(self, k: int) -> bytes:
        output = b""
        for _ in range(k):
            output += self.randbits(8).to_bytes(1, "big")
        return output


class SecureCryptoRandomGenerator(CryptoRandomGenerator):
    def randbelow(self, n: int) -> int:
        return secrets.randbelow(n)

    def randbits(self, k: int) -> bool:
        return secrets.randbits(k)


class DeterministicCryptoRandomGenerator(CryptoRandomGenerator):
    def __init__(self, seed: int = 0):
        seed_bytes = seed.to_bytes(256, "big")
        self.state = hashlib.sha256(seed_bytes).digest()
        self.ptr = 0

    def _next_bit(self) -> int:
        if self.ptr == 256:
            # Compute next state
            self.state = hashlib.sha256(self.state).digest()
            self.ptr = -1
        byte_idx = self.ptr // 8
        bit_idx = self.ptr % 8
        bit = (self.state[byte_idx] >> bit_idx) & 1
        self.ptr += 1
        return bit

    def randbelow(self, n: int) -> int:
        num_bits = n.bit_length()
        r = self.randbits(num_bits)
        while r >= n:
            # Rejection sampling
            r = self.randbits(num_bits)
        return r

    def randbits(self, k: int) -> int:
        r = 0
        for i in range(k):
            r |= self._next_bit() << i
        return r
