from __future__ import print_function
import argparse
import os
import torch
import torch.nn as nn
import torch.nn.functional as F
import torch.optim as optim
from torchvision import datasets, transforms
from torch.optim.lr_scheduler import StepLR
import random
import matplotlib.pyplot as plt
import numpy as np


class Net(nn.Module):
    def __init__(self):
        super(Net, self).__init__()
        self.fc1 = nn.Linear(28 * 28, 1024)
        self.fc2 = nn.Linear(1024, 10)

    def forward(self, x):
        x = torch.flatten(x, 1)
        x = self.fc1(x)
        x = F.relu(x)
        x = self.fc2(x)
        output = F.log_softmax(x, dim=1)
        return output


class CNN(nn.Module):
    def __init__(self):
        super(CNN, self).__init__()
        self.conv1 = nn.Conv2d(1, 32, 3, 1)
        self.conv2 = nn.Conv2d(32, 64, 3, 1)
        self.fc1 = nn.Linear(9216, 128)
        self.fc2 = nn.Linear(128, 10)

    def forward(self, x):
        x = self.conv1(x)
        x = F.relu(x)
        x = self.conv2(x)
        x = F.relu(x)
        x = F.max_pool2d(x, 2)
        x = torch.flatten(x, 1)
        x = self.fc1(x)
        x = F.relu(x)
        x = self.fc2(x)
        output = F.log_softmax(x, dim=1)
        return output


def generate_label_noise(s, noise_level):
    p = torch.rand(size=(s,)).cuda()
    rand_ind = torch.randint(low=0, high=10, size=(s,)).cuda()
    label_noise = (p < noise_level) * rand_ind
    return label_noise


def train(args, model, device, train_loader, optimizer, epoch, label_noise, classes=10):
    model.train()
    train_loss = 0
    correct = 0

    lossfunc = nn.CrossEntropyLoss()

    for batch_idx, (data, target) in enumerate(train_loader):
        data, target = data.to(device), target.to(device)
        if args.train_type == "signal":
            target = torch.nn.functional.one_hot(target, num_classes=classes)
        elif args.train_type == "noisy_label":
            noisy_target_ori = (target + label_noise[
                                         batch_idx * args.batch_size:batch_idx * args.batch_size + data.size()[0]]) % 10
            noisy_target = noisy_target_ori
        elif args.train_type == "pure_noise":
            target = torch.ones(classes).cuda() / classes
        else:
            raise NameError
        optimizer.zero_grad()
        output = model(data)
        loss = lossfunc(output, noisy_target)

        pred = output.argmax(dim=1, keepdim=True)
        correct += pred.eq(noisy_target_ori.view_as(pred)).sum().item()
        train_loss += loss.item() * args.batch_size

        loss.backward()
        optimizer.step()
        if batch_idx % args.log_interval == 0:
            print('Train Epoch: {} [{}/{} ({:.0f}%)]\tLoss: {:.6f}'.format(
                epoch, batch_idx * len(data), len(train_loader.dataset),
                       100. * batch_idx / len(train_loader), loss.item()))
            if args.dry_run:
                break

    return train_loss / len(train_loader.dataset), 100 - 100. * correct / len(train_loader.dataset)


def test(args, model, device, test_loader):
    model.eval()
    test_loss = 0
    correct = 0

    lossfunc = nn.CrossEntropyLoss()

    with torch.no_grad():
        for data, target in test_loader:
            data, target = data.to(device), target.to(device)
            target_ori = target
            target = target_ori
            output = model(data)
            test_loss += lossfunc(output, target).item() * args.batch_size  # sum up batch loss
            pred = output.argmax(dim=1, keepdim=True)  # get the index of the max log-probability
            correct += pred.eq(target_ori.view_as(pred)).sum().item()

    test_loss /= len(test_loader.dataset)

    print('\nTest set: Average loss: {:.4f}, Accuracy: {}/{} ({:.0f}%)\n'.format(
        test_loss, correct, len(test_loader.dataset),
        100. * correct / len(test_loader.dataset)))

    return test_loss, 100 - 100. * correct / len(test_loader.dataset)


def plot_err(epochs, train_err, test_err, path, title):
    fig, ax = plt.subplots()
    ax2 = ax.twinx()
    ax.set_ylim(0, 60)
    ax2.set_ylim(0, 60)
    x = [i + 1 for i in range(epochs)]

    line1 = ax.plot(x, test_err[0:epochs], label="Test Error", color='#1f77b4', linewidth=2.5)
    line2 = ax2.plot(x, train_err[0:epochs], label="Training Error", color='#ff7f0e', linewidth=2.5)

    ax.set_xlabel('Epochs', fontsize=25)
    ax.set_ylabel('Test Error', fontsize=25)
    ax2.set_ylabel('Training Error', fontsize=25)

    fig.legend(loc=1, bbox_to_anchor=(1, 1), bbox_transform=ax.transAxes, fontsize=20)

    ax.spines['bottom'].set_linewidth(1.5)
    ax.spines['top'].set_linewidth(1.5)
    ax.spines['left'].set_linewidth(1.5)
    ax.spines['right'].set_linewidth(1.5)

    fig.set_size_inches(8, 6)

    plt.savefig(path + title, dpi=300)
    plt.cla()


def save_and_plot(args, dir, train_err_list, train_loss_list, test_err_list, test_loss_list):
    np.save(dir + '/train_err.npy', train_err_list)
    np.save(dir + '/test_err.npy', test_err_list)

    plot_err(args.epochs, train_err_list, test_err_list, dir, '/err.png')


def plot_mnist(dir, epochs):
    train_err_list = np.load(dir + '/train_err.npy')
    test_err_list = np.load(dir + '/test_err.npy')

    plot_err(epochs, train_err_list, test_err_list, dir, '/err.png')


def main():
    # Training settings
    parser.add_argument('--batch-size', type=int, default=1024, metavar='N',
                        help='input batch size for training (default: 64)')
    parser.add_argument('--test-batch-size', type=int, default=1000, metavar='N',
                        help='input batch size for testing (default: 1000)')
    parser.add_argument('--epochs', type=int, default=500, metavar='N',
                        help='number of epochs to train (default: 14)')
    parser.add_argument('--lr', type=float, default=0.5, metavar='LR',
                        help='learning rate (default: 0.5)')
    parser.add_argument('--gamma', type=float, default=0.995, metavar='M',
                        help='Learning rate step gamma (default: 0.7)')
    parser.add_argument('--no-cuda', action='store_true', default=False,
                        help='disables CUDA training')
    parser.add_argument('--gpuid', type=str, default='0')
    parser.add_argument('--dry-run', action='store_true', default=False,
                        help='quickly check a single pass')
    parser.add_argument('--seed', type=int, default=1, metavar='S',
                        help='random seed (default: 1)')
    parser.add_argument('--log-interval', type=int, default=10, metavar='N',
                        help='how many batches to wait before logging training status')
    parser.add_argument('--save-model', action='store_true', default=False,
                        help='For Saving the current Model')
    parser.add_argument('--dir', type=str, default='./mnist/',
                        help='For Saving the current Model, direction dir')
    parser.add_argument('--train-type', type=str, default='noisy_label',
                        help='noisy_label, signal, pure_noise')
    parser.add_argument('--noise-level', type=float, default=0.2,
                        help='noise_level, used in the noisy label training')
    args = parser.parse_args()
    use_cuda = not args.no_cuda and torch.cuda.is_available()

    torch.manual_seed(args.seed)

    device = torch.device("cuda" if use_cuda else "cpu")
    torch.cuda.set_device(int(args.gpuid))

    dir = args.dir

    if not os.path.exists(dir):
        os.makedirs(dir, exist_ok=True)

    train_kwargs = {'batch_size': args.batch_size}
    test_kwargs = {'batch_size': args.test_batch_size}
    if use_cuda:
        cuda_kwargs = {'num_workers': 1,
                       'pin_memory': True,
                       'shuffle': False}
        train_kwargs.update(cuda_kwargs)
        test_kwargs.update(cuda_kwargs)

    transform = transforms.Compose([
        transforms.ToTensor(),
        transforms.Normalize((0.1307,), (0.3081,))
    ])
    dataset1 = datasets.MNIST('./data', train=True, download=True,
                              transform=transform)
    dataset2 = datasets.MNIST('./data', train=False,
                              transform=transform)
    train_loader = torch.utils.data.DataLoader(dataset1, **train_kwargs)
    test_loader = torch.utils.data.DataLoader(dataset2, **test_kwargs)

    label_noise = generate_label_noise(60000, args.noise_level)

    # model = Net().to(device)
    model = CNN().to(device)

    optimizer = optim.SGD(model.parameters(), lr=args.lr)

    scheduler = StepLR(optimizer, step_size=1, gamma=args.gamma)

    train_loss_list = []
    train_err_list = []
    test_loss_list = []
    test_err_list = []
    for epoch in range(1, args.epochs + 1):
        train_loss, train_err = train(args, model, device, train_loader, optimizer, epoch, label_noise)
        test_loss, test_err = test(args, model, device, test_loader)
        scheduler.step()
        train_loss_list.append(train_loss)
        train_err_list.append(train_err)
        test_loss_list.append(test_loss)
        test_err_list.append(test_err)
    if args.save_model:
        torch.save(model.state_dict(), "mnist.pt")

    save_and_plot(args, dir, train_err_list, train_loss_list, test_err_list, test_loss_list)


if __name__ == '__main__':
    main()
