# YOLOv5 🚀 by Ultralytics, GPL-3.0 license
"""
Run inference on images, videos, directories, streams, etc.

Usage - sources:
    $ python path/to/detect.py --weights yolov5s.pt --source 0              # webcam
                                                             img.jpg        # image
                                                             vid.mp4        # video
                                                             path/          # directory
                                                             path/*.jpg     # glob
                                                             'https://youtu.be/Zgi9g1ksQHc'  # YouTube
                                                             'rtsp://example.com/media.mp4'  # RTSP, RTMP, HTTP stream

Usage - formats:
    $ python path/to/detect.py --weights yolov5s.pt                 # PyTorch
                                         yolov5s.torchscript        # TorchScript
                                         yolov5s.onnx               # ONNX Runtime or OpenCV DNN with --dnn
                                         yolov5s.xml                # OpenVINO
                                         yolov5s.engine             # TensorRT
                                         yolov5s.mlmodel            # CoreML (MacOS-only)
                                         yolov5s_saved_model        # TensorFlow SavedModel
                                         yolov5s.pb                 # TensorFlow GraphDef
                                         yolov5s.tflite             # TensorFlow Lite
                                         yolov5s_edgetpu.tflite     # TensorFlow Edge TPU
"""

import argparse
import os
import sys
from pathlib import Path
import platform

import cv2
import torch
import torch.backends.cudnn as cudnn

FILE = Path(__file__).resolve()
ROOT = FILE.parents[0]  # YOLOv5 root directory
if str(ROOT) not in sys.path:
    sys.path.append(str(ROOT))  # add ROOT to PATH
ROOT = Path(os.path.relpath(ROOT, Path.cwd()))  # relative

from models.common import DetectMultiBackend
from utils.datasets import IMG_FORMATS, VID_FORMATS, LoadImages, LoadStreams
from utils.general import (LOGGER, check_file, check_img_size, check_imshow, check_requirements, colorstr,
                           increment_path, non_max_suppression, print_args, scale_coords, strip_optimizer, xyxy2xywh)
from utils.plots import Annotator, colors, save_one_box
from utils.torch_utils import select_device, time_sync

from models.gradcam import YOLOV5LayerCAM, YOLOV5CALayerCAM
from models.yolo_v5_object_detector import YOLOV5TorchObjectDetector

from utils.metrics import bbox_iou
import numpy as np

  
names = ['person', 'bicycle', 'car', 'motorcycle', 'airplane', 'bus', 'train', 'truck', 'boat', 'traffic light',
         'fire hydrant', 'stop sign', 'parking meter', 'bench', 'bird', 'cat', 'dog', 'horse', 'sheep', 'cow',
         'elephant', 'bear', 'zebra', 'giraffe', 'backpack', 'umbrella', 'handbag', 'tie', 'suitcase', 'frisbee',
         'skis', 'snowboard', 'sports ball', 'kite', 'baseball bat', 'baseball glove', 'skateboard', 'surfboard',
         'tennis racket', 'bottle', 'wine glass', 'cup', 'fork', 'knife', 'spoon', 'bowl', 'banana', 'apple',
         'sandwich', 'orange', 'broccoli', 'carrot', 'hot dog', 'pizza', 'donut', 'cake', 'chair', 'couch',
         'potted plant', 'bed', 'dining table', 'toilet', 'tv', 'laptop', 'mouse', 'remote', 'keyboard', 'cell phone',
         'microwave', 'oven', 'toaster', 'sink', 'refrigerator', 'book', 'clock', 'vase', 'scissors', 'teddy bear',
         'hair drier', 'toothbrush']
# @torch.no_grad()
def run(weights=ROOT / 'yolov5s.pt',  # model.pt path(s)
        source=ROOT / 'data/images',  # file/dir/URL/glob, 0 for webcam
        data=ROOT / 'data/coco128.yaml',  # dataset.yaml path
        imgsz=(640, 640),  # inference size (height, width)
        conf_thres=0.45,  # confidence threshold
        iou_thres=0.45,  # NMS IOU threshold
        max_det=1000,  # maximum detections per image
        device='',  # cuda device, i.e. 0 or 0,1,2,3 or cpu
        view_img=False,  # show results
        save_txt=False,  # save results to *.txt
        save_conf=False,  # save confidences in --save-txt labels
        save_crop=False,  # save cropped prediction boxes
        nosave=False,  # do not save images/videos
        classes=None,  # filter by class: --class 0, or --class 0 2 3
        agnostic_nms=False,  # class-agnostic NMS
        augment=False,  # augmented inference
        visualize=False,  # visualize features
        update=False,  # update all models
        project=ROOT / 'runs/detect',  # save results to project/name
        name='exp',  # save results to project/name
        exist_ok=False,  # existing project/name ok, do not increment
        line_thickness=3,  # bounding box thickness (pixels)
        hide_labels=False,  # hide labels
        hide_conf=False,  # hide confidences
        half=False,  # use FP16 half-precision inference
        dnn=False,  # use OpenCV DNN for ONNX inference
        trigger_detect = True,
        CA = True
        ):
    source = str(source)
    save_img = not nosave and not source.endswith('.txt')  # save inference images
    is_file = Path(source).suffix[1:] in (IMG_FORMATS + VID_FORMATS)
    is_url = source.lower().startswith(('rtsp://', 'rtmp://', 'http://', 'https://'))
    webcam = source.isnumeric() or source.endswith('.txt') or (is_url and not is_file)
    if is_url and is_file:
        source = check_file(source)  # download

    # Directories
    save_dir = increment_path(Path(project) / name, exist_ok=exist_ok)  # increment run
    (save_dir / 'labels' if save_txt else save_dir).mkdir(parents=True, exist_ok=True)  # make dir

    # Load model
    device = select_device(device)
    model = YOLOV5TorchObjectDetector(weights, device, imgsz, names=names, confidence=conf_thres, iou_thresh=iou_thres)
    if CA:
        saliency_method = YOLOV5CALayerCAM(model=model, layer_name='model_8_cv3_act', num_class = 80, target_cls = {0})
    else:
        saliency_method = YOLOV5LayerCAM(model=model, layer_name='model_8_cv3_act', layercam=True)

    # Dataloader
    if webcam:
        view_img = check_imshow()
        cudnn.benchmark = True  # set True to speed up constant image size inference
        dataset = LoadStreams(source, img_size=imgsz)
        bs = len(dataset)  # batch_size
    else:
        dataset = LoadImages(source, img_size=imgsz)
        bs = 1  # batch_size
    vid_path, vid_writer = [None] * bs, [None] * bs

    # Run inference
    dt, seen, windows = [0.0, 0.0, 0.0], 0, []

    for path, im, im0s, vid_cap, s in dataset: 
        torch.cuda.empty_cache()
        t1 = time_sync()
        im = torch.from_numpy(im).to(device)
        im = im.half() if half else im.float()  # uint8 to fp16/32
        im /= 255  # 0 - 255 to 0.0 - 1.0
        if len(im.shape) == 3:
            im = im[None]  # expand for batch dim
        t2 = time_sync()
        dt[0] += t2 - t1

        # Inference
        seen += 1
        if webcam:  # batch_size >= 1
            p, im0, frame = path[0], im0s[0].copy(), dataset.count
            s += f'{0}: '
        else:
            p, im0, frame = path, im0s.copy(), getattr(dataset, 'frame', 0)
        p = Path(p)  # to Path
        save_path = str(save_dir / p.name)  # im.jpg
        txt_path = str(save_dir / 'labels' / p.stem) + ('' if dataset.mode == 'image' else f'_{frame}')  # im.txt
        s += '%gx%g ' % im.shape[2:]  # print string
        gn = torch.tensor(im0.shape)[[1, 0, 1, 0]]  # normalization gain whwh
        imc = im0.copy() if save_crop else im0  # for save_crop
        annotator = Annotator(im0, line_width=line_thickness, example=str(names))
        masks, logits, [boxes, classes, class_names, conf] = saliency_method(im)  # 得到预测结果     

        t3 = time_sync()
        dt[1] += t3 - t2

        if len(conf[0]):
            for c in classes[0]:
                n = (torch.tensor(classes[0]) == c).sum()  # detections per class
                s += f"{n} {names[int(c)]}{'s' * (n > 1)}, "  # add to string

            boxes = list(scale_coords(im.shape[2:], torch.tensor(boxes[0], dtype = torch.float64), im0.shape).round())
            for i, (sm, box, cf, class_name, c) in enumerate(zip(masks, boxes, conf[0], class_names[0], classes[0])):
                # detect trigger!
                if trigger_detect:
                    trigger_flag = False
                    true_label = c
                    if CA:
                        for k,v in sm.items():
                            mask = v.cpu().data.numpy()

                            ind = np.argwhere(mask == 1)
                            mask = mask[0][0]

                            _,binary = cv2.threshold(mask, 0.25, 1.01, cv2.THRESH_BINARY)
                            contours, _ = cv2.findContours(binary.astype(np.uint8),cv2.RETR_EXTERNAL,cv2.CHAIN_APPROX_SIMPLE)
                            for contour in contours:
                                (x,y,w,h) = cv2.boundingRect(contour)
                                xyxy = scale_coords(im0.shape, box.clone().unsqueeze(0), im.shape[2:]).round()[0]
                                if ind[0,2]<y or ind[0,2]>y+h or ind[0,3]<x or ind[0,3]>x+w:
                                    continue
                                cx = int(x+w//2)
                                cy = int(y+h//2)
                                bw = int(min(max(w, (xyxy[2] - xyxy[0]) * 0.2), (xyxy[2] - xyxy[0]) * 0.3)) 
                                bh = int(min(max(h, (xyxy[3] - xyxy[1]) * 0.2), (xyxy[3] - xyxy[1]) * 0.3))       

                                im_temp = im.clone()   
                                im_temp[:,:,cy-bh//2:cy+bh//2,cx-bw//2:cx+bw//2] = 114/255

                                with torch.no_grad():
                                    im_temp = torch.tensor(cv2.blur(im_temp.squeeze().permute(1,2,0).cpu().data.numpy(),(3,3)),dtype=torch.float32).to(device).unsqueeze(0).permute(0,3,1,2)
                                    [boxes_, classes_, _, _], _ = model(im_temp)
                                if not boxes_[0]:
                                    break
                                boxes_ = list(scale_coords(im.shape[2:], torch.tensor(boxes_[0], dtype = torch.float64), im0.shape).round())


                                cnt = 0
                                for box_, c_ in zip(boxes_, classes_[0]):
                                    iou = bbox_iou(box_, box, x1y1x2y2 = True, CIoU = True)
                                    if iou > 0.45:
                                        cnt += 1
                                        if int(c_) == int(k):
                                            true_label = int(c_)
                                            print('change label')
                                            trigger_box = scale_coords(im.shape[2:], torch.tensor([cx-bw//2,cy-bh//2,cx+bw//2,cy+bh//2], dtype = torch.float64).unsqueeze(0), im0.shape).round()[0]
                                            trigger_flag = True
                                if trigger_flag and (cnt == 1 or len(classes_[0])>len(classes[0])):
                                    print("trigger detect!!!")
                                    c = true_label
                                    annotator.box_label(trigger_box.numpy(), "trigger", color=colors(81, True))
                                    break 
                            if trigger_flag and (cnt == 1 or len(classes_[0])>len(classes[0])):
                                break    
                    else:
                        mask = sm.cpu().data.numpy()
                        mask = mask[0][0]

                        _,binary = cv2.threshold(mask, 0.25, 1.01, cv2.THRESH_BINARY)
                        contours, _ = cv2.findContours(binary.astype(np.uint8),cv2.RETR_EXTERNAL,cv2.CHAIN_APPROX_SIMPLE)
                        for contour in contours:
                            (x,y,w,h) = cv2.boundingRect(contour)
                            xyxy = scale_coords(im0.shape, box.clone().unsqueeze(0), im.shape[2:]).round()[0]

                            cx = int(x+w//2)
                            cy = int(y+h//2)
                            bw = int(min(max(w, (xyxy[2] - xyxy[0]) * 0.2), (xyxy[2] - xyxy[0]) * 0.3)) 
                            bh = int(min(max(h, (xyxy[3] - xyxy[1]) * 0.2), (xyxy[3] - xyxy[1]) * 0.3))       

                            im_temp = im.clone()   
                            im_temp[:,:,cy-bh//2:cy+bh//2,cx-bw//2:cx+bw//2] = 114/255

                            with torch.no_grad():
                                im_temp = torch.tensor(cv2.blur(im_temp.squeeze().permute(1,2,0).cpu().data.numpy(),(3,3)),dtype=torch.float32).to(device).unsqueeze(0).permute(0,3,1,2)
                                [boxes_, classes_, _, _], _ = model(im_temp)
                            if not boxes_[0]:
                                break
                            boxes_ = list(scale_coords(im.shape[2:], torch.tensor(boxes_[0], dtype = torch.float64), im0.shape).round())


                            cnt = 0
                            for box_, c_ in zip(boxes_, classes_[0]):
                                iou = bbox_iou(box_, box, x1y1x2y2 = True, CIoU = True)
                                if iou > 0.45:
                                    cnt += 1
                                    if int(c_) != int(c):
                                        true_label = int(c_)
                                        print('change label')
                                        trigger_box = scale_coords(im.shape[2:], torch.tensor([cx-bw//2,cy-bh//2,cx+bw//2,cy+bh//2], dtype = torch.float64).unsqueeze(0), im0.shape).round()[0]
                                        trigger_flag = True
                            if trigger_flag and (cnt == 1 or len(classes_[0])>len(classes[0])):
                                print("trigger detect!!!")
                                c = true_label
                                annotator.box_label(trigger_box.numpy(), "trigger", color=colors(81, True))
                                break     
                #detect trigger over!

                if save_txt:  # Write to file
                    xywh = (xyxy2xywh(torch.tensor(box).view(1, 4)) / gn).view(-1).tolist()  # normalized xywh
                    line = (c, *xywh, cf) if save_conf else (c, *xywh)  # label format
                    with open(txt_path + '.txt', 'a') as f:
                        f.write(('%g ' * len(line)).rstrip() % line + '\n')

                if save_img or save_crop or view_img:  # Add bbox to image
                    c = int(c)  # integer class
                    # if names[c] == 'person' or names[c] == 'cup':
                    #     label = None if hide_labels else (names[c] if hide_conf else f'{names[c]} {cf:.2f}')
                    if True:
                        label = names[c]
                        annotator.box_label(box, label, color=colors(c, True))
                    if save_crop:
                        save_one_box(box, imc, file=save_dir / 'crops' / names[c] / f'{p.stem}.jpg', BGR=True)

            # Stream results
            im0 = annotator.result()
            view_img = True
            if view_img:
                if platform.system() == 'Linux' and p not in windows:
                    windows.append(p)
                    cv2.namedWindow(str(p), cv2.WINDOW_NORMAL | cv2.WINDOW_KEEPRATIO)  # allow window resize (Linux)
                    cv2.resizeWindow(str(p), im0.shape[1], im0.shape[0])
                cv2.imshow(str(p), im0)
                cv2.waitKey(1)  # 1 millisecond
            # Save results (image with detections)
            if save_img:
                if dataset.mode == 'image':
                    cv2.imwrite(save_path, im0)
                else:  # 'video' or 'stream'
                    if vid_path[0] != save_path:  # new video
                        vid_path[0] = save_path
                        if isinstance(vid_writer[0], cv2.VideoWriter):
                            vid_writer[0].release()  # release previous video writer
                        if vid_cap:  # video
                            fps = vid_cap.get(cv2.CAP_PROP_FPS)
                            w = int(vid_cap.get(cv2.CAP_PROP_FRAME_WIDTH))
                            h = int(vid_cap.get(cv2.CAP_PROP_FRAME_HEIGHT))
                        else:  # stream
                            fps, w, h = 30, im0.shape[1], im0.shape[0]
                        save_path = str(Path(save_path).with_suffix('.mp4'))  # force *.mp4 suffix on results videos
                        vid_writer[0] = cv2.VideoWriter(save_path, cv2.VideoWriter_fourcc(*'mp4v'), fps, (w, h))
                    vid_writer[0].write(im0)
        t4 = time_sync()
        dt[2] += t4 - t2
        # Print time (inference-only)
        LOGGER.info(f'{s}Done. ({t4 - t2:.3f}s)')


    # Print results
    t = tuple(x / seen * 1E3 for x in dt)  # speeds per image
    LOGGER.info(f'Speed: %.1fms pre-process, %.1fms inference, %.1fms total per image at shape {(1, 3, *imgsz)}' % t)
    if save_txt or save_img:
        s = f"\n{len(list(save_dir.glob('labels/*.txt')))} labels saved to {save_dir / 'labels'}" if save_txt else ''
        LOGGER.info(f"Results saved to {colorstr('bold', save_dir)}{s}")
    if update:
        strip_optimizer(weights)  # update model (to fix SourceChangeWarning)


def parse_opt():
    parser = argparse.ArgumentParser()
    parser.add_argument('--weights', nargs='+', type=str, default=ROOT / 'weights/yolov5s_100e_256b_pre.pt',
                        help='model path(s)')
    # parser.add_argument('--source', type=str, default=ROOT / 'data/odsrihs', help='file/dir/URL/glob, 0 for webcam')
    parser.add_argument('--source', type=str, default='rtsp://admin:13134934677lixin@192.168.131.210:554/h264/ch1/main/av_stream', help='file/dir/URL/glob, 0 for webcam')
    parser.add_argument('--data', type=str, default=ROOT / 'data/coco128.yaml', help='(optional) dataset.yaml path')
    parser.add_argument('--imgsz', '--img', '--img-size', nargs='+', type=int, default=[640], help='inference size h,w')
    parser.add_argument('--conf-thres', type=float, default=0.45, help='confidence threshold')
    parser.add_argument('--iou-thres', type=float, default=0.45, help='NMS IoU threshold')
    parser.add_argument('--max-det', type=int, default=1000, help='maximum detections per image')
    parser.add_argument('--device', default='0', help='cuda device, i.e. 0 or 0,1,2,3 or cpu')
    parser.add_argument('--view-img', action='store_true', help='show results')
    parser.add_argument('--save-txt', action='store_true', help='save results to *.txt')
    parser.add_argument('--save-conf', action='store_true', help='save confidences in --save-txt labels')
    parser.add_argument('--save-crop', action='store_true', help='save cropped prediction boxes')
    parser.add_argument('--nosave', action='store_true', help='do not save images/videos')
    parser.add_argument('--classes', nargs='+', type=int, help='filter by class: --classes 0, or --classes 0 2 3')
    parser.add_argument('--agnostic-nms', action='store_true', help='class-agnostic NMS')
    parser.add_argument('--augment', action='store_true', help='augmented inference')
    parser.add_argument('--visualize', action='store_true', help='visualize features')
    parser.add_argument('--update', action='store_true', help='update all models')
    parser.add_argument('--project', default=ROOT / 'runs/detect', help='save results to project/name')
    parser.add_argument('--name', default='exp', help='save results to project/name')
    parser.add_argument('--exist-ok', action='store_true', help='existing project/name ok, do not increment')
    parser.add_argument('--line-thickness', default=3, type=int, help='bounding box thickness (pixels)')
    parser.add_argument('--hide-labels', default=False, action='store_true', help='hide labels')
    parser.add_argument('--hide-conf', default=False, action='store_true', help='hide confidences')
    parser.add_argument('--half', action='store_true', help='use FP16 half-precision inference')
    parser.add_argument('--dnn', action='store_true', help='use OpenCV DNN for ONNX inference')
    parser.add_argument('--trigger_detect', action='store_true', help='perform trigger detection module')
    parser.add_argument('--CA', action='store_true', help='perform CA-LayerCAM')
    opt = parser.parse_args()
    opt.imgsz *= 2 if len(opt.imgsz) == 1 else 1  # expand
    print_args(FILE.stem, opt)
    return opt


def main(opt):
    check_requirements(exclude=('tensorboard', 'thop'))
    run(**vars(opt))


if __name__ == "__main__":
    opt = parse_opt()
    main(opt)
