# Copyright (c) Meta Platforms, Inc. and affiliates.
# All rights reserved.

# This source code is licensed under the license found in the
# LICENSE file in the root directory of this source tree.
# --------------------------------------------------------
# References:
# DeiT: https://github.com/facebookresearch/deit
# BEiT: https://github.com/microsoft/unilm/tree/master/beit
# --------------------------------------------------------
import argparse
import datetime
import json
import numpy as np
import os
import time
from pathlib import Path
import copy


import torch
import torch.backends.cudnn as cudnn
from torch.utils.tensorboard import SummaryWriter
import torchvision.transforms as transforms
import torchvision.datasets as datasets

import timm

# assert timm.__version__ == "0.3.2"  # version check
import timm.optim.optim_factory as optim_factory

import util.misc as misc
from util.misc import NativeScalerWithGradNormCount as NativeScaler

import models_mae

from engine_pretrain import train_one_epoch
import random
import psutil


def get_args_parser():
    parser = argparse.ArgumentParser('MAE pre-training', add_help=False)
    parser.add_argument('--full_schedule', default=False, action='store_true')
    parser.add_argument('--prob_mask', default=False, action='store_true')
    parser.add_argument('--min_portion', default=0.75, type=float)
    parser.add_argument('--min_ratio', default=0.05, type=float)
    parser.add_argument('--mask_batch_size', default=64, type=int)
    parser.add_argument('--schedule_exp', default=2, type=int)
    parser.add_argument('--debug', default=False, action='store_true')
    parser.add_argument('--store_mask', default=False, action='store_true')
    parser.add_argument('--uniform_prob', default=False, action='store_true')
    parser.add_argument('--count_convergence', default=False, action='store_true')
    parser.add_argument('--resume_from_start', default=False, action='store_true')
    parser.add_argument('--count_convergence_path', default=None, type=str)
    parser.add_argument('--not_load_lr', default=None, type=str)
    parser.add_argument('--partial_dataset', action='store_true')
    parser.add_argument('--wholestart', action='store_true')
    parser.add_argument('--ada_wholestart', action='store_true')
    parser.add_argument('--resume_mask_epoch', default=None, type=int)
    parser.add_argument('--use_encoder', action='store_true')
    parser.add_argument('--ref_cluster', default='large', type=str)
    parser.add_argument('--hint_prob', default=False, action='store_true')
    parser.add_argument('--hint_ratio', default=None, type=float)
    parser.add_argument('--hint_portion', default=0.5, type=float)
    parser.add_argument('--hint_schedule', default=False, action='store_true')
    parser.add_argument('--enc_target_attn', default=11.0, type=float)
    parser.add_argument('--wholestart_ratio', default=0.8, type=float)
    parser.add_argument('--informed_mask', action='store_true')
    parser.add_argument('--test', action='store_true')
    parser.add_argument('--backend', default='nccl', type=str)
    parser.add_argument('--fixed_mask', action='store_true')
    parser.add_argument('--informed_mask_warmup', default=0, type=int)
    parser.add_argument('--target_attn', default=4.0, type=float)
    parser.add_argument('--informed_masking_threshold', default=40, type=int)
    parser.add_argument('--batch_size', default=64, type=int,
                        help='Batch size per GPU (effective batch size is batch_size * accum_iter * # gpus')
    parser.add_argument('--epochs', default=400, type=int)
    parser.add_argument('--save_epochs', default=5, type=int)
    parser.add_argument('--accum_iter', default=1, type=int,
                        help='Accumulate gradient iterations (for increasing the effective batch size under memory constraints)')
    parser.add_argument('--train_ratio', default=1.0, type=float,
                        help='Ratio of training data to use')

    # Model parameters
    parser.add_argument('--model', default='mae_vit_large_patch16', type=str, metavar='MODEL',
                        help='Name of model to train')

    parser.add_argument('--input_size', default=224, type=int,
                        help='images input size')

    parser.add_argument('--mask_ratio', default=0.75, type=float,
                        help='Masking ratio (percentage of removed patches).')

    parser.add_argument('--norm_pix_loss', action='store_true',
                        help='Use (per-patch) normalized pixels as targets for computing loss')
    parser.set_defaults(norm_pix_loss=False)

    # Optimizer parameters
    parser.add_argument('--weight_decay', type=float, default=0.05,
                        help='weight decay (default: 0.05)')

    parser.add_argument('--lr', type=float, default=None, metavar='LR',
                        help='learning rate (absolute lr)')
    parser.add_argument('--blr', type=float, default=1e-3, metavar='LR',
                        help='base learning rate: absolute_lr = base_lr * total_batch_size / 256')
    parser.add_argument('--min_lr', type=float, default=0., metavar='LR',
                        help='lower lr bound for cyclic schedulers that hit 0')

    parser.add_argument('--warmup_epochs', type=int, default=40, metavar='N',
                        help='epochs to warmup LR')

    # Dataset parameters
    parser.add_argument('--data_path', default='/datasets01/imagenet_full_size/061417/', type=str,
                        help='dataset path')

    parser.add_argument('--output_dir', default='./output_dir',
                        help='path where to save, empty for no saving')
    parser.add_argument('--log_dir', default='./output_dir',
                        help='path where to tensorboard log')
    parser.add_argument('--device', default='cuda',
                        help='device to use for training / testing')
    parser.add_argument('--seed', default=0, type=int)
    parser.add_argument('--resume', default='',
                        help='resume from checkpoint')

    parser.add_argument('--start_epoch', default=0, type=int, metavar='N',
                        help='start epoch')
    parser.add_argument('--num_workers', default=10, type=int)
    parser.add_argument('--pin_mem', action='store_true',
                        help='Pin CPU memory in DataLoader for more efficient (sometimes) transfer to GPU.')
    parser.add_argument('--no_pin_mem', action='store_false', dest='pin_mem')
    parser.set_defaults(pin_mem=True)

    # distributed training parameters
    parser.add_argument('--world_size', default=1, type=int,
                        help='number of distributed processes')
    parser.add_argument('--local_rank', default=-1, type=int)
    parser.add_argument('--dist_on_itp', action='store_true')
    parser.add_argument('--dist_url', default='env://',
                        help='url used to set up distributed training')


    return parser

            
class ImgDataset(datasets.ImageFolder):
    def __init__(self, return_path = False, gen_mask = False, pure=False, hint_prob=False, alternate=False,**kwargs):
        super().__init__(**kwargs)
        self.num_retries = 10
        self.mask = torch.rand(1, 14*14 + 3).repeat(self.__len__(),1) # ids_shuffle, cluster_size, first_name, second_name
        self.return_path = return_path
        self.gen_mask = gen_mask
        self.pure = pure
        self.hint_prob = hint_prob
        self.alternate = alternate
        self.is_even = 0
        if return_path:
            self.mask_index = torch.ones(1,1).repeat(self.__len__(),1)
    def __getitem__(self, index: int):
        """
        Args:
            index (int): Index

        Returns:
            tuple: (sample, target) where target is class_index of the target class.
        """
        failed = []
        for _ in range(self.num_retries):
            path, target = self.samples[index]
            ###
            
            ###
            # print(f'{index} - {path}')
            if self.return_path:
                path_split = path.split('.')[-2].split('/')[-2:]
                path_first, path_second = path_split
                path_first = int(path_first[1:])
                path_second = int(path_second.split('_')[-1])
            mask = self.mask[index]
            if self.alternate:
                mask = mask[self.is_even]
            try:
                sample = self.loader(path)
            except:
                try:
                    sample = self.loader(path) # one more time
                except:
                    failed.append(path)
                    index = random.randint(0, len(self.samples) - 1)
                    continue
            if self.transform is not None:
                if self.gen_mask:
                    sample = self.transform(sample)
                else:
                    if self.pure:
                        sample, mask = self.transform(sample, mask)
                    else:
                        mask, etc = mask[:196], mask[196:]
                        sample, mask = self.transform(sample, mask)
                        # mask = torch.cat([mask, etc])
            if self.target_transform is not None:
                target = self.target_transform(target)
            
            if self.return_path:
                mask_idx = self.mask_index[index]
                # print(path_first, path_second, mask_idx)
                return sample, target, index, mask, path_first, path_second, mask_idx
            return sample, target, mask
        else:
            print('Failed to load {} after {} retries'.format(
                failed, self.num_retries
            ))
    def __mask_len__(self):
        return self.mask.shape[0]
    def update_mask_num(self):
        bef_num = self.__mask_len__()
        self.mask = self.mask[:self.__len__()]
        print(f'Updated self.mask from {bef_num} to {self.__mask_len__()}')
    def update_sample_num(self, index):
        bef_num = self.__len__()
        self.samples = list(np.array(self.samples)[index])
        # path1 = ('/data2/dataset/imagenet/1K_dataset/train/n02814533/n02814533_12979.JPEG', torch.tensor(1))
        # path2 = ('/data2/dataset/imagenet/1K_dataset/train/n01806567/n01806567_12379.JPEG', torch.tensor(1))
        # path3 = ('/data2/dataset/imagenet/1K_dataset/train/n07695742/n07695742_10443.JPEG', torch.tensor(1))
        # path4 = ('/data2/dataset/imagenet/1K_dataset/train/n02487347/n02487347_2480.JPEG', torch.tensor(1))
        # self.samples = [path1, path2, path3, path4]*32
        print(f'Updated self.samples from {bef_num} to {self.__len__()}')
    def is_even_epoch(self, epoch):
        self.is_even = epoch%2
    
    

def main(args):
    misc.init_distributed_mode(args)
    
    print('job dir: {}'.format(os.path.dirname(os.path.realpath(__file__))))
    print("{}".format(args).replace(', ', ',\n'))

    device = torch.device(args.device)

    # fix the seed for reproducibility
    seed = args.seed + misc.get_rank()
    torch.manual_seed(seed)
    np.random.seed(seed)

    cudnn.benchmark = True

    # simple augmentation
    if args.informed_mask:
        transform_train = misc.trainCompose([
        # transform_train = transforms.Compose([
                misc.maskRandomResizedCrop(args.input_size, scale=(0.2, 1.0), interpolation=3),  # 3 is bicubic
                misc.maskRandomHorizontalFlip(),
                # transforms.Resize((args.input_size, args.input_size)),
                transforms.ToTensor(),
                transforms.Normalize(mean=[0.485, 0.456, 0.406], std=[0.229, 0.224, 0.225])])
        dataset_train = ImgDataset(return_path = False,gen_mask=False, pure = False, hint_prob=args.hint_prob, alternate=args.ref_cluster == 'alternate',
                                   root=os.path.join(args.data_path, 'train'), transform=transform_train)
    else:
        transform_train = transforms.Compose([
                transforms.RandomResizedCrop(args.input_size, scale=(0.2, 1.0), interpolation=3),  # 3 is bicubic
                transforms.RandomHorizontalFlip(),
                transforms.ToTensor(),
                transforms.Normalize(mean=[0.485, 0.456, 0.406], std=[0.229, 0.224, 0.225])])
        dataset_train = misc.NormalImgDataset(root=os.path.join(args.data_path, 'train'), transform=transform_train)
        
    transform_make_mask = transforms.Compose([
            transforms.Resize((args.input_size, args.input_size)),
            transforms.ToTensor(),
            transforms.Normalize(mean=[0.485, 0.456, 0.406], std=[0.229, 0.224, 0.225])])
    dataset_make_mask = ImgDataset(return_path = True, gen_mask = True, root=os.path.join(args.data_path, 'train'), transform=transform_make_mask)
    
    # dataset_make_mask = datasets.ImageFolder(os.path.join(args.data_path, 'train'),transform = transform_make_mask)
    
    if args.debug:
        print('Debugging mode')
        split_num = 512
        index = torch.tensor([i*100 for i in range(split_num)])
        dataset_train.update_sample_num(index)
        dataset_make_mask.update_sample_num(index)
        dataset_train.update_mask_num()
        dataset_make_mask.update_mask_num()
        
        print(dataset_train.samples[-3])
        print(dataset_make_mask.samples[-3])
        print(dataset_train.samples[-2])
        print(dataset_make_mask.samples[-2])
        print(dataset_train.samples[-1])
        print(dataset_make_mask.samples[-1])
        
    # print(dataset_train)
    # print(dataset_make_mask)
    
    # sample, target, index, mask, path_first1, path_second1, _  = dataset_train.__getitem__(100)
    # sample, target, index, mask, path_first2, path_second2, _ = dataset_make_mask.__getitem__(100)
    # print(path_first1, path_second1)
    # print(path_first2, path_second2)
    
    # for i in dataset_train.samples:
    #     print(i)
    if True:  # args.distributed:
        num_tasks = misc.get_world_size()
        global_rank = misc.get_rank()
        sampler_train = torch.utils.data.DistributedSampler(
            dataset_train, num_replicas=num_tasks, rank=global_rank, shuffle=True
        )
        sampler_make_mask = misc.SequentialDistributedSampler(
            dataset_make_mask, args.mask_batch_size, num_replicas=num_tasks, rank=global_rank
        )
        print("Sampler_train = %s" % str(sampler_train))
    else:
        sampler_train = torch.utils.data.RandomSampler(dataset_train)

    if global_rank == 0 and args.log_dir is not None:
        os.makedirs(args.log_dir, exist_ok=True)
        log_writer = SummaryWriter(log_dir=args.log_dir)
    else:
        log_writer = None

    data_loader_train = torch.utils.data.DataLoader(
        dataset_train, sampler=sampler_train,
        batch_size=args.batch_size,
        num_workers=args.num_workers,
        pin_memory=args.pin_mem,
        drop_last=True,
    )
    data_loader_make_mask = torch.utils.data.DataLoader(
        dataset_make_mask, sampler = sampler_make_mask,
        batch_size = args.mask_batch_size,
        num_workers=args.num_workers,
        pin_memory=args.pin_mem,
        drop_last=False,
    )
    
    # define the model
    model = models_mae.__dict__[args.model](norm_pix_loss=args.norm_pix_loss)

    model.to(device)

    model_without_ddp = model
    print("Model = %s" % str(model_without_ddp))

    eff_batch_size = args.batch_size * args.accum_iter * misc.get_world_size()
    
    if args.lr is None:  # only base_lr is specified
        args.lr = args.blr * eff_batch_size / 256

    print("base lr: %.2e" % (args.lr * 256 / eff_batch_size))
    print("actual lr: %.2e" % args.lr)

    print("accumulate grad iterations: %d" % args.accum_iter)
    print("effective batch size: %d" % eff_batch_size)

    if args.distributed:
        model = torch.nn.parallel.DistributedDataParallel(model, device_ids=[args.gpu], find_unused_parameters=True)
        model_without_ddp = model.module
    
    # following timm: set wd as 0 for bias and norm layers
    param_groups = optim_factory.param_groups_weight_decay(model_without_ddp, args.weight_decay)
    optimizer = torch.optim.AdamW(param_groups, lr=args.lr, betas=(0.9, 0.95))
    print(optimizer)
    loss_scaler = NativeScaler()

    misc.load_model(args=args, model_without_ddp=model_without_ddp, optimizer=optimizer, loss_scaler=loss_scaler)

    print(f"Start training for {args.epochs} epochs")
    start_time = time.time()
    informed_started_ratio = 0
    starts_informed_masking = False
    started_informed_masking = False
    
    if args.informed_mask:
        if args.resume: starts_informed_masking = True
        print(f'Dataset: {dataset_train.__len__()}, Maskset: {dataset_train.__mask_len__()}')
    
    for epoch in range(args.start_epoch, args.epochs):
        if args.informed_mask:
            dataset_train.is_even_epoch(epoch)
        if args.informed_mask:
            if starts_informed_masking and not started_informed_masking:
                started_informed_masking = True
                misc.update_mask(model_without_ddp, data_loader_make_mask, device, dataset_train, args.enc_target_attn, 
                                mask_ratio = args.mask_ratio, ref_cluster = args.ref_cluster, store_mask = args.store_mask)
                print(f'Successfully generated {dataset_train.__mask_len__()}/{dataset_train.__len__()} masks')
                # print(dataset_train.samples)
                # print(dataset_train.mask_index)
                # for i in dataset_train.samples:
                #     print(i)
                
        if args.distributed:
            data_loader_train.sampler.set_epoch(epoch)
            
        train_stats, informed_started_ratio = train_one_epoch(
            model, data_loader_train,
            optimizer, device, epoch, loss_scaler, epoch,
            log_writer=log_writer, args=args,  
            started_informed_masking = started_informed_masking,
        )
        
        torch.distributed.barrier()
        
        if args.output_dir and (epoch % args.save_epochs == 0 or epoch + 1 == args.epochs or epoch in [0, 1, 2, 3, 4]):
            misc.save_model(
                args=args, model=model, model_without_ddp=model_without_ddp, optimizer=optimizer,
                loss_scaler=loss_scaler, epoch=epoch)

        log_stats = {**{f'train_{k}': v for k, v in train_stats.items()},
                        'epoch': epoch,}

        if args.output_dir and misc.is_main_process():
            if log_writer is not None:
                log_writer.flush()
            with open(os.path.join(args.output_dir, "log.txt"), mode="a", encoding="utf-8") as f:
                f.write(json.dumps(log_stats) + "\n")

    total_time = time.time() - start_time
    total_time_str = str(datetime.timedelta(seconds=int(total_time)))
    print('Training time {}'.format(total_time_str))


if __name__ == '__main__':
    args = get_args_parser()
    args = args.parse_args()
    if args.output_dir:
        Path(args.output_dir).mkdir(parents=True, exist_ok=True)
    main(args)
