""" Functions to model performativity, i.e., how y arises from x and yhat."""
import numpy as np
from models.framework import PreDataset
from sklearn.linear_model import LinearRegression
from sklearn.pipeline import make_pipeline
from sklearn.preprocessing import StandardScaler


def self_fulfilling(predataset, predictor, params, graph):
    assert isinstance(predataset, PreDataset)
    assert params["threshold"]
    # get predictions
    predictions = predictor.predict(predataset.features)

    # generate new outcomes
    new_y = []
    for x, pred, y in zip(predataset.features, predictions, predataset.outcomes):
        if pred != y and np.random.rand() < params["threshold"]:
            new_y.append(pred)
        else:
            new_y.append(y)
    new_y = np.array(new_y)

    return predataset.features, predictions, new_y, predataset.outcomes

def linear(predataset, predictor, params, graph):
    assert isinstance(predataset, PreDataset)
    assert params["threshold"]

    beta = params["threshold"]
    predictions = predictor.predict(predataset.features)

    linear_model = make_pipeline(StandardScaler(), LinearRegression())
    linear_model.fit(predataset.features, predataset.outcomes)
    first_term = linear_model.predict(predataset.features)

    # generate new outcomes
    new_y = []
    for ax, pred, y in zip(first_term, predictions, predataset.outcomes):
        new_y.append((1 - beta) * ax + beta * pred + 25*np.random.normal())

    return predataset.features, predictions, new_y, predataset.outcomes

def laplace(predataset, predictor, params, graph):
    assert isinstance(predataset, PreDataset)
    assert params["threshold"] and params['lambda']

    beta = params["threshold"]
    predictions = predictor.predict(predataset.features)
    noisy_preds = predictions + np.random.laplace(loc=0,
                                                  scale=params['lambda'],
                                                  size=len(predictions))
    linear_model = make_pipeline(StandardScaler(), LinearRegression())
    linear_model.fit(predataset.features, predataset.outcomes)
    first_term = linear_model.predict(predataset.features)

    # generate new outcomes
    new_y = []
    for ax, pred in zip(first_term, noisy_preds):
        new_y.append((1 - beta) * ax + beta * pred + 25*np.random.normal())

    return predataset.features, noisy_preds, new_y, predataset.outcomes

def discrete(predataset, predictor, params, graph):
    assert isinstance(predataset, PreDataset)
    assert params


