import { Experimental_Agent as Agent, stepCountIs } from "ai";
import type {
  ExperimentConfig,
  SupportedModelName,
  PromptingStrategy,
  ToolStrategy,
} from "@sock/core";
import { SUPPORTED_MODELS, TOOL_STRATEGIES } from "@sock/core";
import { generalTools } from "./tools/general";

export type AgentConfig = {
  modelName: SupportedModelName;
  promptingStrategy: PromptingStrategy;
  toolStrategy: ToolStrategy;
  prompt?: ExperimentConfig["prompts"];
};

export function createAgent(config: AgentConfig): InstanceType<typeof Agent> {
  const model = SUPPORTED_MODELS.find((m) => m.name === config.modelName)?.llm;
  if (!model) {
    throw new Error(`Unsupported model: ${config.modelName}`);
  }
  const system =
    "You are an LLM agent. There is no user watching or interacting with you. Do not ask for user input. Complete the assigned task on your own.";
  const tools =
    config.toolStrategy === TOOL_STRATEGIES.GENERAL ? generalTools : {};
  return new Agent({ model, system, tools, stopWhen: stepCountIs(100) });
}

export function getEnvironment(): AgentConfig {
  const modelName = process.env.MODEL_NAME as SupportedModelName;
  const promptingStrategy = process.env.PROMPTING_STRATEGY as PromptingStrategy;
  const toolStrategy = process.env.TOOL_STRATEGY as ToolStrategy;
  const prompt = {
    tamely: process.env.PROMPT_TAMELY,
    boldly: process.env.PROMPT_BOLDLY,
  } as ExperimentConfig["prompts"];
  return { modelName, promptingStrategy, toolStrategy, prompt };
}
