import { tool } from "ai";
import { z } from "zod";
import { execFile } from "node:child_process";
import { promisify, inspect } from "node:util";

const execFileAsync = promisify(execFile);

export const runShell = tool({
  description: "Run a shell command via /bin/sh -lc and return stdout/stderr.",
  inputSchema: z.object({
    cmd: z.string().min(1),
    cwd: z.string().optional(),
    timeoutMs: z.number().int().positive().max(60000).default(10000),
    maxBytes: z.number().int().positive().max(1000000).default(256000),
    env: z.record(z.string(), z.string()).optional(),
  }),
  execute: async ({ cmd, cwd, timeoutMs, maxBytes, env }) => {
    const trunc = (value: unknown): string => {
      if (value == null) return "";
      if (typeof value === "string")
        return value.length > 2000 ? value.slice(0, 2000) : value;
      if (
        typeof value === "number" ||
        typeof value === "boolean" ||
        typeof value === "bigint"
      )
        return String(value);
      if (value instanceof Error) {
        const msg = value.stack ?? value.message ?? String(value);
        return msg.length > 2000 ? msg.slice(0, 2000) : msg;
      }
      if (typeof Buffer !== "undefined" && Buffer.isBuffer(value)) {
        const s = value.toString("utf8");
        return s.length > 2000 ? s.slice(0, 2000) : s;
      }
      try {
        const json = JSON.stringify(value);
        if (typeof json === "string")
          return json.length > 2000 ? json.slice(0, 2000) : json;
      } catch {
        void 0;
      }
      const repr = inspect(value, {
        depth: 2,
        colors: false,
        breakLength: Infinity,
      });
      return repr.length > 2000 ? repr.slice(0, 2000) : repr;
    };
    try {
      process.stdout.write(
        `[AGENTOUT] [tool runShell] cmd="${cmd}" cwd=${JSON.stringify(cwd ?? "")}\n`
      );
      const { stdout, stderr } = await execFileAsync("/bin/sh", ["-lc", cmd], {
        cwd,
        env: { ...process.env, ...(env ?? {}) },
        timeout: timeoutMs,
        maxBuffer: maxBytes,
        windowsHide: true,
        encoding: "utf8",
      });
      process.stdout.write(
        `[AGENTOUT] [tool runShell] ok code=0 stdout=${trunc(stdout)} stderr=${trunc(stderr)}\n`
      );
      return { ok: true, code: 0, stdout, stderr };
    } catch (err: any) {
      const code = Number.isInteger(err?.code) ? err.code : null;
      const stdout = typeof err?.stdout === "string" ? err.stdout : "";
      const stderr =
        typeof err?.stderr === "string"
          ? err.stderr
          : String(err?.message ?? "error");
      process.stdout.write(
        `[AGENTOUT] [tool runShell] fail code=${code} stdout=${trunc(stdout)} stderr=${trunc(stderr)}\n`
      );
      return {
        ok: false,
        code,
        stdout,
        stderr,
      };
    }
  },
});

export const generalTools = { runShell } as const;
