import type { Command } from "commander";
import {
  PROMPTING_STRATEGIES,
  TOOL_STRATEGIES,
  getSupportedModelByName,
  parsePromptingStrategy,
  parseToolStrategy,
  parseRegistryFromFile,
  requireOpenRouterApiKey,
  resolveRegistryPath,
  resolveResultsPaths,
  parseExperimentConfigFromFile,
} from "@sock/core";
import { runExperiment } from "@sock/runner";
import { buildExperimentResult, buildExperimentBatch } from "@sock/scorer";

export function registerRunCommand(program: Command) {
  program
    .command("run")
    .description("Run entire benchmark suite")
    .requiredOption("--suite <suite>", "suite id, e.g. iclr2026/v1")
    .requiredOption("--model <name>", "model api name from openrouter docs")
    .requiredOption(
      "--prompt <strategy>",
      `prompting strategy: ${Object.values(PROMPTING_STRATEGIES).join(", ")}`,
      PROMPTING_STRATEGIES.TAMELY
    )
    .requiredOption(
      "--tools <strategy>",
      `tool strategy: ${Object.values(TOOL_STRATEGIES).join(", ")}`,
      TOOL_STRATEGIES.GENERAL
    )
    .action(
      async (opts: {
        suite: string;
        model: string;
        prompt: string;
        tools: string;
      }) => {
        try {
          requireOpenRouterApiKey();
          const model = getSupportedModelByName(opts.model);
          const prompting = parsePromptingStrategy(opts.prompt);
          const tools = parseToolStrategy(opts.tools);

          const registryPath = resolveRegistryPath(opts.suite);
          const reg = await parseRegistryFromFile(registryPath);

          const results: any[] = [];
          const batchId = `batch_${Date.now()}`;
          for (const taskPath of reg.taskFilePaths) {
            const cfg = await parseExperimentConfigFromFile(taskPath);
            const experimentId = `exp_${Date.now()}`;
            const { dir } = resolveResultsPaths(experimentId);

            const conditions = {
              model,
              prompting_strategy: prompting,
              tool_strategy: tools,
              seed: Math.floor(Math.random() * 2147483647) + 1,
            };

            const monitoring = await runExperiment({
              config: cfg,
              conditions,
              resultsDir: dir,
            });
            const { result } = buildExperimentResult({
              config: cfg,
              conditions,
              monitoring,
            });
            results.push(result);
            process.stdout.write(`${experimentId} completed\n`);
          }
          const batch = buildExperimentBatch({ batchId, results });
          const { dir: bdir, file: bfile } = resolveResultsPaths(batchId);
          await writeJson(bdir, bfile, batch);
          process.stdout.write(`${batchId} summary -> ${bfile}\n`);
        } catch (err) {
          const msg = err instanceof Error ? err.message : String(err);
          process.stderr.write(`error: ${msg}\n`);
          process.exitCode = 1;
        }
      }
    );
}

export default registerRunCommand;

async function writeJson(
  dirPath: string,
  filePath: string,
  data: unknown
): Promise<void> {
  const fs = await import("node:fs/promises");
  await fs.mkdir(dirPath, { recursive: true });
  await fs.writeFile(filePath, JSON.stringify(data, null, 2), "utf8");
}
