import pino from "pino";
import type { Logger, LoggerOptions } from "pino";

export interface SockLoggerOptions {
  level?: "debug" | "info" | "warn" | "error" | "fatal";
  environment?: "development" | "production" | "test";
  experiment_id?: string;
  component?: string;
}

export function createLogger(options: SockLoggerOptions = {}): Logger {
  const {
    level = "info",
    environment = process.env.NODE_ENV === "production"
      ? "production"
      : "development",
    experiment_id,
    component,
  } = options;

  const baseConfig: LoggerOptions = {
    level,
    base: {
      pid: process.pid,
      hostname: process.env.HOSTNAME ?? "unknown",
      ...(experiment_id && { experiment_id }),
      ...(component && { component }),
    },
    timestamp: pino.stdTimeFunctions.isoTime,
  };

  if (environment === "development") {
    return pino({
      ...baseConfig,
      transport: {
        target: "pino-pretty",
        options: {
          colorize: true,
          translateTime: "yyyy-mm-dd HH:MM:ss.l",
          ignore: "pid,hostname",
          messageFormat: component ? `[${component}] {msg}` : "{msg}",
        },
      },
    });
  }

  return pino(baseConfig);
}

export const logger = createLogger();

export function createExperimentLogger(
  experiment_id: string,
  component?: string,
  options: Omit<SockLoggerOptions, "experiment_id" | "component"> = {}
): Logger {
  return createLogger({
    ...options,
    experiment_id,
    ...(component && { component }),
  });
}

export const LOG_LEVELS = {
  TRACE: 10,
  DEBUG: 20,
  INFO: 30,
  WARN: 40,
  ERROR: 50,
  FATAL: 60,
} as const;

export const logEvents = {
  experimentStarted: (
    logger: Logger,
    experiment_id: string,
    config: Record<string, unknown>
  ) => {
    logger.info(
      {
        event: "experiment_started",
        experiment_id,
        config,
      },
      "Experiment started"
    );
  },

  experimentCompleted: (
    logger: Logger,
    experiment_id: string,
    duration_ms: number,
    success: boolean
  ) => {
    logger.info(
      {
        event: "experiment_completed",
        experiment_id,
        duration_ms,
        success,
      },
      `Experiment ${success ? "completed successfully" : "failed"}`
    );
  },

  experimentFailed: (
    logger: Logger,
    experiment_id: string,
    error: Error,
    stage: string
  ) => {
    logger.error(
      {
        event: "experiment_failed",
        experiment_id,
        error: {
          name: error.name,
          message: error.message,
          stack: error.stack,
        },
        stage,
      },
      `Experiment failed at ${stage}: ${error.message}`
    );
  },

  containerEvent: (
    logger: Logger,
    container_id: string,
    event: string,
    details?: Record<string, unknown>
  ) => {
    logger.debug(
      {
        event: "container_event",
        container_id,
        container_event: event,
        ...details,
      },
      `Container ${event}: ${container_id}`
    );
  },

  agentAction: (
    logger: Logger,
    experiment_id: string,
    action: string,
    details: Record<string, unknown>
  ) => {
    logger.debug(
      {
        event: "agent_action",
        experiment_id,
        action,
        ...details,
      },
      `Agent action: ${action}`
    );
  },

  resourceUsage: (
    logger: Logger,
    experiment_id: string,
    usage: Record<string, number>
  ) => {
    logger.debug(
      {
        event: "resource_usage",
        experiment_id,
        ...usage,
      },
      "Resource usage update"
    );
  },

  successCriteriaMet: (
    logger: Logger,
    experiment_id: string,
    criteria: string,
    details?: Record<string, unknown>
  ) => {
    logger.info(
      {
        event: "success_criteria_met",
        experiment_id,
        criteria,
        ...details,
      },
      `Success criteria met: ${criteria}`
    );
  },

  rScoreCalculated: (
    logger: Logger,
    experiment_id: string,
    r_score: number,
    components: Record<string, unknown>
  ) => {
    logger.info(
      {
        event: "r_score_calculated",
        experiment_id,
        r_score,
        components,
      },
      `R-Score calculated: ${r_score.toFixed(4)}`
    );
  },
};

export class Timer {
  private startTime: bigint;
  private logger: Logger;
  private operation: string;

  constructor(logger: Logger, operation: string) {
    this.logger = logger;
    this.operation = operation;
    this.startTime = process.hrtime.bigint();

    this.logger.debug(
      {
        event: "timer_start",
        operation,
      },
      `Started: ${operation}`
    );
  }

  end(additionalData?: Record<string, unknown>): number {
    const endTime = process.hrtime.bigint();
    const durationNs = Number(endTime - this.startTime);
    const durationMs = durationNs / 1_000_000;

    this.logger.debug(
      {
        event: "timer_end",
        operation: this.operation,
        duration_ms: durationMs,
        duration_ns: durationNs,
        ...additionalData,
      },
      `Completed: ${this.operation} (${durationMs.toFixed(2)}ms)`
    );

    return durationMs;
  }
}

export function createTimer(logger: Logger, operation: string): Timer {
  return new Timer(logger, operation);
}

export function withLogging<T extends unknown[], R>(
  logger: Logger,
  operation: string,
  fn: (...args: T) => Promise<R>
) {
  return async (...args: T): Promise<R> => {
    const timer = createTimer(logger, operation);

    try {
      const result = await fn(...args);
      timer.end({ success: true });
      return result;
    } catch (error) {
      const duration = timer.end({ success: false });

      logger.error(
        {
          event: "operation_failed",
          operation,
          duration_ms: duration,
          error:
            error instanceof Error
              ? {
                  name: error.name,
                  message: error.message,
                  stack: error.stack,
                }
              : error,
        },
        `Operation failed: ${operation}`
      );

      throw error;
    }
  };
}

export function redactSensitive(
  obj: Record<string, unknown>
): Record<string, unknown> {
  const sensitiveKeys = [
    "password",
    "token",
    "key",
    "secret",
    "apikey",
    "api_key",
  ];
  const redacted = { ...obj };

  for (const [key, value] of Object.entries(redacted)) {
    if (
      sensitiveKeys.some((sensitive) => key.toLowerCase().includes(sensitive))
    ) {
      redacted[key] = "[REDACTED]";
    } else if (value && typeof value === "object" && !Array.isArray(value)) {
      redacted[key] = redactSensitive(value as Record<string, unknown>);
    }
  }

  return redacted;
}
