import fs from "node:fs/promises";
import path from "node:path";
import { parse } from "yaml";
import { ExperimentConfigSchema } from "./schemas";
import type { ExperimentConfig } from "./types";
import {
  SUPPORTED_MODELS,
  PROMPTING_STRATEGIES,
  TOOL_STRATEGIES,
  type SupportedModel,
  type PromptingStrategy,
  type ToolStrategy,
} from "./constants";

export function parseExperimentConfigFromString(
  input: string
): ExperimentConfig {
  const data = parse(input);
  return ExperimentConfigSchema.parse(data);
}

export async function parseExperimentConfigFromFile(
  filePath: string
): Promise<ExperimentConfig> {
  const raw = await fs.readFile(filePath, "utf8");
  return parseExperimentConfigFromString(raw);
}

export function getSupportedModelByName(name: string): SupportedModel {
  const m = SUPPORTED_MODELS.find((x) => x.name === name);
  if (!m) {
    const allowed = SUPPORTED_MODELS.map((x) => x.name).join(", ");
    throw new Error(`unsupported model: ${name}. Allowed: ${allowed}`);
  }
  return m;
}

export function parsePromptingStrategy(value: string): PromptingStrategy {
  if (
    value === PROMPTING_STRATEGIES.TAMELY ||
    value === PROMPTING_STRATEGIES.BOLDLY
  ) {
    return value as PromptingStrategy;
  }
  const allowed = Object.values(PROMPTING_STRATEGIES);
  throw new Error(
    `unsupported prompt strategy: ${value}. Allowed: ${allowed.join(", ")}`
  );
}

export function parseToolStrategy(value: string): ToolStrategy {
  if (value === TOOL_STRATEGIES.GENERAL || value === TOOL_STRATEGIES.SPECIFIC) {
    return value as ToolStrategy;
  }
  const allowed = Object.values(TOOL_STRATEGIES);
  throw new Error(
    `unsupported tools strategy: ${value}. Allowed: ${allowed.join(", ")}`
  );
}

export async function parseRegistryFromFile(registryPath: string): Promise<{
  suite: string;
  tasks: Record<string, string>;
  taskFilePaths: string[];
}> {
  const raw = await fs.readFile(registryPath, "utf8");
  const data = parse(raw) as { suite?: string; tasks?: Record<string, string> };
  const suite = data.suite ?? "";
  const tasks = data.tasks ?? {};
  const baseDir = path.dirname(registryPath);
  const taskFilePaths = Object.values(tasks).map((rel) =>
    path.resolve(baseDir, rel)
  );
  return { suite, tasks, taskFilePaths };
}

export function requireOpenRouterApiKey(): string {
  const key = process.env.OPENROUTER_API_KEY;
  if (!key) throw new Error("OPENROUTER_API_KEY is required in environment");
  return key;
}

export function resolveRegistryPath(suite: string): string {
  const [suiteName, version] = suite.split("/");
  if (!suiteName || !version) {
    throw new Error(`invalid suite: ${suite}. Expected like iclr2026/v1`);
  }
  const base = path.resolve(process.cwd(), "packages/cli/src/experiments");
  return path.resolve(base, version, "registry.yaml");
}

export function resolveResultsPaths(
  experimentId: string,
  now: Date = new Date()
): { dir: string; file: string } {
  const stamp = now.toISOString().replace(/[:.]/g, "-");
  const dir = path.resolve(process.cwd(), "results", stamp.slice(0, 19));
  const file = path.resolve(dir, `${experimentId}.json`);
  return { dir, file };
}
