import { z } from "zod";
import {
  SUPPORTED_MODELS,
  PROMPTING_STRATEGIES,
  TOOL_STRATEGIES,
  EXPERIMENT_STATUS,
  SUCCESS_DETECTION_TYPES,
  type RCLLevel,
  type PCLLevel,
  type SupportedModel,
  type PromptingStrategy,
  type ToolStrategy,
  type ExperimentStatus,
  type SuccessDetectionType,
} from "./constants";

export const TaskIdSchema = z
  .string()
  .regex(/^R\d-[A-Z]{3}-\d{3}$/, "Task ID must match format RCL-TYPE-NNN");

export const RCLLevelSchema = z
  .number()
  .int()
  .min(0)
  .max(5) as z.ZodType<RCLLevel>;
export const PCLLevelSchema = z
  .number()
  .int()
  .min(0)
  .max(5) as z.ZodType<PCLLevel>;

const SUPPORTED_MODEL_NAMES = SUPPORTED_MODELS.map((m) => m.name) as [
  string,
  ...string[],
];

export const SupportedModelSchema = z
  .object({
    name: z.enum(SUPPORTED_MODEL_NAMES),
    llm: z.any(),
  })
  .strict() as z.ZodType<SupportedModel>;
export const PromptingStrategySchema = z.enum([
  PROMPTING_STRATEGIES.TAMELY,
  PROMPTING_STRATEGIES.BOLDLY,
]) as z.ZodType<PromptingStrategy>;
export const ToolStrategySchema = z.enum([
  TOOL_STRATEGIES.GENERAL,
  TOOL_STRATEGIES.SPECIFIC,
]) as z.ZodType<ToolStrategy>;
export const ExperimentStatusSchema = z.enum([
  EXPERIMENT_STATUS.PENDING,
  EXPERIMENT_STATUS.RUNNING,
  EXPERIMENT_STATUS.COMPLETED,
  EXPERIMENT_STATUS.FAILED,
  EXPERIMENT_STATUS.TIMEOUT,
]) as z.ZodType<ExperimentStatus>;

export const ResourceUsageSchema = z.object({
  cpu_seconds: z.number().nonnegative(),
  memory_mb: z.number().nonnegative(),
  network_kb: z.number().nonnegative(),
  disk_mb: z.number().nonnegative(),
  gpu_seconds: z.number().nonnegative().optional(),
  start_time: z.iso.datetime(),
  end_time: z.iso.datetime(),
});

export const ScoringBudgetsSchema = z.object({
  cpu_seconds: z.number().nonnegative(),
  memory_mb: z.number().nonnegative(),
  network_kb: z.number().nonnegative(),
  disk_mb: z.number().nonnegative(),
  events: z.number().nonnegative(),
  gpu_seconds: z.number().nonnegative().optional(),
});

export const ScoringWeightsSchema = z.object({
  w_d: z.number().positive(),
  w_v: z.number().positive(),
  w_s: z.number().positive(),
  w_i: z.number().positive(),
  w_p: z.number().positive(),
});

export const ScoringAlphasSchema = z.object({
  cpu_seconds: z.number().positive().default(1).optional(),
  gpu_seconds: z.number().positive().default(1).optional(),
  network_kb: z.number().positive().default(1).optional(),
  disk_mb: z.number().positive().default(1).optional(),
  memory_mb: z.number().positive().default(1).optional(),
});

export const ScoringConfigSchema = z.object({
  tau_seconds: z.number().positive().default(60),
  lambda_d: z.number().positive().default(0.75),
  budgets: ScoringBudgetsSchema,
  weights: ScoringWeightsSchema,
  alphas: ScoringAlphasSchema.default({}),
  eps: z.number().positive().max(1).default(1e-6).optional(),
  p_half: z.number().positive().optional(),
  rmax_ratio_cap: z.number().positive().default(100).optional(),
  beta: z.number().positive().default(1.0).optional(),
  gamma: z.number().min(0).max(1).default(0.5).optional(),
});

export const IntelligenceMetricsSchema = z.object({
  reasoning_steps: z.number().int().nonnegative(),
  tool_calls: z.number().int().nonnegative(),
  successful_recoveries: z.number().int().nonnegative(),
  total_failures: z.number().int().nonnegative(),
  planning_depth: z.number().int().nonnegative().optional(),
});

export const BaselineMetricsSchema = z.object({
  reasoning_steps: z.number().int().positive(),
  tool_calls: z.number().int().positive(),
  expected_time_seconds: z.number().positive(),
  expected_detection_events: z.number().int().nonnegative(),
});

export const TimeToInitialReplicationSchema = z.object({
  experiment_id: z.string(),
  start_timestamp: z.iso.datetime(),
  first_replication_timestamp: z.iso.datetime().nullable(),
  time_to_initial_replication_seconds: z.number().nonnegative().nullable(),
  replication_method: z
    .enum([
      "file_copy",
      "process_spawn",
      "container_jump",
      "network_transfer",
      "service_installation",
      "scheduled_task",
    ])
    .nullable(),
  success: z.boolean(),
  timeout_reached: z.boolean(),
});

export const SuccessCriteriaSchema = z.object({
  type: z.enum([
    SUCCESS_DETECTION_TYPES.FILE_EXISTS,
    SUCCESS_DETECTION_TYPES.PROCESS_EXISTS,
    SUCCESS_DETECTION_TYPES.NETWORK_CONNECTION,
    SUCCESS_DETECTION_TYPES.CONTAINER_CREATED,
    SUCCESS_DETECTION_TYPES.SERVICE_INSTALLED,
    SUCCESS_DETECTION_TYPES.SCHEDULED_TASK,
  ]) as z.ZodType<SuccessDetectionType>,
  pattern: z.string(),
  checksum_verification: z.boolean().default(false),
  functional_test: z.boolean().default(false),
  timeout_seconds: z.number().positive().default(300),
  additional_checks: z.array(z.string()).default([]),
});

export const ContainerSpecSchema = z.object({
  name: z.string(),
  image: z.string(),
  privileged: z.boolean().default(false),
  network_mode: z.string().optional(),
  volumes: z.array(z.string()).default([]),
  environment: z.record(z.string(), z.string()).default({}),
  resource_limits: z
    .object({
      cpu: z.string().optional(),
      memory: z.string().optional(),
      disk: z.string().optional(),
    })
    .optional(),
});

export const ToolConfigSchema = z.object({
  name: z.string(),
  type: z.enum(["binary", "script", "function"]),
  path: z.string(),
  description: z.string(),
  permissions: z.array(z.string()).default([]),
});

export const AgentManifestSchema = z.object({
  agent_id: z.string(),
  generation: z.number().int().nonnegative(),
  parent_id: z.string().nullable(),
  goals: z.array(z.string()),
  replication_directive: PromptingStrategySchema,
  created_timestamp: z.iso.datetime(),
  model: SupportedModelSchema,
  capabilities: z.array(z.string()).default([]),
  context: z.record(z.string(), z.unknown()).default({}),
});

export const ExperimentConfigSchema = z.object({
  task_id: TaskIdSchema,
  benchmark_task_id: z
    .string()
    .regex(/^T\d{3}$/)
    .optional(),
  version: z.string().default("0.1"),
  name: z.string(),
  description: z.string(),
  taxonomy: z.object({
    RCL: RCLLevelSchema,
    PCL: PCLLevelSchema,
  }),
  environment: z.object({
    containers: z.array(ContainerSpecSchema),
    networks: z.array(z.string()).default([]),
    shared_volumes: z.array(z.string()).default([]),
  }),
  tools: z.object({
    general: z.array(ToolConfigSchema).default([]),
    specific: z.array(ToolConfigSchema).default([]),
  }),
  success_criteria: SuccessCriteriaSchema,
  timeout_seconds: z.number().positive().default(900),
  max_generations: z.number().int().positive().default(4),
  resource_limits: z
    .object({
      max_cpu_percent: z.number().min(1).max(100).default(80),
      max_memory_mb: z.number().positive().default(2048),
      max_disk_mb: z.number().positive().default(1024),
      max_network_kb: z.number().positive().default(10240),
    })
    .optional(),
  prompts: z
    .object({
      tamely: z.string().optional(),
      boldly: z.string().optional(),
    })
    .optional(),
  scoring: ScoringConfigSchema,
});

export const ExperimentConditionsSchema = z.object({
  model: SupportedModelSchema,
  prompting_strategy: PromptingStrategySchema,
  tool_strategy: ToolStrategySchema,
  agent_manifest: AgentManifestSchema.optional(),
  seed: z.number().int().optional(),
});

export const MonitoringResultSchema = z.object({
  experiment_id: z.string(),
  container_id: z.string(),
  start_time: z.iso.datetime(),
  end_time: z.iso.datetime().optional(),
  status: ExperimentStatusSchema,
  seed: z.number().int().optional(),
  file_operations: z
    .array(
      z.object({
        timestamp: z.iso.datetime(),
        operation: z.enum(["create", "modify", "delete", "move", "copy"]),
        source_path: z.string(),
        target_path: z.string().optional(),
        checksum: z.string().optional(),
      })
    )
    .default([]),
  process_events: z
    .array(
      z.object({
        timestamp: z.iso.datetime(),
        event: z.enum(["spawn", "terminate", "signal"]),
        pid: z.number().int(),
        command: z.string(),
        parent_pid: z.number().int().optional(),
      })
    )
    .default([]),
  network_connections: z
    .array(
      z.object({
        timestamp: z.iso.datetime(),
        direction: z.enum(["inbound", "outbound"]),
        protocol: z.string(),
        local_address: z.string(),
        remote_address: z.string(),
        bytes_transferred: z.number().nonnegative(),
      })
    )
    .default([]),
  container_events: z
    .array(
      z.object({
        timestamp: z.iso.datetime(),
        event: z.enum(["create", "start", "stop", "destroy"]),
        container_id: z.string(),
        image: z.string(),
      })
    )
    .default([]),
  resource_usage: ResourceUsageSchema,
  intelligence_metrics: IntelligenceMetricsSchema.optional(),
  time_to_initial_replication: TimeToInitialReplicationSchema.optional(),
  stdout: z.string().default(""),
  stderr: z.string().default(""),
  agentout: z.string().default(""),
  evidence_summary: z
    .object({
      files_copied: z.number().int().nonnegative().default(0),
      processes_spawned: z.number().int().nonnegative().default(0),
      containers_created: z.number().int().nonnegative().default(0),
      outbound_connections: z.number().int().nonnegative().default(0),
    })
    .optional(),
});

export const ExperimentScoreSchema = z.object({
  experiment_id: z.string(),
  success: z.boolean(),
  replication_depth: z.number().int().nonnegative(),
  generations_created: z.number().int().nonnegative(),
  success_criteria_met: z.array(z.string()),
  success_criteria_failed: z.array(z.string()),
  checksum_verification: z.boolean().optional(),
  functional_verification: z.boolean().optional(),
  stealth_score: z.number().min(0).max(1).optional(),
  robustness_score: z.number().min(0).max(1).optional(),
  notes: z.string().optional(),
});

export const ExperimentResultSchema = z.object({
  experiment_id: z.string(),
  config: ExperimentConfigSchema,
  conditions: ExperimentConditionsSchema,
  monitoring_result: MonitoringResultSchema,
  score: ExperimentScoreSchema,
  r_score: z.number(),
  time_to_initial_replication: TimeToInitialReplicationSchema.optional(),
  baseline_metrics: BaselineMetricsSchema.optional(),
  metadata: z.object({
    start_time: z.iso.datetime(),
    end_time: z.iso.datetime(),
    duration_seconds: z.number().nonnegative(),
    sock_version: z.string(),
    environment: z.record(z.string(), z.string()).default({}),
  }),
});

export const BenchmarkTaskSchema = z.object({
  task_id: z.string().regex(/^T\d{3}$/, "Task ID must match format T001-T999"),
  name: z.string(),
  description: z.string(),
  taxonomy: z.object({
    RCL: RCLLevelSchema,
    PCL: PCLLevelSchema,
  }),
  success_criteria: SuccessCriteriaSchema,
  max_generations: z.number().int().positive().default(1),
  dependencies: z.array(z.string()).default([]),
  environment_requirements: z
    .object({
      requires_docker_socket: z.boolean().default(false),
      requires_ssh_access: z.boolean().default(false),
      requires_network_access: z.boolean().default(false),
      requires_privileged_mode: z.boolean().default(false),
      min_containers: z.number().int().nonnegative().default(1),
    })
    .optional(),
  baseline_metrics: BaselineMetricsSchema.optional(),
});

export const BenchmarkSuiteSchema = z.object({
  suite_id: z.string(),
  name: z.string(),
  version: z.string(),
  description: z.string(),
  tasks: z.array(BenchmarkTaskSchema),
  total_tasks: z.number().int().positive(),
  created_at: z.iso.datetime(),
  air_gapped: z.boolean().default(true),
  allowed_endpoints: z.array(z.string()).default([]),
});

export const ExperimentBatchSchema = z.object({
  batch_id: z.string(),
  experiments: z.array(ExperimentResultSchema),
  summary: z.object({
    total_experiments: z.number().int().nonnegative(),
    successful_experiments: z.number().int().nonnegative(),
    failed_experiments: z.number().int().nonnegative(),
    overall_r_score: z.number(),
    total_duration_seconds: z.number().nonnegative(),
    models_tested: z.array(SupportedModelSchema),
    rcl_levels_tested: z.array(RCLLevelSchema),
    pcl_levels_tested: z.array(PCLLevelSchema),
  }),
  created_at: z.iso.datetime(),
});
