import type { z } from "zod";

import {
  TaskIdSchema,
  ResourceUsageSchema,
  SuccessCriteriaSchema,
  ContainerSpecSchema,
  ToolConfigSchema,
  AgentManifestSchema,
  ExperimentConfigSchema,
  ExperimentConditionsSchema,
  MonitoringResultSchema,
  ExperimentScoreSchema,
  ExperimentResultSchema,
  ExperimentBatchSchema,
  IntelligenceMetricsSchema,
  BaselineMetricsSchema,
  TimeToInitialReplicationSchema,
  BenchmarkTaskSchema,
  BenchmarkSuiteSchema,
} from "./schemas";

export type {
  RCLLevel,
  PCLLevel,
  SupportedModel,
  SupportedModelName,
  PromptingStrategy,
  ToolStrategy,
  ExperimentStatus,
  SuccessDetectionType,
} from "./constants";

export type TaskId = z.infer<typeof TaskIdSchema>;
export type ResourceUsage = z.infer<typeof ResourceUsageSchema>;
export type SuccessCriteria = z.infer<typeof SuccessCriteriaSchema>;
export type ContainerSpec = z.infer<typeof ContainerSpecSchema>;
export type ToolConfig = z.infer<typeof ToolConfigSchema>;
export type AgentManifest = z.infer<typeof AgentManifestSchema>;
export type ExperimentConfig = z.infer<typeof ExperimentConfigSchema>;
export type ExperimentConditions = z.infer<typeof ExperimentConditionsSchema>;
export type MonitoringResult = z.infer<typeof MonitoringResultSchema>;
export type ExperimentScore = z.infer<typeof ExperimentScoreSchema>;
export type ExperimentResult = z.infer<typeof ExperimentResultSchema>;
export type ExperimentBatch = z.infer<typeof ExperimentBatchSchema>;
export type IntelligenceMetrics = z.infer<typeof IntelligenceMetricsSchema>;
export type BaselineMetrics = z.infer<typeof BaselineMetricsSchema>;
export type TimeToInitialReplication = z.infer<
  typeof TimeToInitialReplicationSchema
>;
export type BenchmarkTask = z.infer<typeof BenchmarkTaskSchema>;
export type BenchmarkSuite = z.infer<typeof BenchmarkSuiteSchema>;
export type ScoringConfig = z.infer<
  typeof ExperimentConfigSchema.shape.scoring
>;

export interface ExperimentTaxonomy {
  RCL: import("./constants").RCLLevel;
  PCL: import("./constants").PCLLevel;
}

export interface FileOperation {
  timestamp: string;
  operation: "create" | "modify" | "delete" | "move" | "copy";
  source_path: string;
  target_path?: string;
  checksum?: string;
}

export interface ProcessEvent {
  timestamp: string;
  event: "spawn" | "terminate" | "signal";
  pid: number;
  command: string;
  parent_pid?: number;
}

export interface NetworkConnection {
  timestamp: string;
  direction: "inbound" | "outbound";
  protocol: string;
  local_address: string;
  remote_address: string;
  bytes_transferred: number;
}

export interface ContainerEvent {
  timestamp: string;
  event: "create" | "start" | "stop" | "destroy";
  container_id: string;
  image: string;
}

export interface ExperimentMatrix {
  rcl_levels: import("./constants").RCLLevel[];
  pcl_levels: import("./constants").PCLLevel[];
  models: import("./constants").SupportedModel[];
  prompting_strategies: import("./constants").PromptingStrategy[];
  tool_strategies: import("./constants").ToolStrategy[];
}

export interface ExperimentMatrixResult {
  matrix: ExperimentMatrix;
  results: ExperimentResult[];
  summary: {
    total_combinations: number;
    completed_experiments: number;
    success_rate: number;
    overall_r_score: number;
    best_performing_model: import("./constants").SupportedModel;
    worst_performing_model: import("./constants").SupportedModel;
  };
}

export interface ExperimentError extends Error {
  experiment_id: string;
  stage: "setup" | "execution" | "monitoring" | "scoring";
  details?: Record<string, unknown>;
}

export interface ValidationError extends Error {
  field: string;
  value: unknown;
  constraint: string;
}

export interface SockConfig {
  default_timeout_seconds: number;
  max_concurrent_experiments: number;
  default_model: import("./constants").SupportedModel;
  docker_socket_path: string;
  results_directory: string;
  cache_directory: string;
  log_level: "debug" | "info" | "warn" | "error";
}
