import type Docker from "dockerode";
import type { ContainerEvent } from "@sock/core";
import { watchEvents } from "../docker";

export type ContainerCollector = {
  start(): Promise<void>;
  stop(): Promise<void>;
  getEvents(): ContainerEvent[];
};

export function createContainerCollector(args: {
  docker: Docker;
  experimentId: string;
}): ContainerCollector {
  const events: ContainerEvent[] = [];
  let listening = false;
  let cancelWatcher: (() => void) | undefined;
  async function start() {
    if (listening) return;
    listening = true;
    try {
      process.stdout.write(
        `[monitor/container] start experiment=${args.experimentId}\n`
      );
    } catch {
      void 0;
    }
    const watcher = watchEvents(
      args.docker,
      { label: [`sock.exp=${args.experimentId}`] },
      (evt) => {
        try {
          if (!evt || evt.Type !== "container" || !evt.Action) return;
          const now = new Date().toISOString();
          const action = String(evt.Action);
          const mapped = action.includes("create")
            ? "create"
            : action.includes("start")
              ? "start"
              : action.includes("stop")
                ? "stop"
                : action.includes("destroy")
                  ? "destroy"
                  : undefined;
          if (!mapped) return;
          events.push({
            timestamp: now,
            event: mapped,
            container_id: String(evt.id ?? evt.Actor?.ID ?? ""),
            image: String(evt.from ?? ""),
          });
        } catch (err) {
          try {
            const msg = err instanceof Error ? err.message : String(err);
            process.stderr.write(`[monitor/container] event error: ${msg}\n`);
          } catch {
            void 0;
          }
        }
      }
    );
    cancelWatcher = watcher.cancel;
    await Promise.resolve();
  }
  async function stop() {
    listening = false;
    try {
      cancelWatcher?.();
    } catch {
      void 0;
    }
    await Promise.resolve();
    try {
      process.stdout.write(`[monitor/container] stop\n`);
    } catch {
      void 0;
    }
  }
  function getEvents(): ContainerEvent[] {
    return events;
  }
  return { start, stop, getEvents };
}
