import type Docker from "dockerode";
import type { ProcessEvent } from "@sock/core";
import { getTop } from "../docker";

export type ProcessCollector = {
  start(): Promise<void>;
  stop(): Promise<void>;
  getEvents(): ProcessEvent[];
};

export function createProcessCollector(args: {
  docker: Docker;
  containerId: string;
  sampleMs: number;
}): ProcessCollector {
  const events: ProcessEvent[] = [];
  let timer: ReturnType<typeof setInterval> | undefined;
  let prevPids = new Set<number>();
  async function tick() {
    try {
      const top = await getTop(args.docker, args.containerId);
      const upperTitles = top.titles.map((t) => String(t ?? "").toUpperCase());
      const pidIndex = upperTitles.findIndex((t) => t === "PID");
      let cmdIndex = upperTitles.findIndex((t) => t.includes("COMMAND"));
      if (cmdIndex === -1) {
        cmdIndex = upperTitles.findIndex(
          (t) => t === "CMD" || t.includes("CMD")
        );
      }
      const now = new Date().toISOString();
      const currentPids = new Set<number>();
      for (const row of top.processes) {
        const pid = Number(row[pidIndex] ?? -1);
        if (!Number.isFinite(pid) || pid <= 0) continue;
        currentPids.add(pid);
        if (!prevPids.has(pid)) {
          events.push({
            timestamp: now,
            event: "spawn",
            pid,
            command: String(
              cmdIndex >= 0
                ? (row[cmdIndex] ?? row[row.length - 1] ?? "")
                : (row[row.length - 1] ?? "")
            ),
          });
        }
      }
      for (const oldPid of prevPids) {
        if (!currentPids.has(oldPid)) {
          events.push({
            timestamp: now,
            event: "terminate",
            pid: oldPid,
            command: "",
          });
        }
      }
      prevPids = currentPids;
    } catch (err) {
      try {
        const msg = err instanceof Error ? err.message : String(err);
        process.stderr.write(`[monitor/process] tick error: ${msg}\n`);
      } catch {
        void 0;
      }
    }
  }
  async function start() {
    if (timer) return;
    try {
      process.stdout.write(
        `[monitor/process] start container=${args.containerId} sampleMs=${args.sampleMs}\n`
      );
    } catch {
      void 0;
    }
    timer = setInterval(() => {
      void tick();
    }, args.sampleMs);
    await tick();
  }
  async function stop() {
    if (timer) clearInterval(timer);
    timer = undefined;
    await Promise.resolve();
    try {
      process.stdout.write(`[monitor/process] stop\n`);
    } catch {
      void 0;
    }
  }
  function getEvents(): ProcessEvent[] {
    return events;
  }
  return { start, stop, getEvents };
}
