import type Docker from "dockerode";
import type { ResourceUsage } from "@sock/core";
import { getStats } from "../docker";

export type ResourceCollector = {
  start(): Promise<void>;
  stop(): Promise<void>;
  getUsage(): ResourceUsage;
};

export function createResourceCollector(args: {
  docker: Docker;
  containerId: string;
  sampleMs: number;
}): ResourceCollector {
  const start = new Date().toISOString();
  let end = start;
  let timer: ReturnType<typeof setInterval> | undefined;
  let cpuSecondsAcc = 0;
  let memoryMbMax = 0;
  let networkKbAcc = 0;
  let diskMbAcc = 0;
  async function startSampling() {
    if (timer) return;
    try {
      process.stdout.write(
        `[monitor/resources] start container=${args.containerId} sampleMs=${args.sampleMs}\n`
      );
    } catch {
      void 0;
    }
    timer = setInterval(() => {
      void sampleOnce();
    }, args.sampleMs);
    await sampleOnce();
  }
  async function sampleOnce() {
    try {
      const stats: any = await getStats(args.docker, args.containerId);
      const cpuTotal = Number(stats?.cpu_stats?.cpu_usage?.total_usage ?? 0);
      const cpuPrev = Number(stats?.precpu_stats?.cpu_usage?.total_usage ?? 0);
      const cpuDelta = Math.max(0, cpuTotal - cpuPrev);
      const systemTotal = Number(stats?.cpu_stats?.system_cpu_usage ?? 0);
      const systemPrev = Number(stats?.precpu_stats?.system_cpu_usage ?? 0);
      const systemDelta = Math.max(0, systemTotal - systemPrev);
      const onlineCpus = Number(stats?.cpu_stats?.online_cpus ?? 1) || 1;
      const cpuPercent =
        systemDelta > 0 ? (cpuDelta / systemDelta) * onlineCpus : 0;
      cpuSecondsAcc += cpuPercent * (args.sampleMs / 1000);

      const memBytes = Number(stats?.memory_stats?.usage ?? 0);
      memoryMbMax = Math.max(memoryMbMax, memBytes / (1024 * 1024));

      const rx = Number(stats?.networks?.eth0?.rx_bytes ?? 0);
      const tx = Number(stats?.networks?.eth0?.tx_bytes ?? 0);
      networkKbAcc = Math.max(networkKbAcc, (rx + tx) / 1024);

      const ioServiceBytes =
        stats?.blkio_stats?.io_service_bytes_recursive ?? [];
      const bytes = ioServiceBytes.reduce(
        (sum: number, e: any) => sum + Number(e?.value ?? 0),
        0
      );
      diskMbAcc = Math.max(diskMbAcc, bytes / (1024 * 1024));
    } catch (err) {
      try {
        const msg = err instanceof Error ? err.message : String(err);
        process.stderr.write(`[monitor/resources] sampleOnce error: ${msg}\n`);
      } catch {
        void 0;
      }
    }
  }
  async function stop() {
    if (timer) clearInterval(timer);
    await sampleOnce();
    end = new Date().toISOString();
    try {
      process.stdout.write(`[monitor/resources] stop\n`);
    } catch {
      void 0;
    }
  }
  function getUsage(): ResourceUsage {
    return {
      cpu_seconds: cpuSecondsAcc,
      memory_mb: memoryMbMax,
      network_kb: networkKbAcc,
      disk_mb: diskMbAcc,
      start_time: start,
      end_time: end,
    };
  }
  return { start: startSampling, stop, getUsage };
}
