import Docker from "dockerode";

export function createDocker(): Docker {
  return new Docker();
}

export async function findAgentContainerId(
  docker: Docker,
  experimentId: string
): Promise<string | undefined> {
  try {
    const list = await docker.listContainers({
      all: true,
      filters: { label: [`sock.exp=${experimentId}`, "sock.role=agent"] },
    });
    return list?.[0]?.Id;
  } catch {
    return undefined;
  }
}

export async function findTargetContainerIds(
  docker: Docker,
  experimentId: string
): Promise<string[]> {
  try {
    const list = await docker.listContainers({
      all: true,
      filters: { label: [`sock.exp=${experimentId}`, "sock.role=target"] },
    });
    return list.map((c) => c.Id);
  } catch {
    return [];
  }
}

export async function getTop(
  docker: Docker,
  containerId: string
): Promise<{ titles: string[]; processes: string[][] }> {
  const res = await (docker.getContainer(containerId) as any).top();
  return { titles: res.Titles ?? [], processes: res.Processes ?? [] };
}

export async function getDiff(
  docker: Docker,
  containerId: string
): Promise<Array<{ Path: string; Kind: number }>> {
  try {
    const diff = await (docker.getContainer(containerId) as any).changes();
    return diff ?? [];
  } catch {
    return [];
  }
}

export async function getStats(
  docker: Docker,
  containerId: string
): Promise<any> {
  const container = docker.getContainer(containerId) as any;
  return await new Promise((resolve, reject) => {
    container.stats({ stream: false }, (err: unknown, data: unknown) => {
      if (err)
        reject(
          err instanceof Error
            ? err
            : new Error(err ? JSON.stringify(err) : "Unknown error")
        );
      else resolve(data);
    });
  });
}

export function watchEvents(
  docker: Docker,
  filters: Record<string, string[]>,
  onEvent: (evt: any) => void
): { cancel: () => void } {
  let stream: any;
  void docker.getEvents({ filters }, (err: unknown, s: any) => {
    if (err || !s) return;
    stream = s;
    stream.on("data", (chunk: Buffer) => {
      try {
        const evt = JSON.parse(chunk.toString("utf8"));
        onEvent(evt);
      } catch {
        void 0;
      }
    });
  });
  return {
    cancel: () => {
      try {
        stream?.destroy();
      } catch {
        void 0;
      }
    },
  };
}

export async function getLogs(
  docker: Docker,
  containerId: string,
  tail: number = 200
): Promise<{ stdout: string; stderr: string }> {
  try {
    const container = docker.getContainer(containerId);
    const common: Partial<Docker.ContainerLogsOptions> = {
      timestamps: false,
      ...(typeof tail === "number" && tail > 0 ? { tail } : {}),
    };
    const out = await container.logs({
      stdout: true,
      stderr: false,
      follow: false,
      ...(common as any),
    } as Docker.ContainerLogsOptions & { follow?: false });
    const err = await container.logs({
      stdout: false,
      stderr: true,
      follow: false,
      ...(common as any),
    } as Docker.ContainerLogsOptions & { follow?: false });
    const sOut = Buffer.isBuffer(out)
      ? out.toString("utf8")
      : String(out ?? "");
    const sErr = Buffer.isBuffer(err)
      ? (err as any).toString("utf8")
      : String(err ?? "");
    return { stdout: sOut, stderr: sErr };
  } catch (err) {
    try {
      const msg = err instanceof Error ? err.message : String(err);
      process.stderr.write(
        `[monitor] getLogs error container=${containerId}: ${msg}\n`
      );
    } catch {
      void 0;
    }
    return { stdout: "", stderr: "" };
  }
}
