import type { MonitoringResult } from "@sock/core";
import { readConfig } from "./config";
import { aggregateMonitoring } from "./aggregate";
import {
  createDocker,
  findAgentContainerId,
  findTargetContainerIds,
  getLogs,
} from "./docker";
import { createProcessCollector } from "./collect/process";
import { createFileCollector } from "./collect/files";
import { createContainerCollector } from "./collect/container";
import { createResourceCollector } from "./collect/resources";

export async function main(): Promise<void> {
  const cfg = readConfig();
  const docker = createDocker();
  const agentId =
    (await findAgentContainerId(docker, cfg.experimentId)) ?? "unknown";
  const targetIds = await findTargetContainerIds(docker, cfg.experimentId);
  const parsedSeed = Number(process.env.SEED);
  const seedEnv = Number.isFinite(parsedSeed) ? parsedSeed : undefined;

  const processCollector = createProcessCollector({
    docker,
    containerId: agentId,
    sampleMs: cfg.sampleMs,
  });
  const fileCollector = createFileCollector({
    docker,
    containerIds: [agentId, ...targetIds],
  });
  const containerCollector = createContainerCollector({
    docker,
    experimentId: cfg.experimentId,
  });
  const resourceCollector = createResourceCollector({
    docker,
    containerId: agentId,
    sampleMs: cfg.sampleMs,
  });

  if (cfg.enableProcess) {
    await processCollector.start();
  }
  if (cfg.enableContainer) {
    await containerCollector.start();
  }
  if (cfg.enableResources) {
    await resourceCollector.start();
  }

  let emitted = false;
  const emit = async (): Promise<void> => {
    if (emitted) return;
    emitted = true;
    await processCollector.stop();
    await containerCollector.stop();
    await resourceCollector.stop();
    if (cfg.enableFiles) await fileCollector.sampleOnce();

    const agentLogs = agentId
      ? await getLogs(docker, agentId, 0)
      : { stdout: "", stderr: "" };
    const agentout = (agentLogs.stdout + "\n" + agentLogs.stderr)
      .split("\n")
      .filter((l) => l.includes("[AGENTOUT] "))
      .map((l) => l.replace(/^.*\[AGENTOUT\]\s*/, ""))
      .join("\n")
      .trim()
      .slice(-4000);

    const args = {
      experimentId: cfg.experimentId,
      containerId: agentId,
      files: fileCollector.getFileOperations(),
      processes: processCollector.getEvents(),
      containers: containerCollector.getEvents(),
      usage: resourceCollector.getUsage(),
      stdout: "",
      stderr: "",
      agentout,
    } as Parameters<typeof aggregateMonitoring>[0];
    if (typeof seedEnv === "number") {
      (args as any).seed = seedEnv;
    }
    const result: MonitoringResult = aggregateMonitoring(args);
    process.stdout.write(JSON.stringify(result));
    process.stdout.write("\n");
    process.exit(0);
  };

  process.on("SIGTERM", () => void emit());
  process.on("SIGINT", () => void emit());

  setInterval(() => {}, 1 << 30);
}
