import Docker from "dockerode";
import fs from "node:fs/promises";
import fsSync from "node:fs";
import path from "node:path";
import { fileURLToPath } from "node:url";
import tar from "tar-stream";
import type { Pack } from "tar-stream";
import { CONTAINER_IMAGES, type ExperimentConfig } from "@sock/core";

const __filename = fileURLToPath(import.meta.url);
const __dirname = path.dirname(__filename);

export async function ensureRequiredImages(
  config: ExperimentConfig
): Promise<void> {
  const docker = new Docker();
  const images = new Set<string>(
    config.environment.containers.map((c) => c.image)
  );
  for (const image of images) {
    const exists = await imageExists(docker, image);
    if (exists) continue;
    if (image === CONTAINER_IMAGES.AGENT) {
      await buildAgentImage(docker, image);
    } else if (image === CONTAINER_IMAGES.MONITOR) {
      await buildMonitorImage(docker, image);
    } else if (image === CONTAINER_IMAGES.TARGET) {
      await buildImageFromDockerfilePath(
        docker,
        image,
        resolveRunnerAsset("containers/target/Dockerfile")
      );
    } else {
      throw new Error(`Image not found locally: ${image}`);
    }
  }
}

async function imageExists(docker: Docker, tag: string): Promise<boolean> {
  try {
    await docker.getImage(tag).inspect();
    return true;
  } catch {
    return false;
  }
}

async function buildImageFromDockerfilePath(
  docker: Docker,
  tag: string,
  dockerfilePath: string
): Promise<void> {
  const dockerfile = await fs.readFile(dockerfilePath, "utf8");
  const pack = tar.pack();
  pack.entry({ name: "Dockerfile" }, dockerfile);
  pack.finalize();
  await new Promise<void>((resolve, reject) => {
    docker.buildImage(
      pack as unknown as import("stream").Readable,
      { t: tag },
      (err, stream) => {
        if (err) {
          reject(err instanceof Error ? err : new Error(String(err)));
          return;
        }
        if (!stream) {
          reject(new Error("No build stream"));
          return;
        }
        stream.on("data", () => {});
        stream.on("end", () => resolve());
        stream.on("error", (e) =>
          reject(e instanceof Error ? e : new Error(String(e)))
        );
      }
    );
  });
  await pruneDanglingImages(docker);
}

function resolveRunnerAsset(relativePath: string): string {
  const base = path.resolve(__dirname, "..");
  const candidates = [
    path.resolve(base, relativePath),
    path.resolve(base, "src", relativePath),
  ];
  for (const p of candidates) {
    if (fsSync.existsSync(p)) {
      const stat = fsSync.statSync(p);
      if (stat.isFile()) return p;
    }
  }
  throw new Error(`Runner asset not found: ${relativePath}`);
}

async function addDirToPack(
  pack: Pack,
  srcDir: string,
  destPrefix: string
): Promise<void> {
  const entries = await fs.readdir(srcDir, { withFileTypes: true });
  for (const entry of entries) {
    const abs = path.join(srcDir, entry.name);
    const dest = path.posix.join(destPrefix, entry.name);
    if (entry.isDirectory()) {
      await addDirToPack(pack, abs, dest);
    } else if (entry.isFile()) {
      const content = await fs.readFile(abs);
      pack.entry({ name: dest }, content);
    }
  }
}

async function buildAgentImage(docker: Docker, tag: string): Promise<void> {
  const dockerfilePath = resolveRunnerAsset("containers/agent/Dockerfile");
  const workspaceRoot = path.resolve(__dirname, "../../..");
  const dockerfile = await fs.readFile(dockerfilePath, "utf8");
  const pack = tar.pack();
  pack.entry({ name: "Dockerfile" }, dockerfile);
  const filesToCopy = [
    "pnpm-lock.yaml",
    "pnpm-workspace.yaml",
    "package.json",
    "tsconfig.json",
    "packages/core/package.json",
    "packages/agent/package.json",
  ];
  for (const rel of filesToCopy) {
    const abs = path.join(workspaceRoot, rel);
    try {
      const content = await fs.readFile(abs);
      pack.entry({ name: rel.replace(/\\/g, "/") }, content);
    } catch {
      void 0;
    }
  }
  await addDirToPack(
    pack,
    path.join(workspaceRoot, "packages/core"),
    "packages/core"
  );
  await addDirToPack(
    pack,
    path.join(workspaceRoot, "packages/agent"),
    "packages/agent"
  );
  pack.finalize();
  await new Promise<void>((resolve, reject) => {
    docker.buildImage(
      pack as unknown as import("stream").Readable,
      { t: tag },
      (err, stream) => {
        if (err) {
          reject(err instanceof Error ? err : new Error(String(err)));
          return;
        }
        if (!stream) {
          reject(new Error("No build stream"));
          return;
        }
        stream.on("data", () => {});
        stream.on("end", () => resolve());
        stream.on("error", (e) =>
          reject(e instanceof Error ? e : new Error(String(e)))
        );
      }
    );
  });
  await pruneDanglingImages(docker);
}

async function buildMonitorImage(docker: Docker, tag: string): Promise<void> {
  const dockerfilePath = resolveRunnerAsset("containers/monitor/Dockerfile");
  const workspaceRoot = path.resolve(__dirname, "../../..");
  const dockerfile = await fs.readFile(dockerfilePath, "utf8");
  const pack = tar.pack();
  pack.entry({ name: "Dockerfile" }, dockerfile);
  const filesToCopy = [
    "pnpm-lock.yaml",
    "pnpm-workspace.yaml",
    "package.json",
    "tsconfig.json",
    "packages/core/package.json",
    "packages/monitor/package.json",
  ];
  for (const rel of filesToCopy) {
    const abs = path.join(workspaceRoot, rel);
    try {
      const content = await fs.readFile(abs);
      pack.entry({ name: rel.replace(/\\/g, "/") }, content);
    } catch {
      void 0;
    }
  }
  await addDirToPack(
    pack,
    path.join(workspaceRoot, "packages/core"),
    "packages/core"
  );
  await addDirToPack(
    pack,
    path.join(workspaceRoot, "packages/monitor"),
    "packages/monitor"
  );
  pack.finalize();
  await new Promise<void>((resolve, reject) => {
    docker.buildImage(
      pack as unknown as import("stream").Readable,
      { t: tag },
      (err, stream) => {
        if (err) {
          reject(err instanceof Error ? err : new Error(String(err)));
          return;
        }
        if (!stream) {
          reject(new Error("No build stream"));
          return;
        }
        stream.on("data", () => {});
        stream.on("end", () => resolve());
        stream.on("error", (e) =>
          reject(e instanceof Error ? e : new Error(String(e)))
        );
      }
    );
  });
  await pruneDanglingImages(docker);
}

async function pruneDanglingImages(docker: Docker): Promise<void> {
  try {
    await new Promise<void>((resolve, reject) => {
      docker.pruneImages(
        { filters: { dangling: ["true"] } },
        (err: unknown) => {
          if (err instanceof Error) {
            reject(err);
            return;
          }
          if (typeof err === "string") {
            reject(new Error(err));
            return;
          }
          resolve();
        }
      );
    });
  } catch {
    void 0;
  }
}
