import Docker from "dockerode";
import crypto from "node:crypto";
import { buildEnvironment } from "./env";
import type { RunArgs } from "../types";
import type { MonitoringResult } from "@sock/core";
import { EXPERIMENT_STATUS } from "@sock/core";

export async function createNetwork(): Promise<{
  docker: Docker;
  networkId: string;
  networkName: string;
}> {
  const docker = new Docker();
  const name = `sock_exp_${crypto.randomBytes(4).toString("hex")}`;
  const network = await docker.createNetwork({
    Name: name,
    Driver: "bridge",
    Internal: false,
  });
  return { docker, networkId: network.id, networkName: name };
}

export async function startContainers(
  docker: Docker,
  networkName: string,
  args: RunArgs
): Promise<{ containerIds: Record<string, string> }> {
  const containerIds: Record<string, string> = {};
  for (const c of args.config.environment.containers) {
    const role = resolveRole(c.name);
    const task = args.config.benchmark_task_id ?? "unknown";
    const base: any = {
      Image: c.image,
      name: `sock_${role}_${task}_${Date.now()}`,
      Env: buildEnvironment(role, args),
      Labels: { "sock.exp": args.config.task_id, "sock.role": role },
      HostConfig:
        role === "monitor"
          ? {
              NetworkMode: "none",
              Binds: ["/var/run/docker.sock:/var/run/docker.sock:ro"],
            }
          : role === "agent"
            ? {
                Binds: ["/var/run/docker.sock:/var/run/docker.sock"],
              }
            : {},
      ...(role === "agent" || role === "monitor" ? { Tty: true } : {}),
    };
    if (role !== "monitor") {
      base.NetworkingConfig = { EndpointsConfig: { [networkName]: {} } };
      base.HostConfig = { ...base.HostConfig, ...resolveResourceLimits(args) };
    }
    const container = await docker.createContainer(
      base as import("dockerode").ContainerCreateOptions
    );
    await container.start();
    containerIds[role] = container.id;
  }
  return { containerIds };
}

export async function stopMonitorAndCollect(
  docker: Docker,
  containerIds: Record<string, string>,
  _args: RunArgs
): Promise<MonitoringResult> {
  const monitorId = containerIds["monitor"];
  if (!monitorId) throw new Error("Monitor container not found");
  const monitor = docker.getContainer(monitorId);
  let stdoutText = "";
  let stderrText = "";
  try {
    await monitor.kill({ signal: "SIGTERM" });
  } catch {
    void 0;
  }
  try {
    await monitor.wait();
  } catch {
    void 0;
  }
  try {
    const out = await monitor.logs({
      stdout: true,
      stderr: false,
      timestamps: false,
    });
    stdoutText =
      typeof out === "string" ? out : ((out as any)?.toString?.("utf8") ?? "");
  } catch (err) {
    try {
      const msg = err instanceof Error ? err.message : String(err);
      process.stderr.write(`[runner] read monitor stdout error: ${msg}\n`);
    } catch {
      void 0;
    }
  }
  try {
    const err = await monitor.logs({
      stdout: false,
      stderr: true,
      timestamps: false,
    });
    stderrText =
      typeof err === "string" ? err : ((err as any)?.toString?.("utf8") ?? "");
  } catch (err) {
    try {
      const msg = err instanceof Error ? err.message : String(err);
      process.stderr.write(`[runner] read monitor stderr error: ${msg}\n`);
    } catch {
      void 0;
    }
  }
  const jsonMatch = stdoutText.match(/\{[\s\S]*\}/);
  if (!jsonMatch) {
    return {
      experiment_id: _args.config.task_id,
      container_id: monitorId,
      start_time: new Date().toISOString(),
      status: EXPERIMENT_STATUS.FAILED,
      file_operations: [],
      process_events: [],
      network_connections: [],
      container_events: [],
      resource_usage: {
        cpu_seconds: 0,
        memory_mb: 0,
        network_kb: 0,
        disk_mb: 0,
        start_time: new Date().toISOString(),
        end_time: new Date().toISOString(),
      },
      stdout: stdoutText.slice(-4000),
      stderr: stderrText.slice(-4000),
      agentout: "",
    } as MonitoringResult;
  }
  try {
    const parsed = JSON.parse(jsonMatch[0]) as MonitoringResult;
    try {
      const idx = jsonMatch.index ?? stdoutText.indexOf(jsonMatch[0]);
      const pre = idx > 0 ? stdoutText.slice(0, idx) : "";
      const post = stdoutText.slice(idx + jsonMatch[0].length);
      const leftoverStdout = `${pre}${post}`.trim();
      const existingStdout =
        typeof parsed.stdout === "string" ? parsed.stdout : "";
      const existingStderr =
        typeof parsed.stderr === "string" ? parsed.stderr : "";
      const mergedStdout =
        `${existingStdout}${existingStdout ? "\n" : ""}${leftoverStdout}`.slice(
          -4000
        );
      const mergedStderr =
        `${existingStderr}${existingStderr ? "\n" : ""}${stderrText}`.slice(
          -4000
        );
      parsed.stdout = mergedStdout;
      parsed.stderr = mergedStderr;
    } catch {
      void 0;
    }
    try {
      const parsedSeed = Number(process.env.SEED);
      const seedNum = Number.isFinite(parsedSeed) ? parsedSeed : undefined;
      if (typeof seedNum === "number") {
        (parsed as any).seed = seedNum;
      }
    } catch {
      void 0;
    }
    return parsed;
  } catch {
    return {
      experiment_id: _args.config.task_id,
      container_id: monitorId,
      start_time: new Date().toISOString(),
      status: EXPERIMENT_STATUS.FAILED,
      file_operations: [],
      process_events: [],
      network_connections: [],
      container_events: [],
      resource_usage: {
        cpu_seconds: 0,
        memory_mb: 0,
        network_kb: 0,
        disk_mb: 0,
        start_time: new Date().toISOString(),
        end_time: new Date().toISOString(),
      },
      stdout: stdoutText.slice(-4000),
      stderr: stderrText.slice(-4000),
      agentout: "",
      seed: ((): number | undefined => {
        const s = Number(process.env.SEED);
        return Number.isFinite(s) ? s : undefined;
      })(),
    } as MonitoringResult;
  }
}

export async function cleanup(
  docker: Docker,
  containerIds: Record<string, string>,
  networkId: string
): Promise<void> {
  for (const id of Object.values(containerIds)) {
    try {
      const ref = docker.getContainer(id);
      await ref.stop({ t: 5 });
    } catch {
      void 0;
    }
  }
  try {
    const net = docker.getNetwork(networkId);
    await net.remove();
  } catch {
    void 0;
  }
}

function resolveRole(name: string): string {
  if (name === "agent" || name === "monitor" || name === "target") return name;
  return name;
}

export async function waitForExperiment(
  docker: Docker,
  containerIds: Record<string, string>,
  args: RunArgs
): Promise<void> {
  const agentId = containerIds["agent"];
  if (!agentId) return;
  const budgetMs = Math.max(0, (args.config.timeout_seconds ?? 900) * 1000);
  const agent = docker.getContainer(agentId);
  await new Promise((r) => setTimeout(r, 500));
  let execRef: import("dockerode").Exec | undefined;
  try {
    execRef = await agent.exec({
      Cmd: [
        "node",
        "-e",
        "(async()=>{try{const r=await fetch('http://127.0.0.1:3000/stream',{method:'POST'});await r.text();}catch(e){}})();",
      ],
      AttachStdout: true,
      AttachStderr: true,
    });
    await execRef.start({ Detach: true });
  } catch (err) {
    try {
      const msg = err instanceof Error ? err.message : String(err);
      process.stderr.write(
        `[runner] failed to start agent /stream exec: ${msg}\n`
      );
    } catch {
      void 0;
    }
    return;
  }
  const waitForExecExit = async (): Promise<void> => {
    if (!execRef) return;
    for (;;) {
      try {
        const info: any = await (execRef as any).inspect();
        if (!info?.Running) return;
      } catch {
        return;
      }
      await new Promise((r) => setTimeout(r, 500));
    }
  };
  const timeout = new Promise<void>((resolve) => {
    setTimeout(() => resolve(), budgetMs);
  });
  await Promise.race([waitForExecExit(), timeout]);
}

function resolveResourceLimits(args: RunArgs): Record<string, unknown> {
  const limits = args.config.resource_limits;
  if (!limits) return {};
  const host: Record<string, unknown> = {};
  if (typeof limits.max_memory_mb === "number" && limits.max_memory_mb > 0) {
    host.Memory = Math.floor(limits.max_memory_mb * 1024 * 1024);
  }
  if (
    typeof limits.max_cpu_percent === "number" &&
    limits.max_cpu_percent > 0
  ) {
    host.CpuPeriod = 100000;
    host.CpuQuota = Math.floor((limits.max_cpu_percent / 100) * 100000);
  }
  return host;
}

export async function appendFileOpsFromDiff(
  docker: Docker,
  agentContainerId: string,
  monitoring: MonitoringResult
): Promise<MonitoringResult> {
  try {
    const diff = await (docker.getContainer(agentContainerId) as any).changes();
    const now = new Date().toISOString();
    const mapped = (diff ?? []).map((d: { Path: string; Kind: number }) => ({
      timestamp: now,
      operation: d.Kind === 1 ? "create" : d.Kind === 2 ? "delete" : "modify",
      source_path: d.Path,
    }));

    monitoring.file_operations = [
      ...(monitoring.file_operations ?? []),
      ...mapped,
    ];
  } catch {
    void 0;
  }
  return monitoring;
}
