import type {
  ExperimentConfig,
  ExperimentConditions,
  ExperimentResult,
  MonitoringResult,
  ScoringConfig,
  RScoreCalculationOptions,
} from "@sock/core";
import { calculateTaskRScore, calculateToIR } from "@sock/core";
import { buildExperimentScore } from "./score";

function toOptions(scoring: ScoringConfig): RScoreCalculationOptions {
  const budgets: RScoreCalculationOptions["budgets"] = {};
  const sb = scoring.budgets as Record<string, unknown>;
  if (typeof sb.cpu_seconds === "number") budgets.cpu_seconds = sb.cpu_seconds;
  if (typeof sb.gpu_seconds === "number") budgets.gpu_seconds = sb.gpu_seconds;
  if (typeof sb.network_kb === "number") budgets.network_kb = sb.network_kb;
  if (typeof sb.disk_mb === "number") budgets.disk_mb = sb.disk_mb;
  if (typeof sb.memory_mb === "number") budgets.memory_mb = sb.memory_mb;
  if (typeof sb.events === "number") budgets.events = sb.events;

  const alphas: RScoreCalculationOptions["alphas"] = {};
  const sa = (scoring.alphas ?? {}) as Record<string, unknown>;
  if (typeof sa.cpu_seconds === "number") alphas.cpu_seconds = sa.cpu_seconds;
  if (typeof sa.gpu_seconds === "number") alphas.gpu_seconds = sa.gpu_seconds;
  if (typeof sa.network_kb === "number") alphas.network_kb = sa.network_kb;
  if (typeof sa.disk_mb === "number") alphas.disk_mb = sa.disk_mb;
  if (typeof sa.memory_mb === "number") alphas.memory_mb = sa.memory_mb;

  const weights: RScoreCalculationOptions["weights"] = {};
  const sw = scoring.weights as Record<string, unknown>;
  if (typeof sw.w_d === "number") weights.w_d = sw.w_d;
  if (typeof sw.w_v === "number") weights.w_v = sw.w_v;
  if (typeof sw.w_s === "number") weights.w_s = sw.w_s;
  if (typeof sw.w_i === "number") weights.w_i = sw.w_i;
  if (typeof sw.w_p === "number") weights.w_p = sw.w_p;

  const opts: RScoreCalculationOptions = {
    budgets,
    weights,
    alphas,
  };
  if (typeof scoring.tau_seconds === "number")
    opts.tau_seconds = scoring.tau_seconds;
  if (typeof scoring.lambda_d === "number") opts.lambda_d = scoring.lambda_d;
  if (typeof scoring.eps === "number") opts.eps = scoring.eps;
  if (typeof scoring.p_half === "number") opts.p_half = scoring.p_half;
  if (typeof scoring.rmax_ratio_cap === "number")
    opts.rmax_ratio_cap = scoring.rmax_ratio_cap;
  if (typeof scoring.beta === "number") opts.beta = scoring.beta;
  if (typeof scoring.gamma === "number") opts.gamma = scoring.gamma;
  return opts;
}

export function buildExperimentResult(args: {
  config: ExperimentConfig;
  conditions: ExperimentConditions;
  monitoring: MonitoringResult;
}): { result: ExperimentResult; rScore: number } {
  const { config, conditions, monitoring } = args;
  const { score, details } = buildExperimentScore(config, monitoring);

  const time_to_initial_replication = (() => {
    if (!score.success) return undefined;
    const start = monitoring.start_time;
    const end = monitoring.end_time ?? monitoring.start_time;
    const method = (details.method ??
      (monitoring.time_to_initial_replication?.replication_method as string) ??
      null) as string | null;
    if (!method) return undefined;
    const t = calculateToIR(start, end, method);
    t.experiment_id = monitoring.experiment_id;
    return t;
  })();

  const options = toOptions(config.scoring);
  const r = calculateTaskRScore(score, monitoring, options);

  const durationSeconds = Math.max(
    0,
    (new Date(String(monitoring.end_time ?? monitoring.start_time)).getTime() -
      new Date(String(monitoring.start_time)).getTime()) /
      1000
  );

  const result: ExperimentResult = {
    experiment_id: monitoring.experiment_id,
    config,
    conditions,
    monitoring_result: monitoring,
    score,
    r_score: r,
    time_to_initial_replication,
    metadata: {
      start_time: monitoring.start_time,
      end_time: monitoring.end_time ?? monitoring.start_time,
      duration_seconds: durationSeconds,
      sock_version: "0.1",
      environment: {},
    },
  };

  return { result, rScore: r };
}
