# evaluate a smoothed classifier on a dataset
import argparse
import os
#import setGPU
from datasets import get_dataset, DATASETS, get_num_classes
from core import Smooth
from time import time
import torch
import datetime

from architectures import get_architecture, CIFAR_ARCHITECTURES, IMAGENET_ARCHITECTURES
from smoothing_distribution import GaussianSampler, GaussianImageSampler, LOneSampler, UnifSampler

#from device import DEVICE
DEVICE = torch.device("cpu")

from tqdm import tqdm

CLASSIFIER_CHOICES = {
    "cifar10": CIFAR_ARCHITECTURES,
    "imagenet": IMAGENET_ARCHITECTURES,
}
SAMPLER_CHOICES = {"gaussian", "gen", "L1"}

parser = argparse.ArgumentParser(description='Certify many examples')
parser.add_argument("dataset", choices=DATASETS, help="which dataset")

args, remaining_args = parser.parse_known_args()

parser.add_argument("base_classifier", choices=CLASSIFIER_CHOICES[args.dataset], type=str, help="which classifier")
parser.add_argument("attack", type=str, help="attack type")
parser.add_argument("sampler", type=str, help="type of smoothing distribution")
parser.add_argument("sigma", type=float, help="noise hyperparameter")
parser.add_argument("--batch", type=int, default=400, help="batch size")
parser.add_argument("--skip", type=int, default=1, help="how many examples to skip")
parser.add_argument("--max", type=int, default=-1, help="stop after this many examples")
parser.add_argument("--split", choices=["train", "test"], default="test", help="train or test set")
parser.add_argument("--N0", type=int, default=100)
parser.add_argument("--N", type=int, default=100000, help="number of samples to use")
parser.add_argument("--alpha", type=float, default=0.001, help="failure probability")
args = parser.parse_args()

if __name__ == "__main__":
    
    # load the base classifier
    base_classifier_file =  'models/base_classifiers/' + args.base_classifier + '_' + args.dataset + '_' + args.attack + '_checkpoint.pth.tar'
    checkpoint = torch.load(base_classifier_file,map_location=DEVICE)
    base_classifier = get_architecture(checkpoint["arch"], args.dataset)
    base_classifier.load_state_dict(checkpoint['state_dict'])
    base_classifier = base_classifier.to(DEVICE)
    
    # you need to put an exception here, in case the base classifier does not exist
    # you need to make sure that the attacks, architectures and datasets are correct

    # choose sampling method for the smoothing distribution 
    if args.sampler=="gaussian" :
        sampler = GaussianSampler(args.sigma)
    elif args.sampler=="gen" :
        sampler = GaussianImageSampler(args.dataset, args.base_classifier, args.attack, args.sigma)
    elif args.sampler=="L1" :
        sampler = LOneSampler(args.sigma)

    # create the smooothed classifier g
    smoothed_classifier = Smooth(base_classifier, get_num_classes(args.dataset), sampler)

    # prepare output file
    output_path = 'results'
    if not os.path.exists(output_path): os.mkdir(output_path)
    f = open(output_path +'/' + args.base_classifier + '_' + args.dataset + '_' + args.attack + '_' + args.sampler + '_' + str(args.sigma) + '_' + str(args.N) + '_' + str(1-args.alpha) + '_certify.csv', 'w')
    print("idx\tlabel\tpredict\tradius\tcorrect\ttime", file=f, flush=True)

    # iterate through the dataset
    dataset = get_dataset(args.dataset, args.split)

    for i in tqdm(range(len(dataset)), desc="Certify images"):

        # only certify every args.skip examples, and stop after args.max examples
        if i % args.skip != 0:
            continue
        if i == args.max:
            break

        (x, label) = dataset[i]

        before_time = time()
        
        # certify the prediction of g around x
        x = x.to(DEVICE)
        prediction, radius = smoothed_classifier.certify(x, args.N0, args.N, args.alpha, args.batch)
        after_time = time()
        correct = int(prediction == label)

        time_elapsed = str(datetime.timedelta(seconds=(after_time - before_time)))
        print("{}\t{}\t{}\t{:.3}\t{}\t{}".format(
            i, label, prediction, radius, correct, time_elapsed), file=f, flush=True)

    f.close()
